//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;

import com.azure.ai.vision.common.FrameFormat;
import com.azure.ai.vision.common.FrameWriter;
import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.FrameSourceJNI;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import java.util.function.Consumer;

/**
 * FrameSource class
 * Represents a source of image frame data, used as input to or output from Vision AI operations.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class FrameSource implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    FrameSource(SafeHandle handle, FrameFormat frameFormat) {
        Contracts.throwIfNull(handle, "handle");
        this.frameSourceHandle = handle;
        this.frameFormat = frameFormat;
        properties = new PropertyCollection(FrameSourceJNI.getFrameSourcePropertiesHandle(handle.getValue()));
    }

    FrameSource(SafeHandle handle, Consumer<FrameSource> frameSourceConsumer, FrameFormat frameFormat) {
        Contracts.throwIfNull(handle, "handle");
        this.frameSourceHandle = handle;
        this.frameFormat = frameFormat;
        this.frameSourceConsumer = frameSourceConsumer;
        properties = new PropertyCollection(FrameSourceJNI.getFrameSourcePropertiesHandle(handle.getValue()));

        if (frameSourceConsumer != null) {
            FrameSourceJNI.frameSourceCallbackSet(handle.getValue(), this);
        }
    }

    /**
     * Initializes a new instance of the FrameSource class.
     * @param format The frame format
     * @return A FrameSource instance
     */
    public static FrameSource fromFormat(FrameFormat format) {
        Contracts.throwIfNull(format, "format");

        PropertiesJNI.copyProperties(format.getProperties().getHandle(), format.getProperties().getHandle(), "adapter.streams.0.");
        format.getProperties().setProperty("adapter.streams.count", String.valueOf(1));

        SafeHandle handle = FrameSourceJNI.createFrameSourceHandle("", "", format.getHandle());
        return new FrameSource(handle, format);
    }

    /**
     * Initializes a new instance of the FrameSource class.
     * @param format The frame format
     * @param frameSourceConsumer The consumer to be called when a new frame is required.
     * @return A FrameSource instance
     */
    public static FrameSource fromFormat(FrameFormat format, Consumer<FrameSource> frameSourceConsumer) {
        Contracts.throwIfNull(format, "format");

        SafeHandle handle = FrameSourceJNI.createFrameSourceHandle("", "", format.getHandle());
        return new FrameSource(handle, frameSourceConsumer, format);
    }

    /**
     * Gets a new instance of FrameWriter that can be used to write (submit) frame data to this FrameSource instance.
     * @return A new FrameWriter instance associated to this FrameSource.
     */
    public FrameWriter getWriter() {
        Contracts.throwIfTrue(disposed, "disposed");
        SafeHandle writerHandle = FrameSourceJNI.getFrameSourceWriterHandle(this.frameSourceHandle.getValue());
        return new FrameWriter(writerHandle);
    }

    /**
     * Closes the frame source for writing.
     */
    public void closeWriter() {
        Contracts.throwIfTrue(disposed, "disposed");
        FrameSourceJNI.frameSourceCloseWriter(this.frameSourceHandle.getValue());
    }

    /**
     * Get frame format from the FrameSource
     *
     * @return A FrameFormat instance associated to this FrameSource.
     */
    public FrameFormat getFormat() {
        Contracts.throwIfTrue(disposed, "disposed");
        return this.frameFormat;
    }
    /*! \cond INTERNAL */

    /**
     * Gets the internal native handle.
     * @return native handle value
     */
    public long getHandle() {
        return this.frameSourceHandle.getValue();
    }

    /*! \endcond */

    /**
     * Explicitly frees any external resource attached to the object
     * 
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {

        if (this.disposed) return;

        if (this.properties != null) {
            this.properties.close();
        }
        if (this.frameSourceHandle != null) {
            this.frameSourceHandle.close();
        }
        this.frameFormat = null;
        this.disposed = true;
    }

    /**
     * Called by native JNI code to notify that a new frame is available.
     * 
     * @param frameSourceHandle the native handle of the frame source
     */
    private void frameSourceCallback(long frameSourceHandle) {
        if (this.frameSourceConsumer != null) {
            this.frameSourceConsumer.accept(this);
        }
    }

    private SafeHandle frameSourceHandle;
    private Consumer<FrameSource> frameSourceConsumer;
    private final PropertyCollection properties;
    private FrameFormat frameFormat;
    private boolean disposed = false;
}
