//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.nio.ByteBuffer;

import com.azure.ai.vision.common.FrameSource;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.FrameWriterJNI;

/**
 * FrameWriter class
 * Represents the ability to write image frame data, for use as input w/ Vision AI scenario operations.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class FrameWriter implements AutoCloseable {

    FrameWriter(SafeHandle handle) {
        Contracts.throwIfNull(handle, "handle");
        this.frameWriterHandle = handle;
    }

    /**
     * Writes a single frame of image data to the underlying FrameSource.
     * @param frame The frame data to be written as ByteBuffer
     */
    public void write(ByteBuffer frame) {
        Contracts.throwIfTrue(disposed, "disposed");
        FrameWriterJNI.writeWithProperties(frameWriterHandle, 0, frame, null);
    }

    /**
     * Writes a single frame of image data to the underlying FrameSource.
     * @param frame The frame data to be written as Frame object
     */
    public void write(Frame frame) {
        Contracts.throwIfTrue(disposed, "disposed");
        if (frame != null) {
            FrameWriterJNI.writeWithProperties(frameWriterHandle, 0, frame.getData(), frame.getProperties().getHandle());
        }
        else {
            FrameWriterJNI.writeWithProperties(frameWriterHandle, 0, null, null);
        }
    }

    /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }
        this.frameWriterHandle.close();
        disposed = true;
    }

    private final SafeHandle frameWriterHandle;
    private boolean disposed;
}
