//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;

import com.azure.ai.vision.common.ImageWriter;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.FrameSourceJNI;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import java.util.function.Consumer;

/**
 * ImageSourceBuffer class
 * Represents a source of image data, used as input to or output from Vision AI operations.
 * <p>
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class ImageSourceBuffer implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Constructs a new {@link ImageSourceBuffer} object
     */
    public ImageSourceBuffer() {
        this.imageSourceBufferHandle = FrameSourceJNI.createFrameSourceHandle("", "", null);
    }

    /**
     * Gets a new instance of {@link ImageWriter} that can be used to write (submit) image data to this {@link ImageSourceBuffer} instance.
     * @return A new ImageWriter instance associated to this {@link ImageSourceBuffer}.
     */
    public ImageWriter getWriter() {
        Contracts.throwIfTrue(disposed, "disposed");
        SafeHandle writerHandle = FrameSourceJNI.getFrameSourceWriterHandle(this.imageSourceBufferHandle.getValue());
        return new ImageWriter(writerHandle);
    }

    /**
     * Closes the frame source for writing.
     */
    public void closeWriter() {
        Contracts.throwIfTrue(disposed, "disposed");
        FrameSourceJNI.frameSourceCloseWriter(this.imageSourceBufferHandle.getValue());
    }

    /*! \cond INTERNAL */

    /**
     * Gets the internal native handle.
     * @return Native handle value
     */
    public long getHandle() {
        return this.imageSourceBufferHandle.getValue();
    }

    /*! \endcond */

    /**
     * Explicitly frees any external resource attached to the object
     */
    @Override
    public void close() {

        if (this.disposed) return;

        if (this.imageSourceBufferHandle != null) {
            this.imageSourceBufferHandle.close();
        }
        this.disposed = true;
    }

    private SafeHandle imageSourceBufferHandle;
    private boolean disposed = false;
}
