//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.nio.ByteBuffer;

import com.azure.ai.vision.common.ImageSourceBuffer;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.FrameWriterJNI;

/**
 * ImageWriter class
 * Represents the ability to write image data, for use as input to Vision AI operations.
 * <p>
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class ImageWriter implements AutoCloseable {

    ImageWriter(SafeHandle handle) {
        Contracts.throwIfNull(handle, "handle");
        this.imageWriterHandle = handle;
    }

    /**
     * Writes a single image to the internal buffer.
     * <p>
     * When used with ImageAnalyzer, the image needs to be in a format that's supported by the Image Analysis service, such as JPEG, PNG or BMP.
     * See <a href="https://aka.ms/ia-input">full list of supported formats</a>.
     * The SDK sends the image buffer as-is to the service. No format conversion is done.
     *
     * @param image The image to be written as ByteBuffer
     */
    public void write(ByteBuffer image) {
        Contracts.throwIfTrue(disposed, "disposed");
        FrameWriterJNI.writeWithProperties(imageWriterHandle, 0, image, null);
    }

    /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        if (disposed) {
            return;
        }
        this.imageWriterHandle.close();
        disposed = true;
    }

    private final SafeHandle imageWriterHandle;
    private boolean disposed;
}
