//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.IntRef;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;

/**
 * Represents collection of properties and their values.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class PropertyCollection implements AutoCloseable {

    /**
     * Internal constructor for PropertyCollection
     * 
     * @param propHandle the native handle to the property collection
     */
    public PropertyCollection(SafeHandle propHandle) {
        Contracts.throwIfNull(propHandle, "propHandle");
        propertiesHandle = propHandle;
    }

    /**
     * Returns the property value.
     * If the name is not available, it returns a default value.
     *
     * @param propertyName The property name.
     * @return Value of the property.
     */
    public String getProperty(String propertyName) {
        Contracts.throwIfNull(propertiesHandle, "collection");
        Contracts.throwIfNullOrWhitespace(propertyName, "propertyName");
        return PropertiesJNI.getPropertyString(propertiesHandle, 0, propertyName, "");
    }

    /**
     * Returns the binary property value.
     * If the name is not available, it returns null.
     *
     * @param propertyName The property name.
     * @return Value of the property.
     */
    public byte[] getPropertyBinary(String propertyName) {
        Contracts.throwIfNull(propertiesHandle, "collection");
        Contracts.throwIfNullOrWhitespace(propertyName, "propertyName");
        return PropertiesJNI.getPropertyBinary(propertiesHandle, 0, propertyName);
    }

    /**
     * Returns the property value.
     * If the name is not available, it returns a default value.
     *
     * @param propertyName The property name.
     * @param defaultValue The default value which is returned if the property is not available in the collection.
     * @return Value of the property.
     */
    public String getProperty(String propertyName, String defaultValue) {
        Contracts.throwIfNull(propertiesHandle, "collection");
        Contracts.throwIfNullOrWhitespace(propertyName, "propertyName");
        return PropertiesJNI.getPropertyString(propertiesHandle, 0, propertyName, defaultValue);
    }

    /**
     * Sets the property value by name
     *
     * @param propertyName The property name.
     * @param value The value of the property.
     */
    public void setProperty(String propertyName, String value) {
        Contracts.throwIfNull(propertiesHandle, "collection");
        Contracts.throwIfNullOrWhitespace(propertyName, "propertyName");
        Contracts.throwIfNull(value, "value");
        PropertiesJNI.setPropertyString(propertiesHandle, 0, propertyName, value);
    }

    /**
     * Dispose of the associated parameter value collection.
     */
    @Override
    public void close() {

        if (propertiesHandle != null) {
            propertiesHandle.close();
            propertiesHandle = null;
        }
    }

    /**
     * Gets the property collection handle
     * @return The property collection handle
     */
    public SafeHandle getHandle() {
        return propertiesHandle;
    }

    private SafeHandle propertiesHandle = null;
}
