//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.IntRef;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;
import com.azure.core.http.HttpClient;


/**
 * VisionServiceAdvancedOptions class
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class VisionServiceAdvancedOptions implements AutoCloseable {

    VisionServiceAdvancedOptions() {
        SafeHandle propertiesHandle = PropertiesJNI.createPropertiesHandle();
        this.propertyCollection = new PropertyCollection(propertiesHandle);
        this.propertyCollection.setProperty("AZAC-SDK-PROGRAMMING-LANGUAGE", "Java");
    }

    /**
     * Gets the http proxy.
     * @return HTTP proxy host string value
     */
    public final String getHttpProxy() {
        return this.propertyCollection.getProperty(VisionServiceOption.HTTP_PROXY_HOST);
    }

    /**
     * Sets the http proxy.
     * @param value The value for HTTP proxy host
     */
    public final void setHttpProxy(String value) {
        this.propertyCollection.setProperty(VisionServiceOption.HTTP_PROXY_HOST, value);
    }

    /**
     * Gets the http proxy port.
     * @return Port value as integer
     */
    public final int getHttpProxyPort() {
        String value = this.propertyCollection.getProperty(VisionServiceOption.HTTP_PROXY_PORT);
        return value.isEmpty() ? 0 : Integer.parseInt(value);
    }

    /**
     * Sets the http proxy port.
     * @param value The value for proxy port
     */
    public final void setHttpProxyPort(int value) {
        this.propertyCollection.setProperty(VisionServiceOption.HTTP_PROXY_PORT, String.valueOf(value));
    }

    /**
     * Gets the http proxy user name.
     * @return HTTP proxy user name string
     */
    public final String getHttpProxyUserName() {
        return this.propertyCollection.getProperty(VisionServiceOption.HTTP_PROXY_USERNAME);
    }

    /**
     * Sets the http proxy user name.
     * @param value The value for HTTP proxy user name
     */
    public final void setHttpProxyUserName(String value) {
        this.propertyCollection.setProperty(VisionServiceOption.HTTP_PROXY_USERNAME, value);
    }

    /**
     * Gets the http proxy password.
     * @return HTTP proxy password as string
     */
    public final String getHttpProxyPassword() {
        return this.propertyCollection.getProperty(VisionServiceOption.HTTP_PROXY_PASSWORD);
    }

    /**
     * Sets the http proxy password.
     * @param value The value for HTTP proxy password as string
     */
    public final void setHttpProxyPassword(String value) {
        this.propertyCollection.setProperty(VisionServiceOption.HTTP_PROXY_PASSWORD, value);
    }

    /**
     * Gets the property collection.
     * @return PropertyCollection object for this VisionServiceAdvancedOptions instance
     */
    public final PropertyCollection getProperties() {
        return propertyCollection;
    }

    /**
     * Gets an alternate HTTP client (if set) to be used for REST calls.
     * @return HttpClient object for this VisionServiceAdvancedOptions instance
     */
    public final HttpClient getHttpClient() { return httpClient; }

    /**
     * Sets an alternate HTTP client to be used for REST calls.
     * @param value as HttpClient object
     */
    public final void setHttpClient(HttpClient value) { httpClient = value;}
  
    /**
     * Explicitly frees any external resource attached to the object
     */
    public final void close() {

        if (this.propertyCollection != null) {
            this.propertyCollection.close();
            this.propertyCollection = null;
        }
    }

    private PropertyCollection propertyCollection;
    private HttpClient httpClient;
}
