//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import java.lang.AutoCloseable;

import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.IntRef;
import com.azure.ai.vision.common.internal.implementation.VisionServiceOptionsJNI;
import com.azure.ai.vision.common.implementation.NativeLibraryLoader;
import com.azure.ai.vision.common.VisionServiceAdvancedOptions;
import com.azure.core.credential.TokenCredential;

/**
 * VisionServiceOptions class represents the Vision service options.
 * <p>
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class VisionServiceOptions implements AutoCloseable {

    // Load the native library. Hold the class active so the
    // class GC does not reclaim it (and the local variables!)
    static Class<?> visionServiceOptionsClass = null;
    static {

        // load the native libraries associated with common package
        try {
            NativeLibraryLoader loader = new NativeLibraryLoader();
            loader.loadNativeBinding();
        }
        catch(java.lang.Error ex) {
            throw new UnsatisfiedLinkError(
                String.format("Could not load a required Vision SDK library because of the following error: %s", ex.getMessage()));
        }
        catch(java.lang.Exception ex2) {
            throw new UnsatisfiedLinkError(
                String.format("Could not load a required Vision SDK library because of the following error: %s", ex2.getMessage()));
        }
        // setup native tmpdir
        VisionServiceOptionsJNI.setTempDir(System.getProperty("java.io.tmpdir"));

        // prevent classgc from freeing this class
        visionServiceOptionsClass = VisionServiceOptions.class;
    }

    /**
     * Initializes a new instance of Vision service options
     * @param url An endpoint url to the service
    */
    public VisionServiceOptions(java.net.URL url) {
        Contracts.throwIfNull(url, "url");

        advancedOptions = new VisionServiceAdvancedOptions();
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
    }

    /**
     * Initializes a new instance of Vision service options.
     * @param url An endpoint url to the service
     * @param options A type of the {@link VisionServiceAdvancedOptions} options.
    */
    public VisionServiceOptions(java.net.URL url, VisionServiceAdvancedOptions options) {
        Contracts.throwIfNull(url, "url");
        Contracts.throwIfNull(options, "options");

        advancedOptions = options;
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
    }

    /**
     * Initializes a new instance of Vision service options.
     * @param url An endpoint url to the service
     * @param key A key to the service
    */
    public VisionServiceOptions(java.net.URL url, String key) {
        Contracts.throwIfNull(url, "url");
        Contracts.throwIfNullOrWhitespace(key, "key");

        advancedOptions = new VisionServiceAdvancedOptions();
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
        advancedOptions.getProperties().setProperty("service.auth.key", key);
    }

    /**
     * Initializes a new instance of Vision service options.
     * @param url An endpoint url to the service
     * @param key A key to the service
     * @param options A type of the {@link VisionServiceAdvancedOptions} options.
    */
    public VisionServiceOptions(java.net.URL url, String key, VisionServiceAdvancedOptions options) {
        Contracts.throwIfNull(url, "url");
        Contracts.throwIfNullOrWhitespace(key, "key");
        Contracts.throwIfNull(options, "options");

        advancedOptions = options;
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
        advancedOptions.getProperties().setProperty("service.auth.key", key);
    }
    
    /**
     * Initializes a new instance of Vision service options.
     * @param url An endpoint url to the service
     * @param credential A TokenCredential to generate AccessToken on demand
    */
    public VisionServiceOptions(java.net.URL url, TokenCredential credential) {
        Contracts.throwIfNull(url, "url");
        Contracts.throwIfNull(credential, "credential");

        tokenCredential = credential;
        advancedOptions = new VisionServiceAdvancedOptions();
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
    }

    /**
     * Initializes a new instance of Vision service options.
     * @param url An endpoint url to the service
     * @param credential A TokenCredential to generate AccessToken on demand
     * @param options A type of the {@link VisionServiceAdvancedOptions} options.
    */
    public VisionServiceOptions(java.net.URL url, TokenCredential credential, VisionServiceAdvancedOptions options) {
        Contracts.throwIfNull(url, "url");
        Contracts.throwIfNull(credential, "credential");
        Contracts.throwIfNull(options, "options");

        tokenCredential = credential;
        advancedOptions = options;
        advancedOptions.getProperties().setProperty("service.endpoint", url.toString());
    }

    /**
     * Gets a token credential object
     * @return TokenCredential object for this {@link VisionServiceAdvancedOptions} instance
     */
    public final TokenCredential getTokenCredential() { return tokenCredential; }

    /**
     * Sets a token credential object to refresh auth tokens
     * @param value as TokenCredential object
     */
    public final void setTokenCredential(TokenCredential value) { tokenCredential = value;}

    /**
     * Gets the advanced options of the vision service options.
     * @return The {@link VisionServiceAdvancedOptions} object.
     */
    public final VisionServiceAdvancedOptions getAdvanced() {
        return advancedOptions;
    }

    /**
     * Dispose of associated resources.
     * 
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public final void close() {
        if (disposed) {
            return;
        }

        if (advancedOptions != null) {
            advancedOptions.close();
            advancedOptions = null;
        }
        disposed = true;
    }

    private VisionServiceAdvancedOptions advancedOptions = null;
    private boolean disposed = false;
    private TokenCredential tokenCredential;
}
