//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common;

import com.azure.ai.vision.common.FrameSource;
import com.azure.ai.vision.common.ImageSourceBuffer;
import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.VisionServiceOptions;
import com.azure.ai.vision.common.internal.implementation.Contracts;
import com.azure.ai.vision.common.internal.implementation.FrameSourceJNI;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.VisionSessionJNI;
import com.azure.ai.vision.common.internal.implementation.VisionSourceJNI;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * VisionSource class represents a source for vision-related operations.
 * <p>
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public final class VisionSource implements AutoCloseable {

    // load the native library.
    static {
        // trigger loading of native library
        try {
            Class.forName(VisionServiceOptions.class.getName());
        }
        catch (ClassNotFoundException ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Internal constructor to create VisionSource from native handle
     *
     * @param handle The native handle
     */
    VisionSource(SafeHandle handle) {
        visionSourceHandle = handle;
        this.propertyCollection = new PropertyCollection(VisionSourceJNI.getSourcePropertiesHandle(handle));
    }

    /**
     * Create VisionSource from an {@link ImageSourceBuffer}, allowing the application
     * to pass in an image from a memory buffer.
     *
     * @param source The image source buffer containing the image to process.
     * @return A new VisionSource instance that uses the specified input source buffer.
     */
    public static VisionSource fromImageSourceBuffer(ImageSourceBuffer source) {
        Contracts.throwIfNull(source, "source");
        SafeHandle handle = VisionSourceJNI.createSourceHandle("", "", source.getHandle());
        return new VisionSource(handle);
    }

    /**
     * Create VisionSource from the frame source
     *
     * @param source The FrameSource to be bound to the VisionSource.
     * @return A new VisionSource instance.
     */
    public static VisionSource fromFrameSource(FrameSource source) {
        Contracts.throwIfNull(source, "source");

        SafeHandle handle = VisionSourceJNI.createSourceHandle("", "", source.getHandle());
        return new VisionSource(handle);
    }

    /**
     * Create VisionSource from the image file
     * 
     * @param filePath The full path name of the local file
     * @return A new VisionSource instance.
     */
    public static VisionSource fromFile(String filePath) {
        SafeHandle handle = VisionSourceJNI.createSourceHandle("source.file.name", filePath, 0);
        return new VisionSource(handle);
    }

    /**
     * Create VisionSource from the specified internet-accessible URL
     * 
     * @param url An accessible URL for the input media
     * @return A new VisionSource instance.
     */
    public static VisionSource fromUrl(java.net.URL url) {
        SafeHandle handle = VisionSourceJNI.createSourceHandle("source.url.name", url.toString(), 0);
        return new VisionSource(handle);
    }

    /**
     * Get the property collection associated to vision source object
     *
     * @return PropertyCollection object
     */
    public final PropertyCollection getProperties() {
        return propertyCollection;
    }

    /**
     * Internal method to get the native handle associated to vision source object
     *
     * @return Native handle as SafeHandle
     */
    public final SafeHandle getHandle() {
        return visionSourceHandle;
    }

    /**
     * Explicitly frees any external resource attached to the object
     * 
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public final void close() {

        if (this.propertyCollection != null) {
            this.propertyCollection.close();
            this.propertyCollection = null;
        }
        if (this.visionSourceHandle != null) {
            this.visionSourceHandle.close();
            this.visionSourceHandle = null;
        }
    }

    private SafeHandle visionSourceHandle;
    private PropertyCollection propertyCollection;

}
