//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common.implementation;

import java.lang.AutoCloseable;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.util.Arrays;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.Files;
import java.nio.file.FileVisitResult;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.DirectoryStream;
import com.azure.ai.vision.common.VisionServiceOptions;


/**
 * A internal helper class for loading native Vision SDK libraries from Java
 *
 * - Native libraries are placed in an OS-specific package assets folder
 *   (/ASSETS/{mac,linux,window}x64)
 * - From this folder, a known list of libraries is extracted into a temporary folder.
 * - On Windows and OSX, these libraries are loaded in the list order (bottom
 *   to top dependency), by full extracted path name.
 *   TODO this may need to be revisited on OSX
 * - On Linux, only the last of these libraries (top dependency) is loaded,
 *   which resolves static dependency via RPATH; dynamic dependencies will be
 *   loaded later, also via RPATH.
 * <p>
 * Workaround for a Windows: Turns out that the temporary folder created here and the native DLL files copied to it never 
 * got delete even though they were marked for deleteOnExit(). This resulted in ever-increasing disk usage on Windows in 
 * the %TEMP% folder, as reported by a customer.
 * This is because on Windows the call to System.load() holds a lock on the files and they cannot be deleted when JVM shuts
 * down (there is no way to "unload" them first since there is no System.unload in Java).
 * The workaround is to make sure next time JVM loads this class we delete all previous unused folders. To mark a folder as "in use",
 * we add an empty file next to it with the same name and extension ".lock". We call deleteOnExit() on this lock file.
 * That file should be deleted when the JVM shuts down, since we did not call System.load() on that file. Hence folders without
 * a corresponding .lock file next to them are okay to be deleted when this class loads.
 * For example, the code in this class creates the following lock file and folder (here "353862700187557707" is an example
 * of the random number Windows picks):
 *   C:/Users/YourUserName/AppData/Local/Temp/vision-sdk-common-native-353862700187557707.lock - An empty file indicating the folder below is in use.
 *   C:/Users/YourUserName/AppData/Local/Temp/vision-sdk-common-native-353862700187557707      - A folder containing the native .dll files.
 * Deleting unlocked folders is done in a separate thread, so we do not slow down the loading of this class for the (very unlikely)
 * case where there are many unlocked folders to delete (e.g. when the device previously run many multiple recognizers in parallel).
 * We start running the cleanup thread *after* the new local temp folder is created and locked.
 * @hidden
 */
public class NativeLibraryLoader {

    /**
     * Internal class to represent a native library.
     */
    protected final class NativeLibrary {
        private String name;
        private boolean required;

        /**
         * Constructor for NativeLibrary.
         * @param name Name of the native library
         * @param required Whether the native library is required
         */
        public NativeLibrary(String name, boolean required) {
            this.name = name;
            this.required = required;
        }

        /**
         * Gets the name of the native library.
         * @return Name of the native library
         */
        public String getName() { return name; }

        /**
         * Gets whether the native library is required.
         * @return Whether the native library is required
         */
        public boolean getRequired() { return required; }
    }

    /**
     * Internal string variable to represent the operating system.
     */
    protected static String operatingSystem;

    private static String tempDirPrefix = "vision-sdk-common-native-";
    private NativeLibrary[] nativeList = new NativeLibrary[0];
    private static File tempDir;
    private static Boolean loaded = false;
    private static final String lockFileExtension = ".lock";
    private static Boolean loadAll = false;

    static {
        operatingSystem = ("" + System.getProperty("os.name")).toLowerCase();

        try {
            if (operatingSystem.contains("windows")) {

                // Start by creating the lock file, which will guard the folder to be created next
                final File lockFile = Files.createTempFile(tempDirPrefix, lockFileExtension).toFile();
                lockFile.createNewFile();
                lockFile.deleteOnExit();

                // Create a folder with a name derived from the lock file name, where the native binaries will be copied
                String tempDirName = lockFile.getAbsolutePath();
                tempDirName = tempDirName.substring(0, tempDirName.length() - lockFileExtension.length());
                tempDir = Files.createDirectory(Paths.get(tempDirName)).toFile();
                tempDir.deleteOnExit();

                // Start a thread to clean up all un-locked temp folders from other Java apps that run before
                Thread thread = new Thread(new DeleteUnlockedTempFolders());
                thread.start();

            } else {

                tempDir = Files.createTempDirectory(tempDirPrefix).toFile();
                tempDir.deleteOnExit();

            }
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    /**
     * Internal helper method to extract and load OS-specific libraries from the JAR file.
     */
    public void loadNativeBinding() throws UnsatisfiedLinkError {
        try {
            if (!loaded) {
                extractNativeLibraries();
                loadExtractedNativeLibraries();
            }
        }
        catch (Exception e) {
            throw new UnsatisfiedLinkError(
                    String.format("Could not extract/load all Vision SDK libraries because we encountered the following error: %s", e.getMessage()));
        }
        finally {
            loaded = true;
        }
    }

    /**
     * Internal helper method to load all native libraries from the temporary folder.
     * 
     * @throws IOException if the extraction fails to write to temporary location
     * @throws UnsatisfiedLinkError if the native library fails to load
     */
    protected void loadExtractedNativeLibraries() throws IOException, UnsatisfiedLinkError {

        // Load extracted libraries (either all or the last one)
        for (int i = 0; i < nativeList.length; i++)
        {
            if (loadAll || (i == nativeList.length - 1)) {
                String libName = nativeList[i].getName();
                String fullPathName = new File(tempDir.getCanonicalPath(), libName).getCanonicalPath();
                if(!fullPathName.startsWith(tempDir.getCanonicalPath())) {
                    throw new SecurityException("illegal path");
                }
                try {
                    //System.out.println("NativeLibraryLoader, loadNativeBinding, System.load: " + fullPathName);
                    System.load(fullPathName);
                }
                catch (UnsatisfiedLinkError e) {
                    // Required library failed to load, throw exception
                    if (nativeList[i].getRequired() == true) {
                        throw new UnsatisfiedLinkError(
                        String.format("Could not load a required Vision SDK library because of the following error: %s", e.getMessage()));
                    }
                }
            }
        }
    }

    /**
     * Internal helper method to extract and load OS-specific libraries from the JAR file.
     * 
     * @throws Exception if the extraction fails to write to temporary location
     */
    private void extractNativeLibraries() throws Exception {
        nativeList = getResourceLines();
        // Extract all operatingSystem specific native libraries to temporary location
        for (NativeLibrary library: nativeList) {
            String prefix = getResourcesPath();
            String path = prefix + library.getName();
            extractResourceFromPath(library, VisionServiceOptions.class.getResourceAsStream(path));
        }
    }

    /**
     * Internal helper method to return the list of libraries to be extracted.
     * 
     * @return List of libraries to be extracted
     */
    private NativeLibrary[] getResourceLines() {

        if (operatingSystem.contains("linux")) {
            return new NativeLibrary[] {
                    new NativeLibrary("libAzure-AI-Vision-Native.so", true),
                    new NativeLibrary("libAzure-AI-Vision-JNI.so", true)
            };
        }
        else if (operatingSystem.contains("windows")) {
            // Note: The order of the libraries is important. The library which depends on other libraries, 
            // needs to be placed in the NativeLibrary array after the libraries it depends on.
            // For dll dependency information, you can use the Dependencies tool (https://github.com/lucasg/Dependencies)
            return new NativeLibrary[] {
                    new NativeLibrary("Azure-AI-Vision-Native.dll", true),
                    new NativeLibrary("Azure-AI-Vision-JNI.dll", true)
                };
        }
        else if (operatingSystem.contains("mac") || operatingSystem.contains("darwin")) {
            return new NativeLibrary[] {
                    new NativeLibrary("libAzure-AI-Vision-Native.dylib", true),
                    new NativeLibrary("libAzure-AI-Vision-JNI.dylib", true)
            };
        }
        return new NativeLibrary[] {};
    }

    /**
     * Internal helper method to return the resource path to jar files ASSETS folder.
     * 
     * @return Path to jar files ASSETS folder
     */
    protected String getResourcesPath() throws UnsatisfiedLinkError {

        String speechPrefix = "/ASSETS/%s%s/";

        // determine if the VM runs on 64 or 32 bit
        String dataModelSize = System.getProperty("sun.arch.data.model");
        if(dataModelSize != null && dataModelSize.equals("64")) {
            dataModelSize = "64";
        }
        else {
            dataModelSize = "32";
        }

        if (operatingSystem.contains("linux")) {
            String osArchitecture = System.getProperty("os.arch");
            if (osArchitecture.contains("aarch64") || osArchitecture.contains("arm")) {
                return String.format(speechPrefix, "linux-arm", dataModelSize);
            }
            else if (osArchitecture.contains("amd64") || osArchitecture.contains("x86_64")) {
                return String.format(speechPrefix, "linux-x", dataModelSize);
            }
        }
        else if (operatingSystem.contains("windows")) {
            loadAll = true; // signal to load all libraries
            return String.format(speechPrefix, "windows-x", dataModelSize);
        }
        else if (operatingSystem.contains("mac")|| operatingSystem.contains("darwin")) {
            String osArchitecture = System.getProperty("os.arch");
            if (osArchitecture.contains("aarch64") || osArchitecture.contains("arm")) {
                return String.format(speechPrefix, "osx-arm", dataModelSize);
            }
            else if (osArchitecture.contains("amd64") || osArchitecture.contains("x86_64")) {
                return String.format(speechPrefix, "osx-x", dataModelSize);
            }
        }

        throw new UnsatisfiedLinkError(
                String.format("The Vision SDK doesn't currently have native support for operating system: %s data model size %s", operatingSystem, dataModelSize));
    }

    /**
     * Internal helper method to extract a resource from the jar file to a temporary location.
     * 
     * @param library The library to be extracted
     * @param inputStream The input stream to the library in installed jar file
     * @throws IOException if the extraction fails to write to temporary location
     */
    protected void extractResourceFromPath(NativeLibrary library, InputStream inputStream) throws IOException {
        String libName = library.getName();
        File temp = new File(tempDir.getCanonicalPath(), libName);
        if (!temp.getCanonicalPath().startsWith(tempDir.getCanonicalPath())) {
            throw new SecurityException("illegal name " + temp.getCanonicalPath());
        }

        temp.createNewFile();
        temp.deleteOnExit();

        if (!temp.exists()) {
            throw new FileNotFoundException(String.format(
                    "Temporary file %s could not be created. Make sure you can write to this location.",
                    temp.getCanonicalPath()));
        }

        InputStream inStream = inputStream;
        if (inStream == null) {
            if (library.getRequired() == true) {
                throw new FileNotFoundException(String.format("Could not find resources for %s in jar.", libName));
            }
            else {
                // Optional library
                return;
            }
        }

        FileOutputStream outStream = null;
        byte[] buffer = new byte[1024*1024];
        int bytesRead;

        try {
            outStream = new FileOutputStream(temp);
            while ((bytesRead = inStream.read(buffer)) >= 0) {
                outStream.write(buffer, 0, bytesRead);
            }
        } catch (IOException e) {
            throw new IOException(String.format("Could not write to temporary file %s.", temp.getCanonicalPath()), e);
        }
        finally {
            safeClose(outStream);
            safeClose(inStream);
        }
    }

    /**
     * Safely close a stream.
     * 
     * @param is The stream to close
     */
    private void safeClose(Closeable is) {
        if (is != null) {
            try {
                is.close();
            } catch (IOException e) {
                // ignored.
            }
        }
    }

    /**
     * Delete all temp folders that are not locked by a running process.
     */
    private static class DeleteUnlockedTempFolders implements Runnable {

        @Override
        public void run() {

            // Define a filter that will return all existing temp folders with names that start with "vision-sdk-native-".
            // Note that we exclude the *.lock suffix, in order to only capture the folders, not their corresponding .lock file.
            FileFilter tmpDirFilter = new FileFilter() {
                public boolean accept(File pathname) {
                    return pathname.getName().startsWith(tempDirPrefix) && !pathname.getName().endsWith(lockFileExtension);
                }
            };

            // List all folders that match this filer
            File tmpDir = new File(System.getProperty("java.io.tmpdir"));
            File[] filteredDirList = tmpDir.listFiles(tmpDirFilter);

            // Delete all folders which do not have a corresponding .lock file
            for (int i = 0; i < filteredDirList.length; i++) {

                // The .lock file we expect to see next to this folder if the folder is still used by some other Java app
                File lockFile = new File(filteredDirList[i].getAbsolutePath() + lockFileExtension);

                if (!lockFile.exists()) {

                    // If the *.lock file DOES NOT exist, it's safe to delete this folder. But before
                    // we can delete the folder, we need to delete all the files in that folder
                    try {
                        Files.walkFileTree(Paths.get(filteredDirList[i].getAbsolutePath()),
                            new SimpleFileVisitor<Path>() {
                                @Override
                                public FileVisitResult visitFile(Path file, BasicFileAttributes attr) throws IOException {
                                    Files.delete(file);
                                    return FileVisitResult.CONTINUE;
                                }
                            }
                        );
                    }
                    catch (IOException e) {
                        // okay to ignore, this is a best-effort delete
                    }
                    // Now that the directory is empty, we can delete it
                    filteredDirList[i].delete();
                }
            }
        }
    }
}