//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.azure.ai.vision.common.implementation;

import static java.lang.Integer.parseInt;

import java.nio.ByteBuffer;
import java.util.concurrent.*;

import com.azure.core.http.HttpMethod;
import com.azure.core.util.Context;
import com.azure.core.http.*;
import reactor.core.publisher.Mono;
import org.reactivestreams.Publisher;
import java.time.Duration;

import com.azure.ai.vision.common.PropertyCollection;
import com.azure.ai.vision.common.internal.implementation.SafeHandle;
import com.azure.ai.vision.common.internal.implementation.VisionEventArgsJNI;
import com.azure.ai.vision.common.internal.implementation.VisionResultJNI;
import com.azure.ai.vision.common.internal.implementation.PropertiesJNI;
import com.azure.ai.vision.common.Diagnostics;

/**
 * This class contains internal helper methods for the HTTP request and response handling.
 * @hidden
 */
public final class SessionUtils {

    /**
     * Internally used common helper method to send an HTTP request and wait for the response.
     * 
     * @param eventArgHandle the native event argument handle
     * @param client the HTTP client
     */
    public static void SendHttpRequest(SafeHandle eventArgHandle, HttpClient client) {

        try(SafeHandle resultHandle = VisionEventArgsJNI.getResultHandle(eventArgHandle)) {
        try(SafeHandle responsePropertiesHandle = VisionResultJNI.getPropertiesHandle(resultHandle)) {
        try(SafeHandle requestPropertiesHandle = VisionEventArgsJNI.getPropertiesHandle(eventArgHandle)) {
        try(PropertyCollection responseProperties = new PropertyCollection(responsePropertiesHandle)) {
        try(PropertyCollection requestProperties = new PropertyCollection(requestPropertiesHandle)) {

            String methodStr = requestProperties.getProperty("service.transport.http.request.method");
            HttpMethod method = HttpMethod.valueOf(methodStr);

            String uri = requestProperties.getProperty("service.transport.http.request.uri");

            HttpRequest request = new HttpRequest(method, uri);

            String allHeaders = requestProperties.getProperty("service.transport.http.request.headers");
            String[] headers = allHeaders.split("\n");
            for(String oneHeader: headers) {
                String headerValue = requestProperties.getProperty("service.transport.http.request.headers." + oneHeader);
                request.setHeader(oneHeader, headerValue);
            }

            byte[] content = PropertiesJNI.getPropertyBinary(requestPropertiesHandle, 0, "service.transport.http.request.content");
            if (content != null) {
                request.setBody(content);
            }

            // Send the request
            String endpointTimeoutSeconds = requestProperties.getProperty("service.endpoint.timeoutseconds", "60");
            int timeout = parseInt(endpointTimeoutSeconds);
            send(request, client, responseProperties, responsePropertiesHandle, timeout);

        }}}}}
        catch(Throwable e){
            e.printStackTrace();
        }
    }

    private static void send(HttpRequest request, HttpClient client, PropertyCollection responseProperties, SafeHandle responsePropertiesHandle, int timeout) {

        try {
            Mono<HttpResponse> responseMono = client.send(request, Context.NONE);
            HttpResponse response = responseMono
                .timeout(Duration.ofSeconds(timeout))
                .block();
        
            int statusCode = response.getStatusCode();
            responseProperties.setProperty("service.transport.http.response.status", String.valueOf(statusCode));
        
            String responseHeaders = "";
            for(HttpHeader header:response.getHeaders()) {
                responseHeaders += header.getName() + "\n";
                responseProperties.setProperty("service.transport.http.response.headers." + header.getName(), header.getValue());
            }
            responseProperties.setProperty("service.transport.http.response.headers", responseHeaders);
        
            byte[] responseContent = response.getBodyAsByteArray().block();
            PropertiesJNI.setPropertyBinary(responsePropertiesHandle, 0, "service.transport.http.response.content", responseContent );
        
        } catch (Exception e) {
            if (e instanceof TimeoutException) {
                Diagnostics.logMessage("SendHttpRequest exception: Timeout waiting for response.");
                PropertiesJNI.setPropertyString(responsePropertiesHandle,0,"service.transport.http.response.exception", "Timeout waiting for response.");
            } else {
                Diagnostics.logMessage("SendHttpRequest exception: " + e.getLocalizedMessage());
                PropertiesJNI.setPropertyString(responsePropertiesHandle,0,"service.transport.http.response.exception", e.getLocalizedMessage());
            }
        }
    }

}
