/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.amqp.implementation.handler;

import com.azure.core.amqp.implementation.AmqpLoggingUtils;
import com.azure.core.amqp.implementation.handler.DeliverySettleMode;
import com.azure.core.amqp.implementation.handler.MessageWithDeliveryTag;
import com.azure.core.amqp.implementation.handler.ReceiverUnsettledDeliveries;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LoggingEventBuilder;
import java.nio.ByteBuffer;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.qpid.proton.Proton;
import org.apache.qpid.proton.amqp.Symbol;
import org.apache.qpid.proton.amqp.messaging.Accepted;
import org.apache.qpid.proton.amqp.messaging.Modified;
import org.apache.qpid.proton.amqp.transport.DeliveryState;
import org.apache.qpid.proton.amqp.transport.ErrorCondition;
import org.apache.qpid.proton.engine.Delivery;
import org.apache.qpid.proton.engine.EndpointState;
import org.apache.qpid.proton.engine.Link;
import org.apache.qpid.proton.engine.Receiver;
import org.apache.qpid.proton.message.Message;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Sinks;

final class ReceiverDeliveryHandler {
    static final UUID DELIVERY_EMPTY_TAG = new UUID(0L, 0L);
    private static final int DELIVERY_TAG_SIZE = 16;
    private final AtomicBoolean isTerminated = new AtomicBoolean();
    private final AtomicBoolean isLinkTerminatedWithError = new AtomicBoolean();
    private final Sinks.Many<Message> messages = Sinks.many().multicast().onBackpressureBuffer();
    private final String entityPath;
    private final String receiveLinkName;
    private final DeliverySettleMode settlingMode;
    private final boolean includeDeliveryTagInMessage;
    private final ClientLogger logger;
    private final ReceiverUnsettledDeliveries unsettledDeliveries;

    ReceiverDeliveryHandler(String entityPath, String receiveLinkName, DeliverySettleMode settlingMode, ReceiverUnsettledDeliveries unsettledDeliveries, boolean includeDeliveryTagInMessage, ClientLogger logger) {
        this.entityPath = entityPath;
        this.receiveLinkName = receiveLinkName;
        this.settlingMode = settlingMode;
        this.unsettledDeliveries = unsettledDeliveries;
        this.includeDeliveryTagInMessage = includeDeliveryTagInMessage;
        this.logger = logger;
    }

    void onDelivery(Delivery delivery) {
        if (this.isPartialOrSettledDelivery(delivery) || this.isDeliverySettledOnClosedLink(delivery)) {
            return;
        }
        switch (this.settlingMode) {
            case SETTLE_ON_DELIVERY: {
                this.handleSettleOnDelivery(delivery);
                break;
            }
            case ACCEPT_AND_SETTLE_ON_DELIVERY: {
                this.handleAcceptAndSettleOnDelivery(delivery);
                break;
            }
            case SETTLE_VIA_DISPOSITION: {
                this.handleSettleViaDisposition(delivery);
                break;
            }
            default: {
                throw this.logger.logExceptionAsError(new RuntimeException("settlingMode is not supported: " + (Object)((Object)this.settlingMode)));
            }
        }
    }

    void onLinkError() {
        this.isLinkTerminatedWithError.set(true);
    }

    Flux<Message> getMessages() {
        return this.messages.asFlux();
    }

    void preClose() {
        this.isTerminated.set(true);
    }

    public void close(String errorMessage) {
        this.isTerminated.set(true);
        this.messages.emitComplete((signalType, emitResult) -> {
            this.logger.atVerbose().addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName).addKeyValue("emitResult", (Object)emitResult).log(errorMessage);
            return false;
        });
    }

    private boolean isPartialOrSettledDelivery(Delivery delivery) {
        if (delivery.isPartial()) {
            Link link = delivery.getLink();
            if (link != null) {
                ErrorCondition condition = link.getRemoteCondition();
                AmqpLoggingUtils.addErrorCondition(this.logger.atVerbose(), condition).addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName).addKeyValue("updatedLinkCredit", (long)link.getCredit()).addKeyValue("remoteCredit", (long)link.getRemoteCredit()).addKeyValue("delivery.isPartial", true).addKeyValue("delivery.isSettled", delivery.isSettled()).log("onDelivery.");
            } else {
                this.logger.atWarning().addKeyValue("entityPath", this.entityPath).addKeyValue("delivery.isSettled", true).log("Partial delivery with no link.");
            }
            return true;
        }
        if (delivery.isSettled()) {
            Link link = delivery.getLink();
            if (link != null) {
                AmqpLoggingUtils.addErrorCondition(this.logger.atInfo(), link.getRemoteCondition()).addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName).addKeyValue("updatedLinkCredit", (long)link.getCredit()).addKeyValue("remoteCredit", (long)link.getRemoteCredit()).addKeyValue("delivery.isSettled", true).log("onDelivery. Was already settled.");
            } else {
                this.logger.atWarning().addKeyValue("entityPath", this.entityPath).addKeyValue("delivery.isSettled", true).log("Settled delivery with no link.");
            }
            return true;
        }
        return false;
    }

    private boolean isDeliverySettledOnClosedLink(Delivery delivery) {
        Link link = delivery.getLink();
        if (link != null && link.getLocalState() == EndpointState.CLOSED) {
            delivery.disposition((DeliveryState)new Modified());
            delivery.settle();
            return true;
        }
        return false;
    }

    private void handleSettleOnDelivery(Delivery delivery) {
        Message message;
        boolean wasSettled = delivery.isSettled();
        try {
            message = this.readAndDecodeTransferDeliveryMessage(delivery, null);
            delivery.settle();
        }
        catch (RuntimeException decodeError) {
            this.handleDeliveryDecodeError(decodeError);
            return;
        }
        this.logOnDelivery(delivery, null, wasSettled);
        this.emitMessage(message, delivery);
    }

    private void handleAcceptAndSettleOnDelivery(Delivery delivery) {
        Message message;
        boolean wasSettled = delivery.isSettled();
        try {
            message = this.readAndDecodeTransferDeliveryMessage(delivery, null);
            delivery.disposition((DeliveryState)Accepted.getInstance());
            delivery.settle();
        }
        catch (RuntimeException decodeError) {
            this.handleDeliveryDecodeError(decodeError);
            return;
        }
        this.logOnDelivery(delivery, null, wasSettled);
        this.emitMessage(message, delivery);
    }

    private void handleSettleViaDisposition(Delivery delivery) {
        boolean wasSettled = delivery.isSettled();
        UUID deliveryTag = ReceiverDeliveryHandler.decodeDeliveryTag(delivery);
        if (!this.unsettledDeliveries.containsDelivery(deliveryTag)) {
            Message message;
            try {
                message = this.readAndDecodeTransferDeliveryMessage(delivery, deliveryTag);
                delivery.getLink().advance();
            }
            catch (RuntimeException decodeError) {
                this.handleDeliveryDecodeError(decodeError);
                return;
            }
            if (this.unsettledDeliveries.onDelivery(deliveryTag, delivery)) {
                this.logOnDelivery(delivery, deliveryTag, wasSettled);
                this.emitMessage(message, delivery);
            } else {
                delivery.disposition((DeliveryState)new Modified());
                delivery.settle();
            }
        } else {
            this.unsettledDeliveries.onDispositionAck(deliveryTag, delivery);
        }
    }

    private Message readAndDecodeTransferDeliveryMessage(Delivery delivery, UUID deliveryTag) {
        int messageSize = delivery.pending();
        byte[] buffer = new byte[messageSize];
        int read = ((Receiver)delivery.getLink()).recv(buffer, 0, messageSize);
        Message message = Proton.message();
        message.decode(buffer, 0, read);
        if (this.includeDeliveryTagInMessage) {
            if (deliveryTag == null) {
                return new MessageWithDeliveryTag(message, ReceiverDeliveryHandler.decodeDeliveryTag(delivery));
            }
            return new MessageWithDeliveryTag(message, deliveryTag);
        }
        return message;
    }

    private void handleDeliveryDecodeError(RuntimeException decodeError) {
        if (!(decodeError instanceof IllegalStateException) || !this.isLinkTerminatedWithError.get() && !this.isTerminated.get()) {
            this.emitError(new IllegalStateException("Unexpected error when decoding Delivery.", decodeError));
            throw decodeError;
        }
        this.emitError(new IllegalStateException("Cannot decode Delivery when ReactorReceiver instance is closed.", decodeError));
    }

    private void emitMessage(Message message, Delivery delivery) {
        this.messages.emitNext((Object)message, (signalType, emitResult) -> {
            this.logger.atWarning().addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName).addKeyValue("emitResult", (Object)emitResult).addKeyValue("delivery", (Object)delivery).log("Could not emit delivery.");
            Link link = delivery.getLink();
            if (emitResult == Sinks.EmitResult.FAIL_OVERFLOW && link.getLocalState() != EndpointState.CLOSED) {
                link.setCondition(new ErrorCondition(Symbol.getSymbol((String)"delivery-buffer-overflow"), "Deliveries are not processed fast enough. Closing local link."));
                link.close();
                return true;
            }
            return false;
        });
    }

    private void emitError(IllegalStateException error) {
        this.messages.emitError((Throwable)error, (signalType, emitResult) -> {
            this.logger.atVerbose().addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName).addKeyValue("emitResult", (Object)emitResult).log("Could not emit messages.error.", new Object[]{error});
            return false;
        });
    }

    private void logOnDelivery(Delivery delivery, UUID deliveryTag, boolean wasSettled) {
        Link link = delivery.getLink();
        if (link == null) {
            return;
        }
        ErrorCondition condition = link.getRemoteCondition();
        LoggingEventBuilder loggingEvent = AmqpLoggingUtils.addErrorCondition(this.logger.atVerbose(), condition).addKeyValue("entityPath", this.entityPath).addKeyValue("linkName", this.receiveLinkName);
        if (deliveryTag != null) {
            loggingEvent.addKeyValue("lockToken", (Object)deliveryTag);
        }
        loggingEvent.addKeyValue("updatedLinkCredit", (long)link.getCredit()).addKeyValue("remoteCredit", (long)link.getRemoteCredit()).addKeyValue("delivery.isSettled", wasSettled).log("onDelivery.");
    }

    private static UUID decodeDeliveryTag(Delivery delivery) {
        byte[] deliveryTag = delivery.getTag();
        if (deliveryTag == null || deliveryTag.length != 16) {
            return DELIVERY_EMPTY_TAG;
        }
        byte[] reorderedBytes = new byte[16];
        for (int i = 0; i < 16; ++i) {
            switch (i) {
                case 0: {
                    int indexInReorderedBytes = 3;
                    break;
                }
                case 1: {
                    int indexInReorderedBytes = 2;
                    break;
                }
                case 2: {
                    int indexInReorderedBytes = 1;
                    break;
                }
                case 3: {
                    int indexInReorderedBytes = 0;
                    break;
                }
                case 4: {
                    int indexInReorderedBytes = 5;
                    break;
                }
                case 5: {
                    int indexInReorderedBytes = 4;
                    break;
                }
                case 6: {
                    int indexInReorderedBytes = 7;
                    break;
                }
                case 7: {
                    int indexInReorderedBytes = 6;
                    break;
                }
                default: {
                    int indexInReorderedBytes = i;
                }
            }
            reorderedBytes[indexInReorderedBytes] = deliveryTag[i];
        }
        ByteBuffer buffer = ByteBuffer.wrap(reorderedBytes);
        long mostSignificantBits = buffer.getLong();
        long leastSignificantBits = buffer.getLong();
        return new UUID(mostSignificantBits, leastSignificantBits);
    }
}

