/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.experimental.implementation;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenCredential;
import com.azure.core.credential.TokenRequestContext;
import com.azure.core.util.logging.ClientLogger;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;
import reactor.core.publisher.Sinks;

public class AccessTokenCacheImpl {
    private static final Duration REFRESH_DELAY = Duration.ofSeconds(30L);
    private static final Duration REFRESH_OFFSET = Duration.ofMinutes(5L);
    private volatile AccessToken cache;
    private volatile OffsetDateTime nextTokenRefresh = OffsetDateTime.now();
    private final AtomicReference<Sinks.One<AccessToken>> wip;
    private final TokenCredential tokenCredential;
    private TokenRequestContext tokenRequestContext;
    private final Predicate<AccessToken> shouldRefresh;
    private final ClientLogger logger = new ClientLogger(AccessTokenCacheImpl.class);

    public AccessTokenCacheImpl(TokenCredential tokenCredential) {
        Objects.requireNonNull(tokenCredential, "The token credential cannot be null");
        this.wip = new AtomicReference();
        this.tokenCredential = tokenCredential;
        this.shouldRefresh = accessToken -> OffsetDateTime.now().isAfter(accessToken.getExpiresAt().minus(REFRESH_OFFSET));
    }

    public Mono<AccessToken> getToken(TokenRequestContext tokenRequestContext) {
        return Mono.defer(this.retrieveToken(tokenRequestContext)).repeatWhenEmpty(longFlux -> longFlux.concatMap(ignored -> Flux.just((Object)true)));
    }

    private Supplier<Mono<? extends AccessToken>> retrieveToken(TokenRequestContext tokenRequestContext) {
        return () -> {
            try {
                if (this.wip.compareAndSet(null, (Sinks.One<AccessToken>)Sinks.one())) {
                    Mono fallback;
                    Mono tokenRefresh;
                    Sinks.One<AccessToken> sinksOne = this.wip.get();
                    OffsetDateTime now = OffsetDateTime.now();
                    Supplier<Mono> tokenSupplier = () -> this.tokenCredential.getToken(this.tokenRequestContext);
                    boolean forceRefresh = this.checkIfWeShouldForceRefresh(tokenRequestContext);
                    if (forceRefresh) {
                        this.tokenRequestContext = tokenRequestContext;
                        tokenRefresh = Mono.defer(() -> this.tokenCredential.getToken(this.tokenRequestContext));
                        fallback = Mono.empty();
                    } else if (this.cache != null && !this.shouldRefresh.test(this.cache)) {
                        tokenRefresh = Mono.empty();
                        fallback = Mono.just((Object)this.cache);
                    } else if (this.cache == null || this.cache.isExpired()) {
                        tokenRefresh = now.isAfter(this.nextTokenRefresh) ? Mono.defer(tokenSupplier) : Mono.defer(tokenSupplier).delaySubscription(Duration.between(now, this.nextTokenRefresh));
                        fallback = Mono.empty();
                    } else {
                        tokenRefresh = now.isAfter(this.nextTokenRefresh) ? Mono.defer(tokenSupplier) : Mono.empty();
                        fallback = Mono.just((Object)this.cache);
                    }
                    return tokenRefresh.materialize().flatMap(this.processTokenRefreshResult(sinksOne, now, (Mono<AccessToken>)fallback)).doOnError(arg_0 -> sinksOne.tryEmitError(arg_0)).doFinally(ignored -> this.wip.set(null));
                }
                return Mono.empty();
            }
            catch (Exception ex) {
                return Mono.error((Throwable)ex);
            }
        };
    }

    private boolean checkIfWeShouldForceRefresh(TokenRequestContext tokenRequestContext) {
        return this.tokenRequestContext == null || !(this.tokenRequestContext.getClaims() == null ? tokenRequestContext.getClaims() == null : tokenRequestContext.getClaims() != null && tokenRequestContext.getClaims().equals(this.tokenRequestContext.getClaims())) || !this.tokenRequestContext.getScopes().equals(tokenRequestContext.getScopes());
    }

    private Function<Signal<AccessToken>, Mono<? extends AccessToken>> processTokenRefreshResult(Sinks.One<AccessToken> sinksOne, OffsetDateTime now, Mono<AccessToken> fallback) {
        return signal -> {
            AccessToken accessToken = (AccessToken)signal.get();
            Throwable error = signal.getThrowable();
            if (signal.isOnNext() && accessToken != null) {
                this.logger.info(AccessTokenCacheImpl.refreshLog(this.cache, now, "Acquired a new access token"));
                this.cache = accessToken;
                sinksOne.tryEmitValue((Object)accessToken);
                this.nextTokenRefresh = OffsetDateTime.now().plus(REFRESH_DELAY);
                return Mono.just((Object)accessToken);
            }
            if (signal.isOnError() && error != null) {
                this.logger.error(AccessTokenCacheImpl.refreshLog(this.cache, now, "Failed to acquire a new access token"));
                this.nextTokenRefresh = OffsetDateTime.now().plus(REFRESH_DELAY);
                return fallback.switchIfEmpty(Mono.error((Throwable)error));
            }
            sinksOne.tryEmitEmpty();
            return fallback;
        };
    }

    private static String refreshLog(AccessToken cache, OffsetDateTime now, String log) {
        StringBuilder info = new StringBuilder(log);
        if (cache == null) {
            info.append(".");
        } else {
            Duration tte = Duration.between(now, cache.getExpiresAt());
            info.append(" at ").append(tte.abs().getSeconds()).append(" seconds ").append(tte.isNegative() ? "after" : "before").append(" expiry. ").append("Retry may be attempted after ").append(REFRESH_DELAY.getSeconds()).append(" seconds.");
            if (!tte.isNegative()) {
                info.append(" The token currently cached will be used.");
            }
        }
        return info.toString();
    }
}

