/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.test.http;

import com.azure.core.http.HttpClient;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.test.http.MockHttpResponse;
import com.azure.core.test.models.NetworkCallRecord;
import com.azure.core.test.models.RecordedData;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;

public final class PlaybackClient
implements HttpClient {
    private static final String X_MS_CLIENT_REQUEST_ID = "x-ms-client-request-id";
    private static final String X_MS_ENCRYPTION_KEY_SHA256 = "x-ms-encryption-key-sha256";
    private static final Pattern DOUBLE_SLASH_CLEANER = Pattern.compile("(?<!https?:)//");
    private static final Pattern ARRAYS_TO_STRING_SPLIT = Pattern.compile(", ");
    private final ClientLogger logger = new ClientLogger(PlaybackClient.class);
    private final AtomicInteger count = new AtomicInteger(0);
    private final Map<Pattern, String> textReplacementRules;
    private final RecordedData recordedData;

    public PlaybackClient(RecordedData recordedData, Map<String, String> textReplacementRules) {
        Objects.requireNonNull(recordedData, "'recordedData' cannot be null.");
        this.recordedData = recordedData;
        this.textReplacementRules = new HashMap<Pattern, String>();
        if (textReplacementRules != null) {
            for (Map.Entry<String, String> kvp : textReplacementRules.entrySet()) {
                this.textReplacementRules.put(Pattern.compile(kvp.getKey()), kvp.getValue());
            }
        }
    }

    public Mono<HttpResponse> send(HttpRequest request) {
        return Mono.fromCallable(() -> this.playbackHttpResponse(request));
    }

    private HttpResponse playbackHttpResponse(HttpRequest request) {
        String incomingUrl = this.applyReplacementRules(request.getUrl().toString());
        String incomingMethod = request.getHttpMethod().toString();
        String matchingUrl = PlaybackClient.removeHost(incomingUrl);
        NetworkCallRecord networkCallRecord = this.recordedData.findFirstAndRemoveNetworkCall(record -> {
            if (!record.getMethod().equalsIgnoreCase(incomingMethod)) {
                return false;
            }
            String removedHostUri = PlaybackClient.removeHost(record.getUri());
            String cleanedHostUri = DOUBLE_SLASH_CLEANER.matcher(removedHostUri).replaceAll("/");
            String cleanedMatchingUrl = DOUBLE_SLASH_CLEANER.matcher(matchingUrl).replaceAll("/");
            return cleanedHostUri.equalsIgnoreCase(cleanedMatchingUrl);
        });
        this.count.incrementAndGet();
        if (networkCallRecord == null) {
            this.logger.warning("NOT FOUND - Method: {} URL: {}", new Object[]{incomingMethod, incomingUrl});
            this.logger.warning("Records requested: {}.", new Object[]{this.count});
            throw this.logger.logExceptionAsError((RuntimeException)new IllegalStateException("==> Unexpected request: " + incomingMethod + " " + incomingUrl));
        }
        if (networkCallRecord.getException() != null) {
            throw this.logger.logExceptionAsWarning(Exceptions.propagate((Throwable)networkCallRecord.getException().get()));
        }
        if (networkCallRecord.getHeaders().containsKey(X_MS_CLIENT_REQUEST_ID)) {
            request.setHeader(X_MS_CLIENT_REQUEST_ID, networkCallRecord.getHeaders().get(X_MS_CLIENT_REQUEST_ID));
        }
        if (request.getHeaders().getValue(X_MS_ENCRYPTION_KEY_SHA256) != null) {
            networkCallRecord.getResponse().put(X_MS_ENCRYPTION_KEY_SHA256, request.getHeaders().getValue(X_MS_ENCRYPTION_KEY_SHA256));
        }
        int recordStatusCode = Integer.parseInt(networkCallRecord.getResponse().get("StatusCode"));
        HttpHeaders headers = new HttpHeaders();
        for (Map.Entry<String, String> pair : networkCallRecord.getResponse().entrySet()) {
            if (pair.getKey().equals("StatusCode") || pair.getKey().equals("Body")) continue;
            headers.set(pair.getKey(), this.applyReplacementRules(pair.getValue()));
        }
        String rawBody = networkCallRecord.getResponse().get("Body");
        byte[] bytes = null;
        if (rawBody != null) {
            rawBody = this.applyReplacementRules(rawBody);
            String contentType = networkCallRecord.getResponse().get("Content-Type");
            if (contentType != null && (contentType.equalsIgnoreCase("application/octet-stream") || "avro/binary".equalsIgnoreCase(contentType))) {
                if (rawBody.startsWith("[") && rawBody.endsWith("]")) {
                    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
                    for (String piece : ARRAYS_TO_STRING_SPLIT.split(rawBody.substring(1, rawBody.length() - 1))) {
                        outputStream.write(Byte.parseByte(piece));
                    }
                    bytes = outputStream.toByteArray();
                } else {
                    bytes = Base64.getDecoder().decode(rawBody);
                }
            } else {
                bytes = rawBody.getBytes(StandardCharsets.UTF_8);
            }
            if (bytes.length > 0) {
                headers.set("Content-Length", String.valueOf(bytes.length));
            }
        }
        return new MockHttpResponse(request, recordStatusCode, headers, bytes);
    }

    private String applyReplacementRules(String text) {
        for (Map.Entry<Pattern, String> rule : this.textReplacementRules.entrySet()) {
            if (rule.getValue() == null) continue;
            text = rule.getKey().matcher(text).replaceAll(rule.getValue());
        }
        return text;
    }

    private static String removeHost(String url) {
        UrlBuilder urlBuilder = UrlBuilder.parse((String)url);
        if (urlBuilder.getQuery().containsKey("sig")) {
            urlBuilder.setQueryParameter("sig", "REDACTED");
        }
        return String.format("%s%s", urlBuilder.getPath(), urlBuilder.getQueryString());
    }
}

