/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.implementation.util.FluxUtil;
import com.azure.core.util.logging.ClientLogger;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private final HttpLogDetailLevel detailLevel;
    private final boolean prettyPrintJSON;
    private static final int MAX_BODY_LOG_SIZE = 16384;

    public HttpLoggingPolicy(HttpLogDetailLevel detailLevel) {
        this(detailLevel, false);
    }

    public HttpLoggingPolicy(HttpLogDetailLevel detailLevel, boolean prettyPrintJSON) {
        this.detailLevel = detailLevel;
        this.prettyPrintJSON = prettyPrintJSON;
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        Optional<Object> data = context.getData("caller-method");
        String callerMethod = (String)data.orElse("");
        ClientLogger logger = new ClientLogger(callerMethod);
        long startNs = System.nanoTime();
        Mono<Void> logRequest = this.logRequest(logger, context.httpRequest());
        Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate = this.logResponseDelegate(logger, context.httpRequest().url(), startNs);
        return logRequest.then(next.process()).flatMap(logResponseDelegate).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private Mono<Void> logRequest(ClientLogger logger, HttpRequest request) {
        if (this.detailLevel.shouldLogURL()) {
            logger.info("--> {} {}", new Object[]{request.httpMethod(), request.url()});
        }
        if (this.detailLevel.shouldLogHeaders()) {
            for (HttpHeader header : request.headers()) {
                logger.info(header.toString(), new Object[0]);
            }
        }
        Mono reqBodyLoggingMono = Mono.empty();
        if (this.detailLevel.shouldLogBody()) {
            if (request.body() == null) {
                logger.info("(empty body)", new Object[0]);
                logger.info("--> END {}", new Object[]{request.httpMethod()});
            } else {
                boolean isHumanReadableContentType = !"application/octet-stream".equalsIgnoreCase(request.headers().value("Content-Type"));
                long contentLength = this.getContentLength(request.headers());
                if (contentLength < 16384L && isHumanReadableContentType) {
                    try {
                        Mono<byte[]> collectedBytes = FluxUtil.collectBytesInByteBufferStream(request.body());
                        reqBodyLoggingMono = collectedBytes.flatMap(bytes -> {
                            String bodyString = new String((byte[])bytes, StandardCharsets.UTF_8);
                            bodyString = this.prettyPrintIfNeeded(logger, request.headers().value("Content-Type"), bodyString);
                            logger.info("{}-byte body:%n{}", contentLength, bodyString);
                            logger.info("--> END {}", new Object[]{request.httpMethod()});
                            return Mono.empty();
                        });
                    }
                    catch (Exception e) {
                        reqBodyLoggingMono = Mono.error((Throwable)e);
                    }
                } else {
                    logger.info("{}-byte body: (content not logged)", contentLength);
                    logger.info("--> END {}", new Object[]{request.httpMethod()});
                }
            }
        }
        return reqBodyLoggingMono;
    }

    private Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate(ClientLogger logger, URL url, long startNs) {
        return response -> {
            long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
            String contentLengthString = response.headerValue("Content-Length");
            String bodySize = contentLengthString == null || contentLengthString.isEmpty() ? "unknown-length" : contentLengthString + "-byte";
            if (this.detailLevel.shouldLogURL()) {
                logger.info("<-- {} {} ({} ms, {} body)", response.statusCode(), url, tookMs, bodySize);
            }
            if (this.detailLevel.shouldLogHeaders()) {
                for (HttpHeader header : response.headers()) {
                    logger.info(header.toString(), new Object[0]);
                }
            }
            if (this.detailLevel.shouldLogBody()) {
                long contentLength = this.getContentLength(response.headers());
                String contentTypeHeader = response.headerValue("Content-Type");
                if (!"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L) {
                    HttpResponse bufferedResponse = response.buffer();
                    return bufferedResponse.bodyAsString().map(bodyStr -> {
                        bodyStr = this.prettyPrintIfNeeded(logger, contentTypeHeader, (String)bodyStr);
                        logger.info("Response body:\n{}", bodyStr);
                        logger.info("<-- END HTTP", new Object[0]);
                        return bufferedResponse;
                    }).switchIfEmpty(Mono.defer(() -> Mono.just((Object)bufferedResponse)));
                }
                logger.info("(body content not logged)", new Object[0]);
                logger.info("<-- END HTTP", new Object[0]);
            } else {
                logger.info("<-- END HTTP", new Object[0]);
            }
            return Mono.just((Object)response);
        };
    }

    private String prettyPrintIfNeeded(ClientLogger logger, String contentType, String body) {
        String result = body;
        if (this.prettyPrintJSON && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(HttpHeaders headers) {
        long contentLength = 0L;
        try {
            contentLength = Long.parseLong(headers.value("content-length"));
        }
        catch (NullPointerException | NumberFormatException runtimeException) {
            // empty catch block
        }
        return contentLength;
    }
}

