/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.implementation.util.FluxUtil;
import com.azure.core.util.logging.ClientLogger;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private final HttpLogOptions httpLogOptions;
    private final boolean prettyPrintJSON;
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        this(httpLogOptions, false);
    }

    HttpLoggingPolicy(HttpLogOptions httpLogOptions, boolean prettyPrintJSON) {
        this.httpLogOptions = httpLogOptions;
        this.prettyPrintJSON = prettyPrintJSON;
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        Optional<Object> data = context.getData("caller-method");
        String callerMethod = (String)data.orElse("");
        ClientLogger logger = new ClientLogger(callerMethod);
        long startNs = System.nanoTime();
        if (this.httpLogOptions != null) {
            Mono<Void> logRequest = this.logRequest(logger, context.getHttpRequest());
            Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate = this.logResponseDelegate(logger, context.getHttpRequest().getUrl(), startNs);
            return logRequest.then(next.process()).flatMap(logResponseDelegate).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
        }
        return Mono.empty();
    }

    private Mono<Void> logRequest(ClientLogger logger, HttpRequest request) {
        HttpLogDetailLevel httpLogLevel = this.httpLogOptions.getLogLevel();
        if (httpLogLevel.shouldLogUrl()) {
            logger.info("--> {} {}", new Object[]{request.getHttpMethod(), request.getUrl()});
            this.formatAllowableQueryParams(this.httpLogOptions.getAllowedQueryParamNames(), request.getUrl().getQuery(), logger);
        }
        if (httpLogLevel.shouldLogHeaders()) {
            this.formatAllowableHeaders(this.httpLogOptions.getAllowedHeaderNames(), request.getHeaders(), logger);
        }
        Mono reqBodyLoggingMono = Mono.empty();
        if (httpLogLevel.shouldLogBody()) {
            if (request.getBody() == null) {
                logger.info("(empty body)", new Object[0]);
                logger.info("--> END {}", new Object[]{request.getHttpMethod()});
            } else {
                boolean isHumanReadableContentType = !"application/octet-stream".equalsIgnoreCase(request.getHeaders().getValue("Content-Type"));
                long contentLength = this.getContentLength(request.getHeaders());
                if (contentLength < 16384L && isHumanReadableContentType) {
                    try {
                        Mono<byte[]> collectedBytes = FluxUtil.collectBytesInByteBufferStream(request.getBody());
                        reqBodyLoggingMono = collectedBytes.flatMap(bytes -> {
                            String bodyString = new String((byte[])bytes, StandardCharsets.UTF_8);
                            bodyString = this.prettyPrintIfNeeded(logger, request.getHeaders().getValue("Content-Type"), bodyString);
                            logger.info("{}-byte body:%n{}", contentLength, bodyString);
                            logger.info("--> END {}", new Object[]{request.getHttpMethod()});
                            return Mono.empty();
                        });
                    }
                    catch (Exception e) {
                        reqBodyLoggingMono = Mono.error((Throwable)e);
                    }
                } else {
                    logger.info("{}-byte body: (content not logged)", contentLength);
                    logger.info("--> END {}", new Object[]{request.getHttpMethod()});
                }
            }
        }
        return reqBodyLoggingMono;
    }

    private void formatAllowableHeaders(Set<String> allowedHeaderNames, HttpHeaders requestResponseHeaders, ClientLogger logger) {
        if (allowedHeaderNames != null && !allowedHeaderNames.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            for (HttpHeader header : requestResponseHeaders) {
                sb.append(header.getName()).append(":");
                if (allowedHeaderNames.contains(header.getName())) {
                    sb.append(header.getValue());
                } else {
                    sb.append(REDACTED_PLACEHOLDER);
                }
                sb.append(System.getProperty("line.separator"));
            }
            logger.info(sb.toString(), new Object[0]);
        }
    }

    private void formatAllowableQueryParams(Set<String> allowedQueryParamNames, String queryString, ClientLogger logger) {
        if (allowedQueryParamNames != null && !allowedQueryParamNames.isEmpty() && queryString != null) {
            String[] queryParams;
            StringBuilder sb = new StringBuilder();
            for (String queryParam : queryParams = queryString.split("&")) {
                String[] queryPair = queryParam.split("=", 2);
                if (queryPair.length == 2) {
                    if (allowedQueryParamNames.contains(queryPair[0])) {
                        sb.append(queryParam);
                    } else {
                        sb.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                    }
                } else {
                    sb.append(queryParam);
                }
                sb.append("&");
            }
            if (sb.length() > 0) {
                logger.info(sb.substring(0, sb.length() - 1), new Object[0]);
            }
        }
    }

    private Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate(ClientLogger logger, URL url, long startNs) {
        return response -> {
            long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
            String contentLengthString = response.getHeaderValue("Content-Length");
            String bodySize = contentLengthString == null || contentLengthString.isEmpty() ? "unknown-length" : contentLengthString + "-byte";
            HttpLogDetailLevel httpLogLevel = this.httpLogOptions.getLogLevel();
            if (httpLogLevel.shouldLogUrl()) {
                logger.info("<-- {} {} ({} ms, {} body)", response.getStatusCode(), url, tookMs, bodySize);
            }
            if (httpLogLevel.shouldLogHeaders()) {
                this.formatAllowableHeaders(this.httpLogOptions.getAllowedHeaderNames(), response.getHeaders(), logger);
            }
            if (httpLogLevel.shouldLogBody()) {
                long contentLength = this.getContentLength(response.getHeaders());
                String contentTypeHeader = response.getHeaderValue("Content-Type");
                if (!"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L) {
                    HttpResponse bufferedResponse = response.buffer();
                    return bufferedResponse.getBodyAsString().map(bodyStr -> {
                        bodyStr = this.prettyPrintIfNeeded(logger, contentTypeHeader, (String)bodyStr);
                        logger.info("Response body:\n{}", bodyStr);
                        logger.info("<-- END HTTP", new Object[0]);
                        return bufferedResponse;
                    }).switchIfEmpty(Mono.defer(() -> Mono.just((Object)bufferedResponse)));
                }
                logger.info("(body content not logged)", new Object[0]);
                logger.info("<-- END HTTP", new Object[0]);
            } else {
                logger.info("<-- END HTTP", new Object[0]);
            }
            return Mono.just((Object)response);
        };
    }

    private String prettyPrintIfNeeded(ClientLogger logger, String contentType, String body) {
        String result = body;
        if (this.prettyPrintJSON && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(HttpHeaders headers) {
        long contentLength = 0L;
        try {
            contentLength = Long.parseLong(headers.getValue("content-length"));
        }
        catch (NullPointerException | NumberFormatException runtimeException) {
            // empty catch block
        }
        return contentLength;
    }
}

