/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private final HttpLogOptions httpLogOptions;
    private final boolean prettyPrintJSON;
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        this(httpLogOptions, false);
    }

    HttpLoggingPolicy(HttpLogOptions httpLogOptions, boolean prettyPrintJSON) {
        this.httpLogOptions = httpLogOptions;
        this.prettyPrintJSON = prettyPrintJSON;
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogOptions == null || this.httpLogOptions.getLogLevel() == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        Optional<Object> data = context.getData("caller-method");
        String callerMethod = (String)data.orElse("");
        ClientLogger logger = new ClientLogger(callerMethod);
        long startNs = System.nanoTime();
        Mono<Void> logRequest = this.logRequest(logger, context.getHttpRequest());
        Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate = this.logResponseDelegate(logger, context.getHttpRequest().getUrl(), startNs);
        return logRequest.then(next.process()).flatMap(logResponseDelegate).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private Mono<Void> logRequest(ClientLogger logger, HttpRequest request) {
        HttpLogDetailLevel httpLogLevel = this.httpLogOptions.getLogLevel();
        if (httpLogLevel.shouldLogUrl()) {
            try {
                UrlBuilder requestUrl = UrlBuilder.parse(request.getUrl());
                requestUrl.setQuery(this.getAllowedQueryString(request.getUrl().getQuery()));
                logger.info("--> {} {}", new Object[]{request.getHttpMethod(), requestUrl.toUrl()});
            }
            catch (MalformedURLException ex) {
                return Mono.error((Throwable)logger.logExceptionAsWarning(new IllegalStateException("Invalid request URL.")));
            }
        }
        if (httpLogLevel.shouldLogHeaders()) {
            this.formatAllowableHeaders(this.httpLogOptions.getAllowedHeaderNames(), request.getHeaders(), logger);
        }
        Mono reqBodyLoggingMono = Mono.empty();
        if (httpLogLevel.shouldLogBody()) {
            if (request.getBody() == null) {
                logger.info("(empty body)", new Object[0]);
                logger.info("--> END {}", new Object[]{request.getHttpMethod()});
            } else {
                boolean isHumanReadableContentType = !"application/octet-stream".equalsIgnoreCase(request.getHeaders().getValue("Content-Type"));
                long contentLength = this.getContentLength(logger, request.getHeaders());
                if (contentLength < 16384L && isHumanReadableContentType) {
                    try {
                        Mono<byte[]> collectedBytes = FluxUtil.collectBytesInByteBufferStream(request.getBody());
                        reqBodyLoggingMono = collectedBytes.flatMap(bytes -> {
                            String bodyString = new String((byte[])bytes, StandardCharsets.UTF_8);
                            bodyString = this.prettyPrintIfNeeded(logger, request.getHeaders().getValue("Content-Type"), bodyString);
                            logger.info("{}-byte body:%n{}", contentLength, bodyString);
                            logger.info("--> END {}", new Object[]{request.getHttpMethod()});
                            return Mono.empty();
                        });
                    }
                    catch (Exception e) {
                        reqBodyLoggingMono = Mono.error((Throwable)e);
                    }
                } else {
                    logger.info("{}-byte body: (content not logged)", contentLength);
                    logger.info("--> END {}", new Object[]{request.getHttpMethod()});
                }
            }
        }
        return reqBodyLoggingMono;
    }

    private void formatAllowableHeaders(Set<String> allowedHeaderNames, HttpHeaders requestResponseHeaders, ClientLogger logger) {
        Set lowerCasedAllowedHeaderNames = allowedHeaderNames.stream().map(String::toLowerCase).collect(Collectors.toSet());
        StringBuilder sb = new StringBuilder();
        for (HttpHeader header : requestResponseHeaders) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (lowerCasedAllowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.getProperty("line.separator"));
        }
        logger.info(sb.toString(), new Object[0]);
    }

    private String getAllowedQueryString(String queryString) {
        String[] queryParams;
        Set lowerCasedAllowedQueryParams = this.httpLogOptions.getAllowedQueryParamNames().stream().map(String::toLowerCase).collect(Collectors.toSet());
        if (CoreUtils.isNullOrEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (lowerCasedAllowedQueryParams.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private Function<HttpResponse, Mono<HttpResponse>> logResponseDelegate(ClientLogger logger, URL url, long startNs) {
        return response -> {
            long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
            String contentLengthString = response.getHeaderValue("Content-Length");
            String bodySize = contentLengthString == null || contentLengthString.isEmpty() ? "unknown-length" : contentLengthString + "-byte";
            HttpLogDetailLevel httpLogLevel = this.httpLogOptions.getLogLevel();
            if (httpLogLevel.shouldLogUrl()) {
                logger.info("<-- {} {} ({} ms, {} body)", response.getStatusCode(), url, tookMs, bodySize);
            }
            if (httpLogLevel.shouldLogHeaders()) {
                this.formatAllowableHeaders(this.httpLogOptions.getAllowedHeaderNames(), response.getHeaders(), logger);
            }
            if (httpLogLevel.shouldLogBody()) {
                long contentLength = this.getContentLength(logger, response.getHeaders());
                String contentTypeHeader = response.getHeaderValue("Content-Type");
                if (!"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L) {
                    HttpResponse bufferedResponse = response.buffer();
                    return bufferedResponse.getBodyAsString().map(bodyStr -> {
                        bodyStr = this.prettyPrintIfNeeded(logger, contentTypeHeader, (String)bodyStr);
                        logger.info("Response body:\n{}", bodyStr);
                        logger.info("<-- END HTTP", new Object[0]);
                        return bufferedResponse;
                    }).switchIfEmpty(Mono.defer(() -> Mono.just((Object)bufferedResponse)));
                }
                logger.info("(body content not logged)", new Object[0]);
                logger.info("<-- END HTTP", new Object[0]);
            } else {
                logger.info("<-- END HTTP", new Object[0]);
            }
            return Mono.just((Object)response);
        };
    }

    private String prettyPrintIfNeeded(ClientLogger logger, String contentType, String body) {
        String result = body;
        if (this.prettyPrintJSON && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        try {
            contentLength = Long.parseLong(headers.getValue("content-length"));
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }
}

