// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.core.util;

import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.serializer.JsonSerializer;
import com.azure.core.util.serializer.JsonSerializerProviders;
import com.azure.core.util.serializer.ObjectSerializer;
import com.azure.core.util.serializer.TypeReference;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;

/**
 * This class is an abstraction for the many ways binary data can be represented.
 * <p>
 * {@link BinaryData} can be created from an {@link InputStream}, a {@link Flux} of {@link ByteBuffer}, a {@link
 * String}, an {@link Object}, or a byte array.
 *
 * <p><strong>Immutable data</strong></p>
 *
 * {@link BinaryData} copies data on construction making it immutable. Various APIs are provided to get data out of
 * {@link BinaryData}, they all start with the {@code 'to'} prefix, for example {@link BinaryData#toBytes()}.
 *
 * <p><strong>Create an instance from a byte array</strong></p>
 *
 * <pre>
 * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
 * BinaryData binaryData = BinaryData.fromBytes&#40;data&#41;;
 * System.out.println&#40;new String&#40;binaryData.toBytes&#40;&#41;, StandardCharsets.UTF_8&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from a String</strong></p>
 *
 * <pre>
 * final String data = &quot;Some Data&quot;;
 * &#47;&#47; Following will use default character set as StandardCharsets.UTF_8
 * BinaryData binaryData = BinaryData.fromString&#40;data&#41;;
 * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from an InputStream</strong></p>
 *
 * <pre>
 * final ByteArrayInputStream inputStream = new ByteArrayInputStream&#40;&quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;&#41;;
 * BinaryData binaryData = BinaryData.fromStream&#40;inputStream&#41;;
 * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from an Object</strong></p>
 *
 * <pre>
 * class Person &#123;
 *     &#123;@literal @&#125;JsonProperty
 *     private String name;
 * 
 *     &#123;@literal @&#125;JsonSetter
 *     public Person setName&#40;String name&#41; &#123;
 *         this.name = name;
 *         return this;
 *     &#125;
 * 
 *     &#123;@literal @&#125;JsonGetter
 *     public String getName&#40;&#41; &#123;
 *         return name;
 *     &#125;
 * &#125;
 * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
 * 
 * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
 * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
 * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
 * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
 * 
 * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
 * </pre>
 *
 * @see ObjectSerializer
 * @see JsonSerializer
 * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
 */
public final class BinaryData {
    private static final ClientLogger LOGGER = new ClientLogger(BinaryData.class);
    private static final BinaryData EMPTY_DATA = new BinaryData(new byte[0]);
    private static final int STREAM_READ_SIZE = 1024;

    private static final Object LOCK = new Object();

    private static volatile JsonSerializer defaultJsonSerializer;

    private final byte[] data;

    private String dataAsStringCache;

    /**
     * Create an instance of {@link BinaryData} from the given byte array.
     *
     * @param data The byte array that {@link BinaryData} will represent.
     */
    BinaryData(byte[] data) {
        this.data = data;
    }

    /**
     * Creates an instance of {@link BinaryData} from the given {@link InputStream}.
     * <p>
     * If {@code inputStream} is null or empty an empty {@link BinaryData} is returned.
     * <p>
     * <b>NOTE:</b> The {@link InputStream} is not closed by this function.
     *
     * <p><strong>Create an instance from an InputStream</strong></p>
     *
     * <pre>
     * final ByteArrayInputStream inputStream = new ByteArrayInputStream&#40;&quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;&#41;;
     * BinaryData binaryData = BinaryData.fromStream&#40;inputStream&#41;;
     * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     * </pre>
     *
     * @param inputStream The {@link InputStream} that {@link BinaryData} will represent.
     * @return A {@link BinaryData} representing the {@link InputStream}.
     * @throws UncheckedIOException If any error happens while reading the {@link InputStream}.
     */
    public static BinaryData fromStream(InputStream inputStream) {
        if (Objects.isNull(inputStream)) {
            return EMPTY_DATA;
        }

        try {
            ByteArrayOutputStream dataOutputBuffer = new ByteArrayOutputStream();
            int nRead;
            byte[] data = new byte[STREAM_READ_SIZE];
            while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
                dataOutputBuffer.write(data, 0, nRead);
            }

            return new BinaryData(dataOutputBuffer.toByteArray());

        } catch (IOException ex) {
            throw LOGGER.logExceptionAsError(new UncheckedIOException(ex));
        }
    }

    /**
     * Creates an instance of {@link BinaryData} from the given {@link InputStream}.
     * <p>
     * If {@code inputStream} is null or empty an empty {@link BinaryData} is returned.
     * <p>
     * <b>NOTE:</b> The {@link InputStream} is not closed by this function.
     *
     * <p><strong>Create an instance from an InputStream</strong></p>
     *
     * <pre>
     * final ByteArrayInputStream inputStream = new ByteArrayInputStream&#40;&quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;&#41;;
     * 
     * Mono&lt;BinaryData&gt; binaryDataMono = BinaryData.fromStreamAsync&#40;inputStream&#41;;
     * 
     * Disposable subscriber = binaryDataMono
     *     .map&#40;binaryData -&gt; &#123;
     *         System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     *         return true;
     *     &#125;&#41;
     *     .subscribe&#40;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param inputStream The {@link InputStream} that {@link BinaryData} will represent.
     * @return A {@link Mono} of {@link BinaryData} representing the {@link InputStream}.
     * @throws UncheckedIOException If any error happens while reading the {@link InputStream}.
     */
    public static Mono<BinaryData> fromStreamAsync(InputStream inputStream) {
        return Mono.fromCallable(() -> fromStream(inputStream));
    }

    /**
     * Creates an instance of {@link BinaryData} from the given {@link Flux} of {@link ByteBuffer}.
     * <p>
     * If the {@code data} is null an empty {@link BinaryData} will be returned.
     * <p>
     * <b>Note:</b> This will collect all bytes from the {@link ByteBuffer ByteBuffers} resulting in {@link
     * ByteBuffer#hasRemaining() hasRemaining} to return false.
     *
     * <p><strong>Create an instance from a Flux of ByteBuffer</strong></p>
     *
     * <pre>
     * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
     * final Flux&lt;ByteBuffer&gt; dataFlux = Flux.just&#40;ByteBuffer.wrap&#40;data&#41;&#41;;
     * 
     * Mono&lt;BinaryData&gt; binaryDataMono = BinaryData.fromFlux&#40;dataFlux&#41;;
     * 
     * Disposable subscriber = binaryDataMono
     *     .map&#40;binaryData -&gt; &#123;
     *         System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     *         return true;
     *     &#125;&#41;
     *     .subscribe&#40;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param data The {@link Flux} of {@link ByteBuffer} that {@link BinaryData} will represent.
     * @return A {@link Mono} of {@link BinaryData} representing the {@link Flux} of {@link ByteBuffer}.
     */
    public static Mono<BinaryData> fromFlux(Flux<ByteBuffer> data) {
        if (Objects.isNull(data)) {
            return Mono.just(EMPTY_DATA);
        }

        return FluxUtil.collectBytesInByteBufferStream(data)
            .flatMap(bytes -> Mono.just(new BinaryData(bytes)));
    }

    /**
     * Creates an instance of {@link BinaryData} from the given {@link String}.
     * <p>
     * The {@link String} is converted into bytes using {@link String#getBytes(Charset)} passing {@link
     * StandardCharsets#UTF_8}.
     * <p>
     * If the {@code data} is null or a zero length string an empty {@link BinaryData} will be returned.
     *
     * <p><strong>Create an instance from a String</strong></p>
     *
     * <pre>
     * final String data = &quot;Some Data&quot;;
     * &#47;&#47; Following will use default character set as StandardCharsets.UTF_8
     * BinaryData binaryData = BinaryData.fromString&#40;data&#41;;
     * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     * </pre>
     *
     * @param data The {@link String} that {@link BinaryData} will represent.
     * @return A {@link BinaryData} representing the {@link String}.
     */
    public static BinaryData fromString(String data) {
        if (CoreUtils.isNullOrEmpty(data)) {
            return EMPTY_DATA;
        }

        return new BinaryData(data.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Creates an instance of {@link BinaryData} from the given byte array.
     * <p>
     * If the byte array is null or zero length an empty {@link BinaryData} will be returned.
     *
     * <p><strong>Create an instance from a byte array</strong></p>
     *
     * <pre>
     * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
     * BinaryData binaryData = BinaryData.fromBytes&#40;data&#41;;
     * System.out.println&#40;new String&#40;binaryData.toBytes&#40;&#41;, StandardCharsets.UTF_8&#41;&#41;;
     * </pre>
     *
     * @param data The byte array that {@link BinaryData} will represent.
     * @return A {@link BinaryData} representing the byte array.
     */
    public static BinaryData fromBytes(byte[] data) {
        if (Objects.isNull(data) || data.length == 0) {
            return EMPTY_DATA;
        }

        return new BinaryData(Arrays.copyOf(data, data.length));
    }

    /**
     * Creates an instance of {@link BinaryData} by serializing the {@link Object} using the default {@link
     * JsonSerializer}.
     * <p>
     * If {@code data} is null an empty {@link BinaryData} will be returned.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Creating an instance from an Object</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     * </pre>
     *
     * @param data The object that will be JSON serialized that {@link BinaryData} will represent.
     * @return A {@link BinaryData} representing the JSON serialized object.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public static BinaryData fromObject(Object data) {
        return fromObject(data, getDefaultSerializer());
    }

    /**
     * Creates an instance of {@link BinaryData} by serializing the {@link Object} using the default {@link
     * JsonSerializer}.
     * <p>
     * If {@code data} is null an empty {@link BinaryData} will be returned.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Creating an instance from an Object</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * Disposable subscriber = BinaryData.fromObjectAsync&#40;data&#41;
     *     .subscribe&#40;binaryData -&gt; System.out.println&#40;binaryData.toString&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param data The object that will be JSON serialized that {@link BinaryData} will represent.
     * @return A {@link Mono} of {@link BinaryData} representing the JSON serialized object.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public static Mono<BinaryData> fromObjectAsync(Object data) {
        return fromObjectAsync(data, getDefaultSerializer());
    }

    /**
     * Creates an instance of {@link BinaryData} by serializing the {@link Object} using the passed {@link
     * ObjectSerializer}.
     * <p>
     * If {@code data} is null an empty {@link BinaryData} will be returned.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Create an instance from an Object</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     * </pre>
     *
     * @param data The object that will be serialized that {@link BinaryData} will represent.
     * @param serializer The {@link ObjectSerializer} used to serialize object.
     * @return A {@link BinaryData} representing the serialized object.
     * @throws NullPointerException If {@code serializer} is null and {@code data} is not null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public static BinaryData fromObject(Object data, ObjectSerializer serializer) {
        if (Objects.isNull(data)) {
            return EMPTY_DATA;
        }

        Objects.requireNonNull(serializer, "'serializer' cannot be null.");

        return new BinaryData(serializer.serializeToBytes(data));
    }

    /**
     * Creates an instance of {@link BinaryData} by serializing the {@link Object} using the passed {@link
     * ObjectSerializer}.
     * <p>
     * If {@code data} is null an empty {@link BinaryData} will be returned.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Create an instance from an Object</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * Disposable subscriber = BinaryData.fromObjectAsync&#40;data, serializer&#41;
     *     .subscribe&#40;binaryData -&gt; System.out.println&#40;binaryData.toString&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param data The object that will be serialized that {@link BinaryData} will represent.
     * @param serializer The {@link ObjectSerializer} used to serialize object.
     * @return A {@link Mono} of {@link BinaryData} representing the serialized object.
     * @throws NullPointerException If {@code serializer} is null and {@code data} is not null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public static Mono<BinaryData> fromObjectAsync(Object data, ObjectSerializer serializer) {
        return Mono.fromCallable(() -> fromObject(data, serializer));
    }

    /**
     * Returns a byte array representation of this {@link BinaryData}.
     *
     * @return A byte array representing this {@link BinaryData}.
     */
    public byte[] toBytes() {
        return Arrays.copyOf(this.data, this.data.length);
    }

    /**
     * Returns a {@link String} representation of this {@link BinaryData} by converting its data using the UTF-8
     * character set.
     *
     * @return A {@link String} representing this {@link BinaryData}.
     */
    public String toString() {
        if (this.dataAsStringCache == null) {
            this.dataAsStringCache = new String(this.data, StandardCharsets.UTF_8);
        }

        return this.dataAsStringCache;
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the default
     * {@link JsonSerializer}.
     * <p>
     * The type, represented by {@link Class}, should be a non-generic class, for generic classes use {@link
     * #toObject(TypeReference)}.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * Person person = binaryData.toObject&#40;Person.class&#41;;
     * System.out.println&#40;person.getName&#40;&#41;&#41;;
     * </pre>
     *
     * @param clazz The {@link Class} representing the Object's type.
     * @param <T> Type of the deserialized Object.
     * @return An {@link Object} representing the JSON deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code clazz} is null.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public <T> T toObject(Class<T> clazz) {
        return toObject(TypeReference.createInstance(clazz), getDefaultSerializer());
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the default
     * {@link JsonSerializer}.
     * <p>
     * The type, represented by {@link TypeReference}, can either be a generic or non-generic type. If the type is
     * generic create a sub-type of {@link TypeReference}, if the type is non-generic use {@link
     * TypeReference#createInstance(Class)}.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * Person person = binaryData.toObject&#40;TypeReference.createInstance&#40;Person.class&#41;&#41;;
     * System.out.println&#40;person.getName&#40;&#41;&#41;;
     * </pre>
     *
     * <p><strong>Get a generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;personList&#41;;
     * 
     * List&lt;Person&gt; persons = binaryData.toObject&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;&#41;;
     * persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * </pre>
     *
     * @param typeReference The {@link TypeReference} representing the Object's type.
     * @param <T> Type of the deserialized Object.
     * @return An {@link Object} representing the JSON deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code typeReference} is null.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public <T> T toObject(TypeReference<T> typeReference) {
        return toObject(typeReference, getDefaultSerializer());
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the passed
     * {@link ObjectSerializer}.
     * <p>
     * The type, represented by {@link Class}, should be a non-generic class, for generic classes use {@link
     * #toObject(TypeReference, ObjectSerializer)}.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Person person = binaryData.toObject&#40;Person.class, serializer&#41;;
     * System.out.println&#40;&quot;Name : &quot; + person.getName&#40;&#41;&#41;;
     * </pre>
     *
     * @param clazz The {@link Class} representing the Object's type.
     * @param serializer The {@link ObjectSerializer} used to deserialize object.
     * @param <T> Type of the deserialized Object.
     * @return An {@link Object} representing the deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code clazz} or {@code serializer} is null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public <T> T toObject(Class<T> clazz, ObjectSerializer serializer) {
        return toObject(TypeReference.createInstance(clazz), serializer);
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the passed
     * {@link ObjectSerializer}.
     * <p>
     * The type, represented by {@link TypeReference}, can either be a generic or non-generic type. If the type is
     * generic create a sub-type of {@link TypeReference}, if the type is non-generic use {@link
     * TypeReference#createInstance(Class)}.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Person person = binaryData.toObject&#40;TypeReference.createInstance&#40;Person.class&#41;, serializer&#41;;
     * System.out.println&#40;&quot;Name : &quot; + person.getName&#40;&#41;&#41;;
     * 
     * </pre>
     *
     * <p><strong>Get a generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;personList, serializer&#41;;
     * 
     * &#47;&#47; Retains the type of the list when deserializing
     * List&lt;Person&gt; persons = binaryData.toObject&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;, serializer&#41;;
     * persons.forEach&#40;person -&gt; System.out.println&#40;&quot;Name : &quot; + person.getName&#40;&#41;&#41;&#41;;
     * </pre>
     *
     * @param typeReference The {@link TypeReference} representing the Object's type.
     * @param serializer The {@link ObjectSerializer} used to deserialize object.
     * @param <T> Type of the deserialized Object.
     * @return An {@link Object} representing the deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code typeReference} or {@code serializer} is null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public <T> T toObject(TypeReference<T> typeReference, ObjectSerializer serializer) {
        Objects.requireNonNull(typeReference, "'typeReference' cannot be null.");
        Objects.requireNonNull(serializer, "'serializer' cannot be null.");

        return serializer.deserializeFromBytes(this.data, typeReference);
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the default
     * {@link JsonSerializer}.
     * <p>
     * The type, represented by {@link Class}, should be a non-generic class, for generic classes use {@link
     * #toObject(TypeReference)}.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * Disposable subscriber = binaryData.toObjectAsync&#40;Person.class&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param clazz The {@link Class} representing the Object's type.
     * @param <T> Type of the deserialized Object.
     * @return A {@link Mono} of {@link Object} representing the JSON deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code clazz} is null.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public <T> Mono<T> toObjectAsync(Class<T> clazz) {
        return toObjectAsync(TypeReference.createInstance(clazz), getDefaultSerializer());
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the default
     * {@link JsonSerializer}.
     * <p>
     * The type, represented by {@link TypeReference}, can either be a generic or non-generic type. If the type is
     * generic create a sub-type of {@link TypeReference}, if the type is non-generic use {@link
     * TypeReference#createInstance(Class)}.
     * <p>
     * <b>Note:</b> A {@link JsonSerializer} implementation must be available on the classpath.
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * Disposable subscriber = binaryData.toObjectAsync&#40;TypeReference.createInstance&#40;Person.class&#41;&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * <p><strong>Get a generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;personList&#41;;
     * 
     * Disposable subscriber = binaryData.toObjectAsync&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;&#41;
     *     .subscribe&#40;persons -&gt; persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param typeReference The {@link TypeReference} representing the Object's type.
     * @param <T> Type of the deserialized Object.
     * @return A {@link Mono} of {@link Object} representing the JSON deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code typeReference} is null.
     * @throws IllegalStateException If a {@link JsonSerializer} implementation cannot be found on the classpath.
     * @see JsonSerializer
     */
    public <T> Mono<T> toObjectAsync(TypeReference<T> typeReference) {
        return toObjectAsync(typeReference, getDefaultSerializer());
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the passed
     * {@link ObjectSerializer}.
     * <p>
     * The type, represented by {@link Class}, should be a non-generic class, for generic classes use {@link
     * #toObject(TypeReference, ObjectSerializer)}.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Disposable subscriber = binaryData.toObjectAsync&#40;Person.class, serializer&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param clazz The {@link Class} representing the Object's type.
     * @param serializer The {@link ObjectSerializer} used to deserialize object.
     * @param <T> Type of the deserialized Object.
     * @return A {@link Mono} of {@link Object} representing the deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code clazz} or {@code serializer} is null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public <T> Mono<T> toObjectAsync(Class<T> clazz, ObjectSerializer serializer) {
        return toObjectAsync(TypeReference.createInstance(clazz), serializer);
    }

    /**
     * Returns an {@link Object} representation of this {@link BinaryData} by deserializing its data using the passed
     * {@link ObjectSerializer}.
     * <p>
     * The type, represented by {@link TypeReference}, can either be a generic or non-generic type. If the type is
     * generic create a sub-type of {@link TypeReference}, if the type is non-generic use {@link
     * TypeReference#createInstance(Class)}.
     * <p>
     * The passed {@link ObjectSerializer} can either be one of the implementations offered by the Azure SDKs or your
     * own implementation.
     *
     * <p><strong>Azure SDK implementations</strong></p>
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson JSON serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">GSON JSON serializer</a></li>
     * </ul>
     *
     * <p><strong>Get a non-generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Disposable subscriber = binaryData
     *     .toObjectAsync&#40;TypeReference.createInstance&#40;Person.class&#41;, serializer&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * <p><strong>Get a generic Object from the BinaryData</strong></p>
     *
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;personList, serializer&#41;;
     * 
     * Disposable subscriber = binaryData
     *     .toObjectAsync&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;, serializer&#41; &#47;&#47; retains the generic type information
     *     .subscribe&#40;persons -&gt; persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param typeReference The {@link TypeReference} representing the Object's type.
     * @param serializer The {@link ObjectSerializer} used to deserialize object.
     * @param <T> Type of the deserialized Object.
     * @return A {@link Mono} of {@link Object} representing the deserialized {@link BinaryData}.
     * @throws NullPointerException If {@code typeReference} or {@code serializer} is null.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public <T> Mono<T> toObjectAsync(TypeReference<T> typeReference, ObjectSerializer serializer) {
        return Mono.fromCallable(() -> toObject(typeReference, serializer));
    }

    /**
     * Returns an {@link InputStream} representation of this {@link BinaryData}.
     *
     * <p><strong>Get an InputStream from the BinaryData</strong></p>
     *
     * <pre>
     * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
     * BinaryData binaryData = BinaryData.fromStream&#40;new ByteArrayInputStream&#40;data&#41;&#41;;
     * final byte[] bytes = new byte[data.length];
     * binaryData.toStream&#40;&#41;.read&#40;bytes, 0, data.length&#41;;
     * System.out.println&#40;new String&#40;bytes&#41;&#41;;
     * </pre>
     *
     * @return An {@link InputStream} representing the {@link BinaryData}.
     */
    public InputStream toStream() {
        return new ByteArrayInputStream(this.data);
    }

    /* This will ensure lazy instantiation to avoid hard dependency on Json Serializer. */
    private static JsonSerializer getDefaultSerializer() {
        if (defaultJsonSerializer == null) {
            synchronized (LOCK) {
                if (defaultJsonSerializer == null) {
                    defaultJsonSerializer = JsonSerializerProviders.createInstance();
                }
            }
        }
        return defaultJsonSerializer;
    }
}
