/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpClient;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpMethod;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.HttpPipelineBuilder;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.MockHttpResponse;
import com.azure.core.http.clients.NoOpHttpClient;
import com.azure.core.http.policy.ExponentialBackoff;
import com.azure.core.http.policy.FixedDelay;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.RetryPolicy;
import com.azure.core.http.policy.RetryStrategy;
import com.azure.core.util.DateTimeRfc1123;
import com.azure.core.util.FluxUtil;
import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.test.StepVerifier;

public class RetryPolicyTests {
    @ParameterizedTest
    @ValueSource(ints={408, 429, 500, 502, 503})
    public void defaultRetryPolicyRetriesExpectedErrorCodes(int returnCode) {
        AtomicInteger attemptCount = new AtomicInteger();
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy()}).httpClient(request -> {
            int count = attemptCount.getAndIncrement();
            if (count == 0) {
                return Mono.just((Object)((Object)new MockHttpResponse(request, returnCode)));
            }
            if (count == 1) {
                return Mono.just((Object)((Object)new MockHttpResponse(request, 200)));
            }
            return Mono.just((Object)((Object)new MockHttpResponse(request, 400)));
        }).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)200, (int)response.getStatusCode())).verifyComplete();
    }

    @ParameterizedTest
    @ValueSource(ints={400, 401, 402, 403, 404, 409, 412, 501, 505})
    public void defaultRetryPolicyDoesntRetryOnErrorCodes(int returnCode) {
        AtomicInteger attemptCount = new AtomicInteger();
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy()}).httpClient(request -> {
            int count = attemptCount.getAndIncrement();
            if (count == 0) {
                return Mono.just((Object)((Object)new MockHttpResponse(request, returnCode)));
            }
            return Mono.just((Object)((Object)new MockHttpResponse(request, 200)));
        }).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)returnCode, (int)response.getStatusCode())).verifyComplete();
    }

    @ParameterizedTest
    @MethodSource(value={"defaultRetryPolicyRetriesAllExceptionsSupplier"})
    public void defaultRetryPolicyRetriesAllExceptions(Throwable throwable) {
        AtomicInteger attemptCount = new AtomicInteger();
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy()}).httpClient(request -> {
            int count = attemptCount.getAndIncrement();
            if (count == 0) {
                return Mono.error((Throwable)throwable);
            }
            return Mono.just((Object)((Object)new MockHttpResponse(request, 200)));
        }).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)200, (int)response.getStatusCode())).verifyComplete();
    }

    private static Stream<Throwable> defaultRetryPolicyRetriesAllExceptionsSupplier() {
        return Stream.of(new Throwable(), new MalformedURLException(), new RuntimeException(), new IllegalStateException(), new TimeoutException());
    }

    @ParameterizedTest
    @MethodSource(value={"customRetryPolicyCanDetermineRetryStatusCodesSupplier"})
    public void customRetryPolicyCanDetermineRetryStatusCodes(RetryStrategy retryStrategy, int[] statusCodes, int expectedStatusCode) {
        AtomicInteger attempt = new AtomicInteger();
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy(retryStrategy)}).httpClient(request -> Mono.just((Object)((Object)new MockHttpResponse(request, statusCodes[attempt.getAndIncrement()])))).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)expectedStatusCode, (int)response.getStatusCode())).verifyComplete();
    }

    private static Stream<Arguments> customRetryPolicyCanDetermineRetryStatusCodesSupplier() {
        RetryStrategy onlyRetries429And503 = RetryPolicyTests.createStatusCodeRetryStrategy(429, 503);
        RetryStrategy onlyRetries409And412 = RetryPolicyTests.createStatusCodeRetryStrategy(409, 412);
        return Stream.of(Arguments.of((Object[])new Object[]{onlyRetries429And503, new int[]{429, 503, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries429And503, new int[]{429, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries429And503, new int[]{503, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries429And503, new int[]{429, 503, 503}, 503}), Arguments.of((Object[])new Object[]{onlyRetries429And503, new int[]{429, 503, 429}, 429}), Arguments.of((Object[])new Object[]{onlyRetries409And412, new int[]{409, 412, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries409And412, new int[]{409, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries409And412, new int[]{412, 404}, 404}), Arguments.of((Object[])new Object[]{onlyRetries409And412, new int[]{409, 412, 409}, 409}), Arguments.of((Object[])new Object[]{onlyRetries409And412, new int[]{409, 412, 412}, 412}));
    }

    @ParameterizedTest
    @MethodSource(value={"customRetryPolicyCanDetermineRetryExceptionsSupplier"})
    public void customRetryPolicyCanDetermineRetryExceptions(RetryStrategy retryStrategy, Throwable[] exceptions, Class<? extends Throwable> expectedException) {
        AtomicInteger attempt = new AtomicInteger();
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy(retryStrategy)}).httpClient(request -> Mono.error((Throwable)exceptions[attempt.getAndIncrement()])).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).verifyError(expectedException);
    }

    private static Stream<Arguments> customRetryPolicyCanDetermineRetryExceptionsSupplier() {
        RetryStrategy onlyRetriesIOExceptions = RetryPolicyTests.createExceptionRetryStrategy(Collections.singletonList(IOException.class));
        RetryStrategy onlyRetriesTimeoutAndRuntimeExceptions = RetryPolicyTests.createExceptionRetryStrategy(Arrays.asList(TimeoutException.class, RuntimeException.class));
        return Stream.of(Arguments.of((Object[])new Object[]{onlyRetriesIOExceptions, new Throwable[]{new IOException(), new IOException(), new RuntimeException()}, RuntimeException.class}), Arguments.of((Object[])new Object[]{onlyRetriesIOExceptions, new Throwable[]{new IOException(), new RuntimeException()}, RuntimeException.class}), Arguments.of((Object[])new Object[]{onlyRetriesIOExceptions, new Throwable[]{new IOException(), new TimeoutException()}, TimeoutException.class}), Arguments.of((Object[])new Object[]{onlyRetriesIOExceptions, new Throwable[]{new IOException(), new IOException(), new IOException()}, IOException.class}), Arguments.of((Object[])new Object[]{onlyRetriesTimeoutAndRuntimeExceptions, new Throwable[]{new TimeoutException(), new RuntimeException(), new IOException()}, IOException.class}), Arguments.of((Object[])new Object[]{onlyRetriesTimeoutAndRuntimeExceptions, new Throwable[]{new TimeoutException(), new IOException()}, IOException.class}), Arguments.of((Object[])new Object[]{onlyRetriesTimeoutAndRuntimeExceptions, new Throwable[]{new RuntimeException(), new IOException()}, IOException.class}), Arguments.of((Object[])new Object[]{onlyRetriesTimeoutAndRuntimeExceptions, new Throwable[]{new TimeoutException(), new RuntimeException(), new TimeoutException()}, TimeoutException.class}), Arguments.of((Object[])new Object[]{onlyRetriesTimeoutAndRuntimeExceptions, new Throwable[]{new TimeoutException(), new RuntimeException(), new RuntimeException()}, RuntimeException.class}));
    }

    @Test
    public void retryMax() {
        int maxRetries = 5;
        HttpPipeline pipeline = new HttpPipelineBuilder().httpClient((HttpClient)new NoOpHttpClient(){
            int count = -1;

            @Override
            public Mono<HttpResponse> send(HttpRequest request) {
                Assertions.assertTrue((this.count++ < 5 ? 1 : 0) != 0);
                return Mono.just((Object)((Object)new MockHttpResponse(request, 500)));
            }
        }).policies(new HttpPipelinePolicy[]{new RetryPolicy((RetryStrategy)new FixedDelay(5, Duration.ofMillis(1L)))}).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)500, (int)response.getStatusCode())).verifyComplete();
    }

    @Test
    public void fixedDelayRetry() {
        int maxRetries = 5;
        long delayMillis = 500L;
        HttpPipeline pipeline = new HttpPipelineBuilder().httpClient((HttpClient)new NoOpHttpClient(){
            int count = -1;
            long previousAttemptMadeAt = -1L;

            @Override
            public Mono<HttpResponse> send(HttpRequest request) {
                if (this.count > 0) {
                    Assertions.assertTrue((System.currentTimeMillis() >= this.previousAttemptMadeAt + 500L ? 1 : 0) != 0);
                }
                Assertions.assertTrue((this.count++ < 5 ? 1 : 0) != 0);
                this.previousAttemptMadeAt = System.currentTimeMillis();
                return Mono.just((Object)((Object)new MockHttpResponse(request, 500)));
            }
        }).policies(new HttpPipelinePolicy[]{new RetryPolicy((RetryStrategy)new FixedDelay(5, Duration.ofMillis(500L)))}).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)500, (int)response.getStatusCode())).verifyComplete();
    }

    @Test
    public void exponentialDelayRetry() {
        int maxRetries = 5;
        long baseDelayMillis = 100L;
        long maxDelayMillis = 1000L;
        ExponentialBackoff exponentialBackoff = new ExponentialBackoff(5, Duration.ofMillis(100L), Duration.ofMillis(1000L));
        HttpPipeline pipeline = new HttpPipelineBuilder().httpClient((HttpClient)new NoOpHttpClient(){
            int count = -1;
            long previousAttemptMadeAt = -1L;

            @Override
            public Mono<HttpResponse> send(HttpRequest request) {
                if (this.count > 0) {
                    long expectedToBeMadeAt;
                    long requestMadeAt = System.currentTimeMillis();
                    Assertions.assertTrue((requestMadeAt >= (expectedToBeMadeAt = this.previousAttemptMadeAt + (1L << this.count - 1) * 95L) ? 1 : 0) != 0);
                }
                Assertions.assertTrue((this.count++ < 5 ? 1 : 0) != 0);
                this.previousAttemptMadeAt = System.currentTimeMillis();
                return Mono.just((Object)((Object)new MockHttpResponse(request, 503)));
            }
        }).policies(new HttpPipelinePolicy[]{new RetryPolicy((RetryStrategy)exponentialBackoff)}).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "http://localhost/"))).assertNext(response -> Assertions.assertEquals((int)503, (int)response.getStatusCode())).verifyComplete();
    }

    @Test
    public void retryConsumesBody() {
        AtomicInteger bodyConsumptionCount = new AtomicInteger();
        final Flux errorBody = Flux.generate(sink -> {
            bodyConsumptionCount.incrementAndGet();
            sink.next((Object)ByteBuffer.wrap("Should be consumed".getBytes(StandardCharsets.UTF_8)));
            sink.complete();
        });
        HttpPipeline pipeline = new HttpPipelineBuilder().policies(new HttpPipelinePolicy[]{new RetryPolicy((RetryStrategy)new FixedDelay(2, Duration.ofMillis(1L)))}).httpClient(request -> Mono.just((Object)new HttpResponse(request){

            public int getStatusCode() {
                return 503;
            }

            public String getHeaderValue(String name) {
                return this.getHeaders().getValue(name);
            }

            public HttpHeaders getHeaders() {
                return new HttpHeaders();
            }

            public Flux<ByteBuffer> getBody() {
                return errorBody;
            }

            public Mono<byte[]> getBodyAsByteArray() {
                return FluxUtil.collectBytesInByteBufferStream(this.getBody());
            }

            public Mono<String> getBodyAsString() {
                return this.getBodyAsString(StandardCharsets.UTF_8);
            }

            public Mono<String> getBodyAsString(Charset charset) {
                return this.getBodyAsByteArray().map(bytes -> new String((byte[])bytes, charset));
            }
        })).build();
        StepVerifier.create((Publisher)pipeline.send(new HttpRequest(HttpMethod.GET, "https://example.com"))).expectNextCount(1L).verifyComplete();
        Assertions.assertEquals((int)2, (int)bodyConsumptionCount.get());
    }

    @ParameterizedTest
    @MethodSource(value={"getWellKnownRetryDelaySupplier"})
    public void getWellKnownRetryDelay(HttpHeaders responseHeaders, RetryStrategy retryStrategy, Duration expected) {
        Assertions.assertEquals((Object)expected, (Object)RetryPolicy.getWellKnownRetryDelay((HttpHeaders)responseHeaders, (int)1, (RetryStrategy)retryStrategy, OffsetDateTime::now));
    }

    private static Stream<Arguments> getWellKnownRetryDelaySupplier() {
        RetryStrategy retryStrategy = (RetryStrategy)Mockito.mock(RetryStrategy.class);
        Mockito.when((Object)retryStrategy.calculateRetryDelay(ArgumentMatchers.anyInt())).thenReturn((Object)Duration.ofSeconds(1L));
        return Stream.of(Arguments.of((Object[])new Object[]{new HttpHeaders(), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("x-ms-retry-after-ms", "10"), retryStrategy, Duration.ofMillis(10L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("x-ms-retry-after-ms", "-10"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("x-ms-retry-after-ms", "ten"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("retry-after-ms", "64"), retryStrategy, Duration.ofMillis(64L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("retry-after-ms", "-10"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("retry-after-ms", "ten"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("Retry-After", "10"), retryStrategy, Duration.ofSeconds(10L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("Retry-After", "-10"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("Retry-After", "ten"), retryStrategy, Duration.ofSeconds(1L)}), Arguments.of((Object[])new Object[]{new HttpHeaders().set("Retry-After", OffsetDateTime.now().minusMinutes(1L).format(DateTimeFormatter.RFC_1123_DATE_TIME)), retryStrategy, Duration.ofSeconds(1L)}));
    }

    @Test
    public void retryAfterDateTime() {
        OffsetDateTime now = OffsetDateTime.now().withNano(0);
        HttpHeaders headers = new HttpHeaders().set("Retry-After", new DateTimeRfc1123(now.plusSeconds(30L)).toString());
        Duration actual = RetryPolicy.getWellKnownRetryDelay((HttpHeaders)headers, (int)1, null, () -> now);
        Assertions.assertEquals((Object)Duration.ofSeconds(30L), (Object)actual);
    }

    private static RetryStrategy createStatusCodeRetryStrategy(final int ... retriableErrorCodes) {
        return new RetryStrategy(){

            public int getMaxRetries() {
                return 2;
            }

            public Duration calculateRetryDelay(int retryAttempts) {
                return Duration.ofMillis(1L);
            }

            public boolean shouldRetry(HttpResponse httpResponse) {
                return Arrays.stream(retriableErrorCodes).anyMatch(retriableErrorCode -> httpResponse.getStatusCode() == retriableErrorCode);
            }
        };
    }

    private static RetryStrategy createExceptionRetryStrategy(final List<Class<? extends Throwable>> retriableExceptions) {
        return new RetryStrategy(){

            public int getMaxRetries() {
                return 2;
            }

            public Duration calculateRetryDelay(int retryAttempts) {
                return Duration.ofMillis(1L);
            }

            public boolean shouldRetryException(Throwable throwable) {
                return retriableExceptions.stream().anyMatch(retriableException -> retriableException.isAssignableFrom(throwable.getClass()));
            }
        };
    }
}

