/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.internal;

import com.azure.cosmos.CosmosClientException;
import com.azure.cosmos.internal.Exceptions;
import com.azure.cosmos.internal.IDocumentClientRetryPolicy;
import com.azure.cosmos.internal.IRetryPolicy;
import com.azure.cosmos.internal.RxDocumentServiceRequest;
import com.azure.cosmos.internal.Utils;
import java.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ResourceThrottleRetryPolicy
implements IDocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ResourceThrottleRetryPolicy.class);
    private static final int DefaultMaxWaitTimeInSeconds = 60;
    private static final int DefaultRetryInSeconds = 5;
    private final int backoffDelayFactor;
    private final int maxAttemptCount;
    private final Duration maxWaitTime;
    private int currentAttemptCount;
    private Duration cumulativeRetryDelay;

    public ResourceThrottleRetryPolicy(int maxAttemptCount, int maxWaitTimeInSeconds) {
        this(maxAttemptCount, maxWaitTimeInSeconds, 1);
    }

    public ResourceThrottleRetryPolicy(int maxAttemptCount) {
        this(maxAttemptCount, 60, 1);
    }

    public ResourceThrottleRetryPolicy(int maxAttemptCount, int maxWaitTimeInSeconds, int backoffDelayFactor) {
        Utils.checkStateOrThrow(maxWaitTimeInSeconds < 2147483, "maxWaitTimeInSeconds", "maxWaitTimeInSeconds must be less than 2147483");
        this.maxAttemptCount = maxAttemptCount;
        this.backoffDelayFactor = backoffDelayFactor;
        this.maxWaitTime = Duration.ofSeconds(maxWaitTimeInSeconds);
        this.currentAttemptCount = 0;
        this.cumulativeRetryDelay = Duration.ZERO;
    }

    @Override
    public Mono<IRetryPolicy.ShouldRetryResult> shouldRetry(Exception exception) {
        Duration retryDelay = Duration.ZERO;
        if (this.currentAttemptCount < this.maxAttemptCount && (retryDelay = this.checkIfRetryNeeded(exception)) != null) {
            ++this.currentAttemptCount;
            logger.warn("Operation will be retried after {} milliseconds. Current attempt {}, Cumulative delay {}", new Object[]{retryDelay.toMillis(), this.currentAttemptCount, this.cumulativeRetryDelay, exception});
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(retryDelay));
        }
        logger.debug("Operation will NOT be retried. Current attempt {}", (Object)this.currentAttemptCount, (Object)exception);
        return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
    }

    private Duration checkIfRetryNeeded(Exception exception) {
        Duration retryDelay = Duration.ZERO;
        CosmosClientException dce = Utils.as(exception, CosmosClientException.class);
        if (dce != null && Exceptions.isStatusCode(dce, 429)) {
            retryDelay = Duration.ofMillis(dce.getRetryAfterInMilliseconds());
            if (this.backoffDelayFactor > 1) {
                retryDelay = Duration.ofNanos(retryDelay.toNanos() * (long)this.backoffDelayFactor);
            }
            if (retryDelay.toMillis() < this.maxWaitTime.toMillis()) {
                this.cumulativeRetryDelay = retryDelay.plus(this.cumulativeRetryDelay);
                if (this.maxWaitTime.toMillis() >= this.cumulativeRetryDelay.toMillis()) {
                    if (retryDelay == Duration.ZERO) {
                        logger.trace("Received retryDelay of 0 with Http 429", (Throwable)exception);
                        retryDelay = Duration.ofSeconds(5L);
                    }
                    return retryDelay;
                }
            }
        }
        return null;
    }
}

