/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation;

import com.azure.cosmos.BridgeInternal;
import com.azure.cosmos.ConsistencyLevel;
import com.azure.cosmos.CosmosClientException;
import com.azure.cosmos.CosmosError;
import com.azure.cosmos.implementation.BackoffRetryUtility;
import com.azure.cosmos.implementation.Exceptions;
import com.azure.cosmos.implementation.GlobalEndpointManager;
import com.azure.cosmos.implementation.ISessionContainer;
import com.azure.cosmos.implementation.OperationType;
import com.azure.cosmos.implementation.PathsHelper;
import com.azure.cosmos.implementation.QueryCompatibilityMode;
import com.azure.cosmos.implementation.ReplicatedResourceClientUtils;
import com.azure.cosmos.implementation.ResourceType;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.RxDocumentServiceResponse;
import com.azure.cosmos.implementation.RxStoreModel;
import com.azure.cosmos.implementation.Strings;
import com.azure.cosmos.implementation.UserAgentContainer;
import com.azure.cosmos.implementation.Utils;
import com.azure.cosmos.implementation.WebExceptionRetryPolicy;
import com.azure.cosmos.implementation.directconnectivity.DirectBridgeInternal;
import com.azure.cosmos.implementation.directconnectivity.HttpUtils;
import com.azure.cosmos.implementation.directconnectivity.StoreResponse;
import com.azure.cosmos.implementation.http.HttpClient;
import com.azure.cosmos.implementation.http.HttpHeaders;
import com.azure.cosmos.implementation.http.HttpRequest;
import com.azure.cosmos.implementation.http.HttpResponse;
import com.azure.cosmos.implementation.http.ReactorNettyRequestRecord;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpResponseStatus;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

class RxGatewayStoreModel
implements RxStoreModel {
    private final Logger logger = LoggerFactory.getLogger(RxGatewayStoreModel.class);
    private final Map<String, String> defaultHeaders = new HashMap<String, String>();
    private final HttpClient httpClient;
    private final QueryCompatibilityMode queryCompatibilityMode;
    private final GlobalEndpointManager globalEndpointManager;
    private ConsistencyLevel defaultConsistencyLevel;
    private ISessionContainer sessionContainer;

    public RxGatewayStoreModel(ISessionContainer sessionContainer, ConsistencyLevel defaultConsistencyLevel, QueryCompatibilityMode queryCompatibilityMode, UserAgentContainer userAgentContainer, GlobalEndpointManager globalEndpointManager, HttpClient httpClient) {
        this.defaultHeaders.put("Cache-Control", "no-cache");
        this.defaultHeaders.put("x-ms-version", "2018-12-31");
        if (userAgentContainer == null) {
            userAgentContainer = new UserAgentContainer();
        }
        this.defaultHeaders.put("User-Agent", userAgentContainer.getUserAgent());
        if (defaultConsistencyLevel != null) {
            this.defaultHeaders.put("x-ms-consistency-level", defaultConsistencyLevel.toString());
        }
        this.defaultConsistencyLevel = defaultConsistencyLevel;
        this.globalEndpointManager = globalEndpointManager;
        this.queryCompatibilityMode = queryCompatibilityMode;
        this.httpClient = httpClient;
        this.sessionContainer = sessionContainer;
    }

    private Flux<RxDocumentServiceResponse> doCreate(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.POST);
    }

    private Flux<RxDocumentServiceResponse> upsert(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.POST);
    }

    private Flux<RxDocumentServiceResponse> read(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.GET);
    }

    private Flux<RxDocumentServiceResponse> replace(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.PUT);
    }

    private Flux<RxDocumentServiceResponse> delete(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.DELETE);
    }

    private Flux<RxDocumentServiceResponse> execute(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.POST);
    }

    private Flux<RxDocumentServiceResponse> readFeed(RxDocumentServiceRequest request) {
        return this.performRequest(request, HttpMethod.GET);
    }

    private Flux<RxDocumentServiceResponse> query(RxDocumentServiceRequest request) {
        if (request.getOperationType() != OperationType.QueryPlan) {
            request.getHeaders().put("x-ms-documentdb-isquery", "true");
        }
        switch (this.queryCompatibilityMode) {
            case SqlQuery: {
                request.getHeaders().put("Content-Type", "application/sql");
                break;
            }
            default: {
                request.getHeaders().put("Content-Type", "application/query+json");
            }
        }
        return this.performRequest(request, HttpMethod.POST);
    }

    public Flux<RxDocumentServiceResponse> performRequest(RxDocumentServiceRequest request, HttpMethod method) {
        try {
            if (request.requestContext.cosmosResponseDiagnostics == null) {
                request.requestContext.cosmosResponseDiagnostics = BridgeInternal.createCosmosResponseDiagnostics();
            }
            URI uri = this.getUri(request);
            HttpHeaders httpHeaders = this.getHttpRequestHeaders(request.getHeaders());
            Flux byteBufObservable = Flux.empty();
            if (request.getContentObservable() != null) {
                byteBufObservable = request.getContentObservable().map(Unpooled::wrappedBuffer);
            } else if (request.getContent() != null) {
                byteBufObservable = Flux.just((Object)Unpooled.wrappedBuffer((byte[])request.getContent()));
            }
            HttpRequest httpRequest = new HttpRequest(method, uri, uri.getPort(), httpHeaders, (Flux<ByteBuf>)byteBufObservable);
            Mono<HttpResponse> httpResponseMono = this.httpClient.send(httpRequest);
            return this.toDocumentServiceResponse(httpResponseMono, request);
        }
        catch (Exception e) {
            return Flux.error((Throwable)e);
        }
    }

    private HttpHeaders getHttpRequestHeaders(Map<String, String> headers) {
        HttpHeaders httpHeaders = new HttpHeaders(this.defaultHeaders.size());
        for (Map.Entry<String, String> entry : this.defaultHeaders.entrySet()) {
            if (headers.containsKey(entry.getKey())) continue;
            httpHeaders.set(entry.getKey(), entry.getValue());
        }
        if (headers != null) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                if (entry.getValue() == null) {
                    httpHeaders.set(entry.getKey(), "");
                    continue;
                }
                httpHeaders.set(entry.getKey(), entry.getValue());
            }
        }
        return httpHeaders;
    }

    private URI getUri(RxDocumentServiceRequest request) throws URISyntaxException {
        URI rootUri = request.getEndpointOverride();
        if (rootUri == null) {
            rootUri = request.getIsMedia() ? (URI)this.globalEndpointManager.getWriteEndpoints().get(0) : this.globalEndpointManager.resolveServiceEndpoint(request);
        }
        String path = PathsHelper.generatePath(request.getResourceType(), request, request.isFeed);
        if (request.getResourceType().equals((Object)ResourceType.DatabaseAccount)) {
            path = "";
        }
        return new URI("https", null, rootUri.getHost(), rootUri.getPort(), this.ensureSlashPrefixed(path), null, null);
    }

    private String ensureSlashPrefixed(String path) {
        if (path == null) {
            return path;
        }
        if (path.startsWith("/")) {
            return path;
        }
        return "/" + path;
    }

    private Flux<RxDocumentServiceResponse> toDocumentServiceResponse(Mono<HttpResponse> httpResponseMono, RxDocumentServiceRequest request) {
        return httpResponseMono.flatMap(httpResponse -> {
            HttpHeaders httpResponseHeaders = httpResponse.headers();
            int httpResponseStatus = httpResponse.statusCode();
            Flux contentObservable = request.getOperationType() == OperationType.Delete ? Flux.just((Object)"") : httpResponse.bodyAsString().switchIfEmpty(Mono.just((Object)"")).flux();
            return contentObservable.flatMap(content -> {
                try {
                    ReactorNettyRequestRecord reactorNettyRequestRecord = httpResponse.request().getReactorNettyRequestRecord();
                    if (reactorNettyRequestRecord != null) {
                        reactorNettyRequestRecord.setTimeCompleted(OffsetDateTime.now());
                        BridgeInternal.setTransportClientRequestTimelineOnDiagnostics(request.requestContext.cosmosResponseDiagnostics, reactorNettyRequestRecord.takeTimelineSnapshot());
                    }
                    this.validateOrThrow(request, HttpResponseStatus.valueOf((int)httpResponseStatus), httpResponseHeaders, (String)content, null);
                    StoreResponse rsp = new StoreResponse(httpResponseStatus, HttpUtils.unescape(httpResponseHeaders.toMap().entrySet()), (String)content);
                    DirectBridgeInternal.setRequestTimeline(rsp, reactorNettyRequestRecord.takeTimelineSnapshot());
                    if (request.requestContext.cosmosResponseDiagnostics != null) {
                        BridgeInternal.recordGatewayResponse(request.requestContext.cosmosResponseDiagnostics, request, rsp, null);
                        DirectBridgeInternal.setCosmosResponseDiagnostics(rsp, request.requestContext.cosmosResponseDiagnostics);
                    }
                    return Flux.just((Object)rsp);
                }
                catch (Exception e) {
                    return Flux.error((Throwable)e);
                }
            }).single();
        }).map(RxDocumentServiceResponse::new).onErrorResume(throwable -> {
            Throwable unwrappedException = reactor.core.Exceptions.unwrap((Throwable)throwable);
            if (!(unwrappedException instanceof Exception)) {
                this.logger.error("Unexpected failure {}", (Object)unwrappedException.getMessage(), (Object)unwrappedException);
                return Mono.error((Throwable)unwrappedException);
            }
            Exception exception = (Exception)unwrappedException;
            if (!(exception instanceof CosmosClientException)) {
                this.logger.error("Network failure", (Throwable)exception);
                CosmosClientException dce = BridgeInternal.createCosmosClientException(0, exception);
                BridgeInternal.setRequestHeaders(dce, request.getHeaders());
                return Mono.error((Throwable)dce);
            }
            if (request.requestContext.cosmosResponseDiagnostics != null) {
                BridgeInternal.recordGatewayResponse(request.requestContext.cosmosResponseDiagnostics, request, null, (CosmosClientException)exception);
                BridgeInternal.setCosmosResponseDiagnostics((CosmosClientException)exception, request.requestContext.cosmosResponseDiagnostics);
            }
            return Mono.error((Throwable)exception);
        }).flux();
    }

    private void validateOrThrow(RxDocumentServiceRequest request, HttpResponseStatus status, HttpHeaders headers, String body, InputStream inputStream) throws CosmosClientException {
        int statusCode = status.code();
        if (statusCode >= 400) {
            if (body == null && inputStream != null) {
                try {
                    body = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
                }
                catch (IOException e) {
                    this.logger.error("Failed to get content from the http response", (Throwable)e);
                    CosmosClientException dce = BridgeInternal.createCosmosClientException(0, e);
                    BridgeInternal.setRequestHeaders(dce, request.getHeaders());
                    throw dce;
                }
                finally {
                    IOUtils.closeQuietly((InputStream)inputStream);
                }
            }
            String statusCodeString = status.reasonPhrase() != null ? status.reasonPhrase().replace(" ", "") : "";
            CosmosError cosmosError = StringUtils.isNotEmpty((CharSequence)body) ? BridgeInternal.createCosmosError(body) : new CosmosError();
            cosmosError = new CosmosError(statusCodeString, String.format("%s, StatusCode: %s", cosmosError.getMessage(), statusCodeString), cosmosError.getPartitionedQueryExecutionInfo());
            CosmosClientException dce = BridgeInternal.createCosmosClientException(statusCode, cosmosError, headers.toMap());
            BridgeInternal.setRequestHeaders(dce, request.getHeaders());
            throw dce;
        }
    }

    private Flux<RxDocumentServiceResponse> invokeAsyncInternal(RxDocumentServiceRequest request) {
        switch (request.getOperationType()) {
            case Create: {
                return this.doCreate(request);
            }
            case Upsert: {
                return this.upsert(request);
            }
            case Delete: {
                return this.delete(request);
            }
            case ExecuteJavaScript: {
                return this.execute(request);
            }
            case Read: {
                return this.read(request);
            }
            case ReadFeed: {
                return this.readFeed(request);
            }
            case Replace: {
                return this.replace(request);
            }
            case SqlQuery: 
            case Query: 
            case QueryPlan: {
                return this.query(request);
            }
        }
        throw new IllegalStateException("Unknown operation setType " + (Object)((Object)request.getOperationType()));
    }

    private Mono<RxDocumentServiceResponse> invokeAsync(RxDocumentServiceRequest request) {
        Callable funcDelegate = () -> this.invokeAsyncInternal(request).single();
        return BackoffRetryUtility.executeRetry(funcDelegate, new WebExceptionRetryPolicy());
    }

    @Override
    public Mono<RxDocumentServiceResponse> processMessage(RxDocumentServiceRequest request) {
        this.applySessionToken(request);
        Mono<RxDocumentServiceResponse> responseObs = this.invokeAsync(request);
        return responseObs.onErrorResume(e -> {
            CosmosClientException dce = Utils.as(e, CosmosClientException.class);
            if (dce == null) {
                this.logger.error("unexpected failure {}", (Object)e.getMessage(), e);
                return Mono.error((Throwable)e);
            }
            if (!ReplicatedResourceClientUtils.isMasterResource(request.getResourceType()) && (dce.getStatusCode() == 412 || dce.getStatusCode() == 409 || dce.getStatusCode() == 404 && !Exceptions.isSubStatusCode(dce, 1002))) {
                this.captureSessionToken(request, dce.getResponseHeaders());
            }
            return Mono.error((Throwable)dce);
        }).map(response -> {
            this.captureSessionToken(request, response.getResponseHeaders());
            return response;
        });
    }

    private void captureSessionToken(RxDocumentServiceRequest request, Map<String, String> responseHeaders) {
        if (request.getResourceType() == ResourceType.DocumentCollection && request.getOperationType() == OperationType.Delete) {
            String resourceId = request.getIsNameBased() ? responseHeaders.get("x-ms-content-path") : request.getResourceId();
            this.sessionContainer.clearTokenByResourceId(resourceId);
        } else {
            this.sessionContainer.setSessionToken(request, responseHeaders);
        }
    }

    private void applySessionToken(RxDocumentServiceRequest request) {
        boolean sessionConsistency;
        Map<String, String> headers = request.getHeaders();
        if (headers != null && !Strings.isNullOrEmpty(request.getHeaders().get("x-ms-session-token"))) {
            if (ReplicatedResourceClientUtils.isMasterResource(request.getResourceType())) {
                request.getHeaders().remove("x-ms-session-token");
            }
            return;
        }
        String requestConsistencyLevel = headers.get("x-ms-consistency-level");
        boolean bl = sessionConsistency = this.defaultConsistencyLevel == ConsistencyLevel.SESSION || !Strings.isNullOrEmpty(requestConsistencyLevel) && Strings.areEqual(requestConsistencyLevel, ConsistencyLevel.SESSION.toString());
        if (!sessionConsistency || ReplicatedResourceClientUtils.isMasterResource(request.getResourceType())) {
            return;
        }
        String sessionToken = this.sessionContainer.resolveGlobalSessionToken(request);
        if (!Strings.isNullOrEmpty(sessionToken)) {
            headers.put("x-ms-session-token", sessionToken);
        }
    }
}

