/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation.directconnectivity;

import com.azure.cosmos.BridgeInternal;
import com.azure.cosmos.CosmosException;
import com.azure.cosmos.implementation.GoneException;
import com.azure.cosmos.implementation.InvalidPartitionException;
import com.azure.cosmos.implementation.PartitionIsMigratingException;
import com.azure.cosmos.implementation.PartitionKeyRangeGoneException;
import com.azure.cosmos.implementation.PartitionKeyRangeIsSplittingException;
import com.azure.cosmos.implementation.Quadruple;
import com.azure.cosmos.implementation.RetryPolicyWithDiagnostics;
import com.azure.cosmos.implementation.RetryWithException;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.ShouldRetryResult;
import com.azure.cosmos.implementation.apachecommons.lang.tuple.Pair;
import com.azure.cosmos.implementation.guava25.base.Preconditions;
import java.time.Duration;
import java.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class GoneAndRetryWithRetryPolicy
extends RetryPolicyWithDiagnostics {
    private static final Logger logger = LoggerFactory.getLogger(GoneAndRetryWithRetryPolicy.class);
    private final GoneRetryPolicy goneRetryPolicy;
    private final RetryWithRetryPolicy retryWithRetryPolicy;
    private final Instant start;
    private volatile Instant end;
    private volatile RetryWithException lastRetryWithException;

    public GoneAndRetryWithRetryPolicy(RxDocumentServiceRequest request, Integer waitTimeInSeconds) {
        this.goneRetryPolicy = new GoneRetryPolicy(request, waitTimeInSeconds);
        this.retryWithRetryPolicy = new RetryWithRetryPolicy(waitTimeInSeconds);
        this.start = Instant.now();
    }

    @Override
    public Mono<ShouldRetryResult> shouldRetry(Exception exception) {
        return this.retryWithRetryPolicy.shouldRetry(exception).flatMap(retryWithResult -> {
            if (retryWithResult.shouldRetry) {
                return Mono.just((Object)retryWithResult);
            }
            return this.goneRetryPolicy.shouldRetry(exception).flatMap(goneRetryResult -> {
                if (!goneRetryResult.shouldRetry) {
                    logger.debug("Operation will NOT be retried. Exception:", (Throwable)exception);
                    this.end = Instant.now();
                }
                return Mono.just((Object)goneRetryResult);
            });
        });
    }

    private Duration getElapsedTime() {
        Instant endSnapshot = this.end != null ? this.end : Instant.now();
        return Duration.between(this.start, endSnapshot);
    }

    class RetryWithRetryPolicy
    extends RetryPolicyWithDiagnostics {
        private static final int DEFAULT_WAIT_TIME_IN_SECONDS = 30;
        private static final int MAXIMUM_BACKOFF_TIME_IN_MS = 15000;
        private static final int INITIAL_BACKOFF_TIME_MS = 10;
        private static final int BACK_OFF_MULTIPLIER = 2;
        private volatile int attemptCount = 1;
        private volatile int currentBackoffMilliseconds = 10;
        private final int waitTimeInSeconds;

        public RetryWithRetryPolicy(Integer waitTimeInSeconds) {
            this.waitTimeInSeconds = waitTimeInSeconds != null ? waitTimeInSeconds : 30;
        }

        @Override
        public Mono<ShouldRetryResult> shouldRetry(Exception exception) {
            if (!(exception instanceof RetryWithException)) {
                logger.debug("Operation will NOT be retried. Current attempt {}, Exception: ", (Object)this.attemptCount, (Object)exception);
                return Mono.just((Object)ShouldRetryResult.noRetry());
            }
            RetryWithException lastRetryWithException = (RetryWithException)((Object)exception);
            GoneAndRetryWithRetryPolicy.this.lastRetryWithException = lastRetryWithException;
            long remainingMilliseconds = (long)this.waitTimeInSeconds * 1000L - GoneAndRetryWithRetryPolicy.this.getElapsedTime().toMillis();
            int currentRetryAttemptCount = this.attemptCount++;
            if (remainingMilliseconds <= 0L) {
                logger.warn("Received RetryWithException after backoff/retry. Will fail the request.", (Throwable)((Object)lastRetryWithException));
                return Mono.just((Object)ShouldRetryResult.error((Exception)((Object)lastRetryWithException)));
            }
            Duration backoffTime = Duration.ofMillis(Math.min(Math.min((long)this.currentBackoffMilliseconds, remainingMilliseconds), 15000L));
            this.currentBackoffMilliseconds *= 2;
            logger.debug("BackoffTime: {} ms.", (Object)backoffTime.toMillis());
            long timeoutInMillSec = remainingMilliseconds - backoffTime.toMillis();
            Duration timeout = timeoutInMillSec > 0L ? Duration.ofMillis(timeoutInMillSec) : Duration.ofMillis(15000L);
            logger.info("Received RetryWithException, will retry, ", (Throwable)exception);
            return Mono.just((Object)ShouldRetryResult.retryAfter(backoffTime, Quadruple.with(false, true, timeout, currentRetryAttemptCount)));
        }
    }

    class GoneRetryPolicy
    extends RetryPolicyWithDiagnostics {
        private static final int DEFAULT_WAIT_TIME_IN_SECONDS = 30;
        private static final int MAXIMUM_BACKOFF_TIME_IN_SECONDS = 15;
        private static final int INITIAL_BACKOFF_TIME = 1;
        private static final int BACK_OFF_MULTIPLIER = 2;
        private final RxDocumentServiceRequest request;
        private volatile int attemptCount = 1;
        private volatile int attemptCountInvalidPartition = 1;
        private volatile int currentBackoffSeconds = 1;
        private final int waitTimeInSeconds;

        public GoneRetryPolicy(RxDocumentServiceRequest request, Integer waitTimeInSeconds) {
            Preconditions.checkNotNull(request, "request must not be null.");
            this.request = request;
            this.waitTimeInSeconds = waitTimeInSeconds != null ? waitTimeInSeconds : 30;
        }

        private boolean isNonRetryableException(Exception exception) {
            if (exception instanceof GoneException || exception instanceof RetryWithException || exception instanceof PartitionIsMigratingException || exception instanceof PartitionKeyRangeIsSplittingException) {
                return false;
            }
            if (exception instanceof InvalidPartitionException) {
                return this.request.getPartitionKeyRangeIdentity() != null && this.request.getPartitionKeyRangeIdentity().getCollectionRid() != null;
            }
            return true;
        }

        private CosmosException logAndWrapExceptionWithLastRetryWithException(Exception exception) {
            String exceptionType;
            if (exception instanceof GoneException) {
                exceptionType = "GoneException";
            } else if (exception instanceof PartitionKeyRangeGoneException) {
                exceptionType = "PartitionKeyRangeGoneException";
            } else if (exception instanceof InvalidPartitionException) {
                exceptionType = "InvalidPartitionException";
            } else if (exception instanceof PartitionKeyRangeIsSplittingException) {
                exceptionType = "PartitionKeyRangeIsSplittingException";
            } else {
                if (exception instanceof CosmosException) {
                    logger.warn("Received CosmosException after backoff/retry. Will fail the request.", (Throwable)exception);
                    return (CosmosException)((Object)exception);
                }
                throw new IllegalStateException("Invalid exception type", exception);
            }
            RetryWithException lastRetryWithExceptionSnapshot = GoneAndRetryWithRetryPolicy.this.lastRetryWithException;
            if (lastRetryWithExceptionSnapshot != null) {
                logger.warn("Received {} after backoff/retry including at least one RetryWithException. Will fail the request with RetryWithException. {}: {}. RetryWithException: {}", new Object[]{exceptionType, exceptionType, exception, lastRetryWithExceptionSnapshot});
                return lastRetryWithExceptionSnapshot;
            }
            logger.warn("Received {} after backoff/retry. Will fail the request. {}", (Object)exceptionType, (Object)exception);
            return BridgeInternal.createServiceUnavailableException(exception);
        }

        @Override
        public Mono<ShouldRetryResult> shouldRetry(Exception exception) {
            long timeoutInMillSec;
            Duration backoffTime = Duration.ofSeconds(0L);
            if (this.isNonRetryableException(exception)) {
                logger.debug("Operation will NOT be retried. Current attempt {}, Exception: ", (Object)this.attemptCount, (Object)exception);
                return Mono.just((Object)ShouldRetryResult.noRetry());
            }
            if (exception instanceof GoneException && !this.request.isReadOnly() && BridgeInternal.hasSendingRequestStarted((CosmosException)((Object)exception)) && !((GoneException)((Object)exception)).isBasedOn410ResponseFromService()) {
                logger.warn("Operation will NOT be retried. Write operations which failed due to transient transport errors can not be retried safely when sending the request to the service because they aren't idempotent. Current attempt {}, Exception: ", (Object)this.attemptCount, (Object)exception);
                return Mono.just((Object)ShouldRetryResult.noRetry(Quadruple.with(true, true, Duration.ofMillis(0L), this.attemptCount)));
            }
            long remainingSeconds = (long)this.waitTimeInSeconds - GoneAndRetryWithRetryPolicy.this.getElapsedTime().toMillis() / 1000L;
            int currentRetryAttemptCount = this.attemptCount;
            if (this.attemptCount++ > 1) {
                if (remainingSeconds <= 0L) {
                    CosmosException exceptionToThrow = this.logAndWrapExceptionWithLastRetryWithException(exception);
                    return Mono.just((Object)ShouldRetryResult.error((Exception)((Object)exceptionToThrow)));
                }
                backoffTime = Duration.ofSeconds(Math.min(Math.min((long)this.currentBackoffSeconds, remainingSeconds), 15L));
                this.currentBackoffSeconds *= 2;
                logger.debug("BackoffTime: {} seconds.", (Object)backoffTime.getSeconds());
            }
            Duration timeout = (timeoutInMillSec = remainingSeconds * 1000L - backoffTime.toMillis()) > 0L ? Duration.ofMillis(timeoutInMillSec) : Duration.ofSeconds(15L);
            logger.debug("Timeout. {} - BackoffTime {} - currentBackoffSeconds {} - CurrentRetryAttemptCount {}", new Object[]{timeout.toMillis(), backoffTime, this.currentBackoffSeconds, currentRetryAttemptCount});
            Pair<Mono<ShouldRetryResult>, Boolean> exceptionHandlingResult = this.handleException(exception);
            Mono<ShouldRetryResult> result = exceptionHandlingResult.getLeft();
            if (result != null) {
                return result;
            }
            boolean forceRefreshAddressCache = exceptionHandlingResult.getRight();
            return Mono.just((Object)ShouldRetryResult.retryAfter(backoffTime, Quadruple.with(forceRefreshAddressCache, true, timeout, currentRetryAttemptCount)));
        }

        private Pair<Mono<ShouldRetryResult>, Boolean> handleException(Exception exception) {
            if (exception instanceof GoneException) {
                return this.handleGoneException((GoneException)((Object)exception));
            }
            if (exception instanceof PartitionIsMigratingException) {
                return this.handlePartitionIsMigratingException((PartitionIsMigratingException)((Object)exception));
            }
            if (exception instanceof InvalidPartitionException) {
                return this.handleInvalidPartitionException((InvalidPartitionException)((Object)exception));
            }
            if (exception instanceof PartitionKeyRangeIsSplittingException) {
                return this.handlePartitionKeyIsSplittingException((PartitionKeyRangeIsSplittingException)((Object)exception));
            }
            throw new IllegalStateException("Invalid exception type", exception);
        }

        private Pair<Mono<ShouldRetryResult>, Boolean> handleGoneException(GoneException exception) {
            logger.info("Received gone exception, will retry, {}", (Object)exception.toString());
            return Pair.of(null, true);
        }

        private Pair<Mono<ShouldRetryResult>, Boolean> handlePartitionIsMigratingException(PartitionIsMigratingException exception) {
            logger.info("Received PartitionIsMigratingException, will retry, {}", (Object)exception.toString());
            this.request.forceCollectionRoutingMapRefresh = true;
            return Pair.of(null, true);
        }

        private Pair<Mono<ShouldRetryResult>, Boolean> handlePartitionKeyIsSplittingException(PartitionKeyRangeIsSplittingException exception) {
            this.request.requestContext.resolvedPartitionKeyRange = null;
            this.request.requestContext.quorumSelectedLSN = -1L;
            this.request.requestContext.quorumSelectedStoreResponse = null;
            logger.info("Received partition key range splitting exception, will retry, {}", (Object)exception.toString());
            this.request.forcePartitionKeyRangeRefresh = true;
            return Pair.of(null, false);
        }

        private Pair<Mono<ShouldRetryResult>, Boolean> handleInvalidPartitionException(InvalidPartitionException exception) {
            this.request.requestContext.quorumSelectedLSN = -1L;
            this.request.requestContext.resolvedPartitionKeyRange = null;
            this.request.requestContext.quorumSelectedStoreResponse = null;
            this.request.requestContext.globalCommittedSelectedLSN = -1L;
            if (this.attemptCountInvalidPartition++ > 2) {
                logger.warn("Received second InvalidPartitionException after backoff/retry. Will fail the request. {}", (Object)exception.toString());
                return Pair.of(Mono.just((Object)ShouldRetryResult.error((Exception)((Object)BridgeInternal.createServiceUnavailableException((Exception)((Object)exception))))), false);
            }
            logger.info("Received invalid collection exception, will retry, {}", (Object)exception.toString());
            this.request.forceNameCacheRefresh = true;
            return Pair.of(null, false);
        }
    }
}

