/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation.caches;

import com.azure.cosmos.implementation.DocumentCollection;
import com.azure.cosmos.implementation.caches.AsyncLazy;
import com.azure.cosmos.implementation.caches.IEqualityComparer;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class AsyncCache<TKey, TValue> {
    private final Logger logger = LoggerFactory.getLogger(AsyncCache.class);
    private final ConcurrentHashMap<TKey, AsyncLazy<TValue>> values;
    private final IEqualityComparer<TValue> equalityComparer;

    public AsyncCache(IEqualityComparer<TValue> equalityComparer, ConcurrentHashMap<TKey, AsyncLazy<TValue>> values) {
        this.equalityComparer = equalityComparer;
        this.values = values;
    }

    public AsyncCache(IEqualityComparer<TValue> equalityComparer) {
        this(equalityComparer, new ConcurrentHashMap());
    }

    public AsyncCache() {
        this((value1, value2) -> {
            if (value1 == value2) {
                return true;
            }
            if (value1 == null || value2 == null) {
                return false;
            }
            return value1.equals(value2);
        });
    }

    public void set(TKey key, TValue value) {
        this.logger.debug("set cache[{}]={}", key, value);
        this.values.put(key, new AsyncLazy<TValue>(value));
    }

    public Mono<TValue> getAsync(TKey key, TValue obsoleteValue, Callable<Mono<TValue>> singleValueInitFunc) {
        AsyncLazy initialLazyValue = this.values.get(key);
        if (initialLazyValue != null) {
            this.logger.debug("cache[{}] exists", key);
            return initialLazyValue.single().flux().flatMap(value -> {
                if (!this.equalityComparer.areEqual(value, obsoleteValue)) {
                    this.logger.debug("Returning cache[{}] as it is different from obsoleteValue", key);
                    return Flux.just((Object)value);
                }
                this.logger.debug("cache[{}] result value is obsolete ({}), computing new value", key, obsoleteValue);
                AsyncLazy<Callable> asyncLazy = new AsyncLazy<Callable>(singleValueInitFunc);
                AsyncLazy actualValue = this.values.merge(key, asyncLazy, (lazyValue1, lazyValue2) -> lazyValue1 == initialLazyValue ? lazyValue2 : lazyValue1);
                return actualValue.single().flux();
            }, err -> {
                this.logger.debug("cache[{}] resulted in error, computing new value", key, err);
                AsyncLazy<Callable> asyncLazy = new AsyncLazy<Callable>(singleValueInitFunc);
                AsyncLazy resultAsyncLazy = this.values.merge(key, asyncLazy, (lazyValue1, lazyValu2) -> lazyValue1 == initialLazyValue ? lazyValu2 : lazyValue1);
                return resultAsyncLazy.single().flux();
            }, Flux::empty).single();
        }
        this.logger.debug("cache[{}] doesn't exist, computing new value", key);
        AsyncLazy<Callable<Mono<TValue>>> asyncLazy = new AsyncLazy<Callable<Mono<TValue>>>(singleValueInitFunc);
        AsyncLazy resultAsyncLazy = this.values.merge(key, asyncLazy, (lazyValue1, lazyValu2) -> lazyValue1 == initialLazyValue ? lazyValu2 : lazyValue1);
        return resultAsyncLazy.single();
    }

    public void remove(TKey key) {
        this.values.remove(key);
    }

    public Mono<TValue> removeAsync(TKey key) {
        AsyncLazy<TValue> lazy = this.values.remove(key);
        return lazy.single();
    }

    public void clear() {
        this.values.clear();
    }

    public void refresh(TKey key, Callable<Mono<TValue>> singleValueInitFunc) {
        this.logger.debug("refreshing cache[{}]", key);
        AsyncLazy initialLazyValue = this.values.get(key);
        if (initialLazyValue != null && (initialLazyValue.isSucceeded() || initialLazyValue.isFaulted())) {
            AsyncLazy<Callable<Mono<TValue>>> newLazyValue = new AsyncLazy<Callable<Mono<TValue>>>(singleValueInitFunc);
            this.values.merge(key, newLazyValue, (lazyValue1, lazyValu2) -> lazyValue1 == initialLazyValue ? lazyValu2 : lazyValue1);
        }
    }

    public static abstract class SerializableAsyncCache<TKey, TValue>
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private static transient Logger logger = LoggerFactory.getLogger(SerializableAsyncCache.class);
        protected transient AsyncCache<TKey, TValue> cache;

        protected SerializableAsyncCache() {
        }

        public static <TKey, TValue> SerializableAsyncCache<TKey, TValue> from(AsyncCache<TKey, TValue> cache, Class<TKey> keyClass, Class<TValue> valueClass) {
            if (keyClass == String.class && valueClass == DocumentCollection.class) {
                SerializableAsyncCollectionCache sacc = new SerializableAsyncCollectionCache();
                sacc.cache = cache;
                return sacc;
            }
            throw new RuntimeException("not supported");
        }

        protected abstract void serializeKey(ObjectOutputStream var1, TKey var2) throws IOException;

        protected abstract void serializeValue(ObjectOutputStream var1, TValue var2) throws IOException;

        protected abstract TKey deserializeKey(ObjectInputStream var1) throws IOException;

        protected abstract TValue deserializeValue(ObjectInputStream var1) throws IOException, ClassNotFoundException;

        public AsyncCache<TKey, TValue> toAsyncCache() {
            return this.cache;
        }

        private void writeObject(ObjectOutputStream oos) throws IOException {
            logger.info("Serializing {}", this.getClass());
            HashMap paris = new HashMap();
            for (Map.Entry entry : ((AsyncCache)this.cache).values.entrySet()) {
                Object key = entry.getKey();
                Optional value = ((AsyncLazy)entry.getValue()).tryGet();
                if (!value.isPresent()) continue;
                paris.put(key, value.get());
            }
            oos.writeInt(paris.size());
            for (Map.Entry entry : paris.entrySet()) {
                this.serializeKey(oos, entry.getKey());
                this.serializeValue(oos, entry.getValue());
            }
            oos.writeObject(((AsyncCache)this.cache).equalityComparer);
        }

        private void readObject(ObjectInputStream ois) throws ClassNotFoundException, IOException {
            logger.info("Deserializing {}", this.getClass());
            int size = ois.readInt();
            ConcurrentHashMap<TKey, AsyncLazy<TValue>> pairs = new ConcurrentHashMap<TKey, AsyncLazy<TValue>>();
            for (int i = 0; i < size; ++i) {
                TKey key = this.deserializeKey(ois);
                TValue value = this.deserializeValue(ois);
                pairs.put(key, new AsyncLazy<TValue>(value));
            }
            IEqualityComparer equalityComparer = (IEqualityComparer)ois.readObject();
            this.cache = new AsyncCache(equalityComparer, pairs);
        }

        public static class SerializableAsyncCollectionCache
        extends SerializableAsyncCache<String, DocumentCollection> {
            private static final long serialVersionUID = 1L;

            private SerializableAsyncCollectionCache() {
            }

            @Override
            protected void serializeKey(ObjectOutputStream oos, String s) throws IOException {
                oos.writeUTF(s);
            }

            @Override
            protected void serializeValue(ObjectOutputStream oos, DocumentCollection documentCollection) throws IOException {
                oos.writeObject(DocumentCollection.SerializableDocumentCollection.from(documentCollection));
            }

            @Override
            protected String deserializeKey(ObjectInputStream ois) throws IOException {
                return ois.readUTF();
            }

            @Override
            protected DocumentCollection deserializeValue(ObjectInputStream ois) throws IOException, ClassNotFoundException {
                return ((DocumentCollection.SerializableDocumentCollection)ois.readObject()).getWrappedItem();
            }
        }
    }
}

