/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation;

import com.azure.cosmos.CosmosException;
import com.azure.cosmos.implementation.DocumentClientRetryPolicy;
import com.azure.cosmos.implementation.Exceptions;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.ShouldRetryResult;
import com.azure.cosmos.implementation.Utils;
import java.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ResourceThrottleRetryPolicy
extends DocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ResourceThrottleRetryPolicy.class);
    private static final Duration DEFAULT_MAX_WAIT_TIME_IN_SECONDS = Duration.ofSeconds(60L);
    private static final Duration DEFAULT_RETRY_IN_SECONDS = Duration.ofSeconds(5L);
    private final int backoffDelayFactor;
    private final int maxAttemptCount;
    private final Duration maxWaitTime;
    private int currentAttemptCount;
    private Duration cumulativeRetryDelay;

    public ResourceThrottleRetryPolicy(int maxAttemptCount, Duration maxWaitTime) {
        this(maxAttemptCount, maxWaitTime, 1);
    }

    public ResourceThrottleRetryPolicy(int maxAttemptCount) {
        this(maxAttemptCount, DEFAULT_MAX_WAIT_TIME_IN_SECONDS, 1);
    }

    public ResourceThrottleRetryPolicy(int maxAttemptCount, Duration maxWaitTime, int backoffDelayFactor) {
        Utils.checkStateOrThrow(maxWaitTime.getSeconds() <= 2147483L, "maxWaitTime", "maxWaitTime must not be larger than 2147483");
        this.maxAttemptCount = maxAttemptCount;
        this.backoffDelayFactor = backoffDelayFactor;
        this.maxWaitTime = maxWaitTime;
        this.currentAttemptCount = 0;
        this.cumulativeRetryDelay = Duration.ZERO;
    }

    @Override
    public Mono<ShouldRetryResult> shouldRetry(Exception exception) {
        Duration retryDelay = Duration.ZERO;
        if (this.currentAttemptCount < this.maxAttemptCount && (retryDelay = this.checkIfRetryNeeded(exception)) != null) {
            ++this.currentAttemptCount;
            logger.debug("Operation will be retried after {} milliseconds. Current attempt {}, Cumulative delay {}", new Object[]{retryDelay.toMillis(), this.currentAttemptCount, this.cumulativeRetryDelay, exception});
            return Mono.just((Object)ShouldRetryResult.retryAfter(retryDelay));
        }
        if (retryDelay != null) {
            logger.warn("Operation will NOT be retried. Current attempt {}", (Object)this.currentAttemptCount, (Object)exception);
        } else {
            logger.debug("Operation will NOT be retried - not a throttled request. Current attempt {}", (Object)this.currentAttemptCount, (Object)exception);
        }
        return Mono.just((Object)ShouldRetryResult.noRetry());
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
    }

    private Duration checkIfRetryNeeded(Exception exception) {
        Duration retryDelay = Duration.ZERO;
        CosmosException dce = Utils.as(exception, CosmosException.class);
        if (dce != null && Exceptions.isStatusCode(dce, 429)) {
            retryDelay = dce.getRetryAfterDuration();
            if (this.backoffDelayFactor > 1) {
                retryDelay = Duration.ofNanos(retryDelay.toNanos() * (long)this.backoffDelayFactor);
            }
            if (retryDelay.toMillis() < this.maxWaitTime.toMillis()) {
                this.cumulativeRetryDelay = retryDelay.plus(this.cumulativeRetryDelay);
                if (this.maxWaitTime.toMillis() >= this.cumulativeRetryDelay.toMillis()) {
                    if (retryDelay == Duration.ZERO) {
                        logger.trace("Received retryDelay of 0 with Http 429", (Throwable)exception);
                        retryDelay = DEFAULT_RETRY_IN_SECONDS;
                    }
                    return retryDelay;
                }
            }
        }
        return null;
    }
}

