// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
package com.azure.cosmos.models;

import com.azure.cosmos.CosmosDiagnostics;
import com.azure.cosmos.implementation.ResourceResponse;

import java.time.Duration;
import java.util.Map;

/**
 * The cosmos response
 * @param <T> the type of resource
 */
public class CosmosResponse<T> {
    private T properties;
    final ResourceResponse<?> resourceResponseWrapper;

    CosmosResponse(ResourceResponse<?> resourceResponse) {
        this.resourceResponseWrapper = resourceResponse;
    }

    CosmosResponse(T properties) {
        this.properties = properties;
        this.resourceResponseWrapper = null;
    }

    CosmosResponse(ResourceResponse<?> resourceResponse, T properties) {
        this.resourceResponseWrapper = resourceResponse;
        this.properties = properties;
    }

    /**
     * Gets properties.
     *
     * @return the properties
     */
    public T getProperties() {
        return properties;
    }

    CosmosResponse<T> setProperties(T resourceSettings) {
        this.properties = resourceSettings;
        return this;
    }

    /**
     * Gets the maximum size limit for this entity (in megabytes (MB) for server resources and in count for master
     * resources).
     *
     * @return the max resource quota.
     */
    public String getMaxResourceQuota() {
        if (resourceResponseWrapper == null) {
            return null;
        }

        return resourceResponseWrapper.getMaxResourceQuota();
    }

    /**
     * Gets the current size of this entity (in megabytes (MB) for server resources and in count for master resources)
     *
     * @return the current resource quota usage.
     */
    public String getCurrentResourceQuotaUsage() {
        if (resourceResponseWrapper == null) {
            return null;
        }

        return resourceResponseWrapper.getCurrentResourceQuotaUsage();
    }

    /**
     * Gets the Activity ID for the request.
     *
     * @return the activity getId.
     */
    public String getActivityId() {
        if (resourceResponseWrapper == null) {
            return null;
        }

        return resourceResponseWrapper.getActivityId();
    }

    /**
     * Gets the request charge as request units (RU) consumed by the operation.
     * <p>
     * For more information about the RU and factors that can impact the effective charges please visit
     * <a href="https://docs.microsoft.com/en-us/azure/cosmos-db/request-units">Request Units in Azure Cosmos DB</a>
     *
     * @return the request charge.
     */
    public double getRequestCharge() {
        if (resourceResponseWrapper == null) {
            return 0d;
        }

        return resourceResponseWrapper.getRequestCharge();
    }

    /**
     * Gets the HTTP status code associated with the response.
     *
     * @return the status code.
     */
    public int getStatusCode() {
        if (resourceResponseWrapper == null) {
            return 0;
        }

        return resourceResponseWrapper.getStatusCode();
    }

    /**
     * Gets the token used for managing client's consistency requirements.
     *
     * @return the session token.
     */
    public String getSessionToken() {
        if (resourceResponseWrapper == null) {
            return null;
        }

        return resourceResponseWrapper.getSessionToken();
    }

    /**
     * Gets the headers associated with the response.
     *
     * @return the response headers.
     */
    public Map<String, String> getResponseHeaders() {
        if (resourceResponseWrapper == null) {
            return null;
        }
        return resourceResponseWrapper.getResponseHeaders();
    }

    /**
     * Gets the diagnostics information for the current request to Azure Cosmos DB service.
     *
     * @return diagnostics information for the current request to Azure Cosmos DB service.
     */
    public CosmosDiagnostics getDiagnostics() {
        if (resourceResponseWrapper == null) {
            return null;
        }
        return resourceResponseWrapper.getDiagnostics();
    }

    /**
     * Gets the end-to-end request latency for the current request to Azure Cosmos DB service.
     *
     * @return end-to-end request latency for the current request to Azure Cosmos DB service.
     */
    public Duration getDuration() {
        if (resourceResponseWrapper == null) {
            return null;
        }

        return resourceResponseWrapper.getDuration();
    }
}
