/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation;

import com.azure.cosmos.CosmosException;
import com.azure.cosmos.CosmosRegionSwitchHint;
import com.azure.cosmos.SessionRetryOptions;
import com.azure.cosmos.implementation.Configs;
import com.azure.cosmos.implementation.IRetryPolicy;
import com.azure.cosmos.implementation.ImplementationBridgeHelpers;
import com.azure.cosmos.implementation.RetryContext;
import com.azure.cosmos.implementation.ShouldRetryResult;
import com.azure.cosmos.implementation.directconnectivity.TimeoutHelper;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class SessionTokenMismatchRetryPolicy
implements IRetryPolicy {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionTokenMismatchRetryPolicy.class);
    private static final int BACKOFF_MULTIPLIER = 2;
    private final Duration maximumBackoff;
    private final TimeoutHelper waitTimeTimeoutHelper = new TimeoutHelper(Duration.ofMillis(Configs.getSessionTokenMismatchDefaultWaitTimeInMs()));
    private final AtomicInteger retryCount;
    private Duration currentBackoff;
    private RetryContext retryContext;
    private final AtomicInteger maxRetryAttemptsInCurrentRegion;
    private final SessionRetryOptions sessionRetryOptions;

    public SessionTokenMismatchRetryPolicy(RetryContext retryContext, int waitTimeInMilliseconds, SessionRetryOptions sessionRetryOptions) {
        this.maximumBackoff = Duration.ofMillis(Configs.getSessionTokenMismatchMaximumBackoffTimeInMs());
        this.retryCount = new AtomicInteger();
        this.retryCount.set(0);
        this.currentBackoff = Duration.ofMillis(Configs.getSessionTokenMismatchInitialBackoffTimeInMs());
        this.maxRetryAttemptsInCurrentRegion = new AtomicInteger(Configs.getMaxRetriesInLocalRegionWhenRemoteRegionPreferred());
        this.retryContext = retryContext;
        this.sessionRetryOptions = sessionRetryOptions;
    }

    public SessionTokenMismatchRetryPolicy(RetryContext retryContext, SessionRetryOptions sessionRetryOptions) {
        this(retryContext, Configs.getSessionTokenMismatchDefaultWaitTimeInMs(), sessionRetryOptions);
    }

    @Override
    public Mono<ShouldRetryResult> shouldRetry(Exception e) {
        if (!(e instanceof CosmosException)) {
            return Mono.just((Object)ShouldRetryResult.noRetryOnNonRelatedException());
        }
        CosmosException cosmosException = (CosmosException)((Object)e);
        if (cosmosException.getStatusCode() != 404 || cosmosException.getSubStatusCode() != 1002) {
            LOGGER.debug("SessionTokenMismatchRetryPolicy not retrying because StatusCode or SubStatusCode not found.");
            return Mono.just((Object)ShouldRetryResult.noRetryOnNonRelatedException());
        }
        if (this.waitTimeTimeoutHelper.isElapsed()) {
            LOGGER.warn("SessionTokenMismatchRetryPolicy not retrying because it has exceeded the time limit. Retry count = {}", (Object)this.retryCount);
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        if (!this.shouldRetryLocally(this.sessionRetryOptions, this.retryCount.get())) {
            LOGGER.debug("SessionTokenMismatchRetryPolicy not retrying because it a retry attempt for the current region and fallback to a different region is preferred ");
            return Mono.just((Object)ShouldRetryResult.noRetry());
        }
        Duration effectiveBackoff = Duration.ZERO;
        if (this.retryCount.getAndIncrement() > 0) {
            effectiveBackoff = SessionTokenMismatchRetryPolicy.getEffectiveBackoff(this.currentBackoff, this.waitTimeTimeoutHelper.getRemainingTime());
            this.currentBackoff = SessionTokenMismatchRetryPolicy.getEffectiveBackoff(Duration.ofMillis(this.currentBackoff.toMillis() * 2L), this.maximumBackoff);
        }
        LOGGER.debug("SessionTokenMismatchRetryPolicy will retry. Retry count = {}.  Backoff time = {} ms", (Object)this.retryCount, (Object)effectiveBackoff.toMillis());
        return Mono.just((Object)ShouldRetryResult.retryAfter(effectiveBackoff));
    }

    @Override
    public RetryContext getRetryContext() {
        return this.retryContext;
    }

    private static Duration getEffectiveBackoff(Duration backoff, Duration remainingTime) {
        if (backoff.compareTo(remainingTime) > 0) {
            return remainingTime;
        }
        return backoff;
    }

    private boolean shouldRetryLocally(SessionRetryOptions sessionRetryOptions, int sessionTokenMismatchRetryAttempts) {
        if (sessionRetryOptions == null) {
            return true;
        }
        CosmosRegionSwitchHint regionSwitchHint = ImplementationBridgeHelpers.CosmosSessionRetryOptionsHelper.getCosmosSessionRetryOptionsAccessor().getRegionSwitchHint(sessionRetryOptions);
        if (regionSwitchHint == null || regionSwitchHint == CosmosRegionSwitchHint.LOCAL_REGION_PREFERRED) {
            return true;
        }
        return regionSwitchHint != CosmosRegionSwitchHint.REMOTE_REGION_PREFERRED || sessionTokenMismatchRetryAttempts != this.maxRetryAttemptsInCurrentRegion.get() - 1;
    }
}

