/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation;

import com.azure.cosmos.CosmosDiagnostics;
import com.azure.cosmos.CosmosException;
import com.azure.cosmos.ThrottlingRetryOptions;
import com.azure.cosmos.implementation.DiagnosticsClientContext;
import com.azure.cosmos.implementation.DocumentClientRetryPolicy;
import com.azure.cosmos.implementation.Exceptions;
import com.azure.cosmos.implementation.GlobalEndpointManager;
import com.azure.cosmos.implementation.IRetryPolicy;
import com.azure.cosmos.implementation.OperationType;
import com.azure.cosmos.implementation.ResourceThrottleRetryPolicy;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.Utils;
import com.azure.cosmos.implementation.apachecommons.collections.list.UnmodifiableList;
import com.azure.cosmos.implementation.directconnectivity.WebExceptionUtility;
import java.net.URI;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ClientRetryPolicy
extends DocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ClientRetryPolicy.class);
    static final int RetryIntervalInMS = 1000;
    static final int MaxRetryCount = 120;
    private static final int MaxServiceUnavailableRetryCount = 1;
    private static final int MAX_QUERY_PLAN_AND_ADDRESS_RETRY_COUNT = 2;
    private final DocumentClientRetryPolicy throttlingRetry;
    private final GlobalEndpointManager globalEndpointManager;
    private final boolean enableEndpointDiscovery;
    private int failoverRetryCount;
    private int sessionTokenRetryCount;
    private boolean isReadRequest;
    private boolean canUseMultipleWriteLocations;
    private URI locationEndpoint;
    private RetryContext retryContext;
    private CosmosDiagnostics cosmosDiagnostics;
    private AtomicInteger cnt = new AtomicInteger(0);
    private int serviceUnavailableRetryCount;
    private int queryPlanAddressRefreshCount;
    private RxDocumentServiceRequest request;

    public ClientRetryPolicy(DiagnosticsClientContext diagnosticsClientContext, GlobalEndpointManager globalEndpointManager, boolean enableEndpointDiscovery, ThrottlingRetryOptions throttlingRetryOptions) {
        this.throttlingRetry = new ResourceThrottleRetryPolicy(throttlingRetryOptions.getMaxRetryAttemptsOnThrottledRequests(), throttlingRetryOptions.getMaxRetryWaitTime());
        this.globalEndpointManager = globalEndpointManager;
        this.failoverRetryCount = 0;
        this.enableEndpointDiscovery = enableEndpointDiscovery;
        this.sessionTokenRetryCount = 0;
        this.canUseMultipleWriteLocations = false;
        this.cosmosDiagnostics = diagnosticsClientContext.createDiagnostics();
    }

    @Override
    public Mono<IRetryPolicy.ShouldRetryResult> shouldRetry(Exception e) {
        logger.debug("retry count {}, isReadRequest {}, canUseMultipleWriteLocations {}, due to failure:", new Object[]{this.cnt.incrementAndGet(), this.isReadRequest, this.canUseMultipleWriteLocations, e});
        if (this.locationEndpoint == null) {
            logger.error("locationEndpoint is null because ClientRetryPolicy::onBeforeRequest(.) is not invoked, probably request creation failed due to invalid options, serialization setting, etc.");
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.error(e));
        }
        this.retryContext = null;
        CosmosException clientException = Utils.as(e, CosmosException.class);
        if (clientException != null && clientException.getDiagnostics() != null) {
            this.cosmosDiagnostics = clientException.getDiagnostics();
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 3)) {
            logger.warn("Endpoint not writable. Will refresh cache and retry ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(false, true);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 1008) && this.isReadRequest) {
            logger.warn("Endpoint not available for reads. Will refresh cache and retry. ", (Throwable)e);
            return this.shouldRetryOnEndpointFailureAsync(true, false);
        }
        if (WebExceptionUtility.isNetworkFailure(e)) {
            if (clientException != null && Exceptions.isSubStatusCode(clientException, 10001)) {
                if (this.isReadRequest || WebExceptionUtility.isWebExceptionRetriable(e)) {
                    logger.warn("Gateway endpoint not reachable. Will refresh cache and retry. ", (Throwable)e);
                    return this.shouldRetryOnEndpointFailureAsync(this.isReadRequest, false);
                }
                return this.shouldNotRetryOnEndpointFailureAsync(this.isReadRequest, false);
            }
            if (clientException != null && WebExceptionUtility.isReadTimeoutException((Exception)((Object)clientException)) && Exceptions.isSubStatusCode(clientException, 10002)) {
                if (this.request.getOperationType() == OperationType.QueryPlan || this.request.isAddressRefresh()) {
                    return this.shouldRetryQueryPlanAndAddress();
                }
            } else {
                logger.warn("Backend endpoint not reachable. ", (Throwable)e);
                return this.shouldRetryOnBackendServiceUnavailableAsync(this.isReadRequest, WebExceptionUtility.isWebExceptionRetriable(e));
            }
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 404) && Exceptions.isSubStatusCode(clientException, 1002)) {
            return Mono.just((Object)this.shouldRetryOnSessionNotAvailable());
        }
        return this.throttlingRetry.shouldRetry(e);
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldRetryQueryPlanAndAddress() {
        if (this.queryPlanAddressRefreshCount++ > 2) {
            logger.warn("shouldRetryQueryPlanAndAddress() No more retrying on endpoint {}, operationType = {}, count = {}, isAddressRefresh = {}", new Object[]{this.locationEndpoint, this.request.getOperationType(), this.queryPlanAddressRefreshCount, this.request.isAddressRefresh()});
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        logger.warn("shouldRetryQueryPlanAndAddress() Retrying on endpoint {}, operationType = {}, count = {}, isAddressRefresh = {}", new Object[]{this.locationEndpoint, this.request.getOperationType(), this.queryPlanAddressRefreshCount, this.request.isAddressRefresh()});
        Duration retryDelay = Duration.ZERO;
        return Mono.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(retryDelay));
    }

    private IRetryPolicy.ShouldRetryResult shouldRetryOnSessionNotAvailable() {
        ++this.sessionTokenRetryCount;
        if (!this.enableEndpointDiscovery) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        if (this.canUseMultipleWriteLocations) {
            UnmodifiableList<URI> endpoints;
            UnmodifiableList<URI> unmodifiableList = endpoints = this.isReadRequest ? this.globalEndpointManager.getReadEndpoints() : this.globalEndpointManager.getWriteEndpoints();
            if (this.sessionTokenRetryCount > endpoints.size()) {
                return IRetryPolicy.ShouldRetryResult.noRetry();
            }
            this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, this.sessionTokenRetryCount > 1);
            return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
        }
        if (this.sessionTokenRetryCount > 1) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, false);
        return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh);
        Duration retryDelay = Duration.ZERO;
        if (!isReadRequest) {
            logger.debug("Failover happening. retryCount {}", (Object)this.failoverRetryCount);
            if (this.failoverRetryCount > 1) {
                retryDelay = Duration.ofMillis(1000L);
            }
        } else {
            retryDelay = Duration.ofMillis(1000L);
        }
        return refreshLocationCompletable.then(Mono.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(retryDelay)));
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldNotRetryOnEndpointFailureAsync(boolean isReadRequest, boolean forceRefresh) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        Mono<Void> refreshLocationCompletable = this.refreshLocation(isReadRequest, forceRefresh);
        return refreshLocationCompletable.then(Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry()));
    }

    private Mono<Void> refreshLocation(boolean isReadRequest, boolean forceRefresh) {
        ++this.failoverRetryCount;
        if (isReadRequest) {
            logger.warn("marking the endpoint {} as unavailable for read", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForRead(this.locationEndpoint);
        } else {
            logger.warn("marking the endpoint {} as unavailable for write", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForWrite(this.locationEndpoint);
        }
        this.retryContext = new RetryContext(this.failoverRetryCount, false);
        return this.globalEndpointManager.refreshLocationAsync(null, forceRefresh);
    }

    private Mono<IRetryPolicy.ShouldRetryResult> shouldRetryOnBackendServiceUnavailableAsync(boolean isReadRequest, boolean isWebExceptionRetriable) {
        if (!isReadRequest && !isWebExceptionRetriable) {
            logger.warn("shouldRetryOnBackendServiceUnavailableAsync() Not retrying on write with non retriable exception. Retry count = {}", (Object)this.serviceUnavailableRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        if (this.serviceUnavailableRetryCount++ > 1) {
            logger.warn("shouldRetryOnBackendServiceUnavailableAsync() Not retrying. Retry count = {}", (Object)this.serviceUnavailableRetryCount);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        if (!this.canUseMultipleWriteLocations && !isReadRequest) {
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        int availablePreferredLocations = this.globalEndpointManager.getPreferredLocationCount();
        if (availablePreferredLocations <= 1) {
            logger.warn("shouldRetryOnServiceUnavailable() Not retrying. No other regions available for the request. AvailablePreferredLocations = {}", (Object)availablePreferredLocations);
            return Mono.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        logger.warn("shouldRetryOnServiceUnavailable() Retrying. Received on endpoint {}, IsReadRequest = {}", (Object)this.locationEndpoint, (Object)isReadRequest);
        this.retryContext = new RetryContext(this.serviceUnavailableRetryCount, true);
        return Mono.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO));
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
        this.request = request;
        this.isReadRequest = request.isReadOnlyRequest();
        this.canUseMultipleWriteLocations = this.globalEndpointManager.canUseMultipleWriteLocations(request);
        if (request.requestContext != null) {
            request.requestContext.cosmosDiagnostics = this.cosmosDiagnostics;
        }
        if (request.requestContext != null) {
            request.requestContext.clearRouteToLocation();
        }
        if (this.retryContext != null) {
            request.requestContext.routeToLocation(this.retryContext.retryCount, this.retryContext.retryRequestOnPreferredLocations);
        }
        this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(request);
        if (request.requestContext != null) {
            request.requestContext.routeToLocation(this.locationEndpoint);
        }
    }

    private static class RetryContext {
        public int retryCount;
        public boolean retryRequestOnPreferredLocations;

        public RetryContext(int retryCount, boolean retryRequestOnPreferredLocations) {
            this.retryCount = retryCount;
            this.retryRequestOnPreferredLocations = retryRequestOnPreferredLocations;
        }
    }
}

