/*
 * Decompiled with CFR 0.152.
 */
package com.azure.data.schemaregistry;

import com.azure.core.http.rest.Response;
import com.azure.core.http.rest.SimpleResponse;
import com.azure.core.util.Context;
import com.azure.core.util.logging.ClientLogger;
import com.azure.data.schemaregistry.implementation.AzureSchemaRegistry;
import com.azure.data.schemaregistry.implementation.models.SchemaId;
import com.azure.data.schemaregistry.implementation.models.SchemasGetByIdHeaders;
import com.azure.data.schemaregistry.models.SchemaProperties;
import com.azure.data.schemaregistry.models.SerializationType;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.function.Function;
import reactor.core.publisher.Mono;

public final class SchemaRegistryAsyncClient {
    private final ClientLogger logger = new ClientLogger(SchemaRegistryAsyncClient.class);
    static final Charset SCHEMA_REGISTRY_SERVICE_ENCODING = StandardCharsets.UTF_8;
    static final int MAX_SCHEMA_MAP_SIZE_DEFAULT = 1000;
    static final int MAX_SCHEMA_MAP_SIZE_MINIMUM = 10;
    private final AzureSchemaRegistry restService;
    private final Integer maxSchemaMapSize;
    private final ConcurrentSkipListMap<String, Function<String, Object>> typeParserMap;
    private final Map<String, SchemaProperties> idCache;
    private final Map<String, SchemaProperties> schemaStringCache;

    SchemaRegistryAsyncClient(AzureSchemaRegistry restService, int maxSchemaMapSize, ConcurrentSkipListMap<String, Function<String, Object>> typeParserMap) {
        this.restService = restService;
        this.maxSchemaMapSize = maxSchemaMapSize;
        this.typeParserMap = typeParserMap;
        this.idCache = new ConcurrentHashMap<String, SchemaProperties>();
        this.schemaStringCache = new ConcurrentHashMap<String, SchemaProperties>();
    }

    SchemaRegistryAsyncClient(AzureSchemaRegistry restService, Map<String, SchemaProperties> idCache, Map<String, SchemaProperties> schemaStringCache, ConcurrentSkipListMap<String, Function<String, Object>> typeParserMap) {
        this.restService = restService;
        this.idCache = idCache;
        this.schemaStringCache = schemaStringCache;
        this.typeParserMap = typeParserMap;
        this.maxSchemaMapSize = 1000;
    }

    public Mono<SchemaProperties> registerSchema(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType) {
        if (this.schemaStringCache.containsKey(this.getSchemaStringCacheKey(schemaGroup, schemaName, schemaString))) {
            this.logger.verbose("Cache hit schema string. Group: '{}', name: '{}', schema type: '{}', payload: '{}'", new Object[]{schemaGroup, schemaName, serializationType, schemaString});
            return Mono.fromCallable(() -> this.schemaStringCache.get(this.getSchemaStringCacheKey(schemaGroup, schemaName, schemaString)));
        }
        return this.registerSchemaWithResponse(schemaGroup, schemaName, schemaString, serializationType).map(response -> (SchemaProperties)response.getValue());
    }

    public Mono<Response<SchemaProperties>> registerSchemaWithResponse(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType) {
        return this.registerSchemaWithResponse(schemaGroup, schemaName, schemaString, serializationType, Context.NONE);
    }

    Mono<Response<SchemaProperties>> registerSchemaWithResponse(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType, Context context) {
        this.logger.verbose("Registering schema. Group: '{}', name: '{}', serialization type: '{}', payload: '{}'", new Object[]{schemaGroup, schemaName, serializationType, schemaString});
        return this.restService.getSchemas().registerWithResponseAsync(schemaGroup, schemaName, com.azure.data.schemaregistry.implementation.models.SerializationType.AVRO, schemaString).handle((response, sink) -> {
            if (response == null) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new NullPointerException("Client returned null response")));
                return;
            }
            if (response.getStatusCode() == 400) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new IllegalStateException("Invalid schema registration attempted")));
                return;
            }
            SchemaId schemaId = response.getValue();
            SchemaProperties registered = new SchemaProperties(schemaId.getId(), serializationType, schemaName, schemaString.getBytes(SCHEMA_REGISTRY_SERVICE_ENCODING));
            this.resetIfNeeded();
            this.schemaStringCache.putIfAbsent(this.getSchemaStringCacheKey(schemaGroup, schemaName, schemaString), registered);
            this.logger.verbose("Cached schema string. Group: '{}', name: '{}'", new Object[]{schemaGroup, schemaName});
            SimpleResponse schemaRegistryObjectSimpleResponse = new SimpleResponse(response.getRequest(), response.getStatusCode(), response.getHeaders(), (Object)registered);
            sink.next((Object)schemaRegistryObjectSimpleResponse);
        });
    }

    public Mono<SchemaProperties> getSchema(String schemaId) {
        if (this.idCache.containsKey(schemaId)) {
            this.logger.verbose("Cache hit for schema id '{}'", new Object[]{schemaId});
            return Mono.fromCallable(() -> this.idCache.get(schemaId));
        }
        return this.getSchemaWithResponse(schemaId).map(Response::getValue);
    }

    public Mono<Response<SchemaProperties>> getSchemaWithResponse(String schemaId) {
        return this.getSchemaWithResponse(schemaId, Context.NONE);
    }

    Mono<Response<SchemaProperties>> getSchemaWithResponse(String schemaId, Context context) {
        Objects.requireNonNull(schemaId, "'schemaId' should not be null");
        return this.restService.getSchemas().getByIdWithResponseAsync(schemaId).handle((response, sink) -> {
            if (response == null) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new NullPointerException("Client returned null response")));
                return;
            }
            if (response.getStatusCode() == 404) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new IllegalStateException(String.format("Schema does not exist, id %s", schemaId))));
                return;
            }
            SerializationType serializationType = SerializationType.fromString(((SchemasGetByIdHeaders)response.getDeserializedHeaders()).getXSchemaType());
            SchemaProperties schemaObject = new SchemaProperties(schemaId, serializationType, null, response.getValue().getBytes(SCHEMA_REGISTRY_SERVICE_ENCODING));
            this.resetIfNeeded();
            this.idCache.putIfAbsent(schemaId, schemaObject);
            this.logger.verbose("Cached schema object. Path: '{}'", new Object[]{schemaId});
            SimpleResponse schemaRegistryObjectSimpleResponse = new SimpleResponse(response.getRequest(), response.getStatusCode(), response.getHeaders(), (Object)schemaObject);
            sink.next((Object)schemaRegistryObjectSimpleResponse);
        });
    }

    public Mono<String> getSchemaId(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType) {
        String schemaStringCacheKey = this.getSchemaStringCacheKey(schemaGroup, schemaName, schemaString);
        if (this.schemaStringCache.containsKey(schemaStringCacheKey)) {
            this.logger.verbose("Cache hit schema string. Group: '{}', name: '{}'", new Object[]{schemaGroup, schemaName});
            return Mono.fromCallable(() -> this.schemaStringCache.get(schemaStringCacheKey).getSchemaId());
        }
        return this.getSchemaIdWithResponse(schemaGroup, schemaName, schemaString, serializationType).map(response -> (String)response.getValue());
    }

    public Mono<Response<String>> getSchemaIdWithResponse(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType) {
        return this.getSchemaIdWithResponse(schemaGroup, schemaName, schemaString, serializationType, Context.NONE);
    }

    Mono<Response<String>> getSchemaIdWithResponse(String schemaGroup, String schemaName, String schemaString, SerializationType serializationType, Context context) {
        return this.restService.getSchemas().queryIdByContentWithResponseAsync(schemaGroup, schemaName, com.azure.data.schemaregistry.implementation.models.SerializationType.AVRO, schemaString).handle((response, sink) -> {
            if (response == null) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new NullPointerException("Client returned null response")));
                return;
            }
            if (response.getStatusCode() == 404) {
                sink.error((Throwable)this.logger.logExceptionAsError((RuntimeException)new IllegalStateException("Existing matching schema not found.")));
                return;
            }
            SchemaId schemaId = response.getValue();
            this.resetIfNeeded();
            this.schemaStringCache.putIfAbsent(this.getSchemaStringCacheKey(schemaGroup, schemaName, schemaString), new SchemaProperties(schemaId.getId(), serializationType, schemaName, schemaString.getBytes(SCHEMA_REGISTRY_SERVICE_ENCODING)));
            this.logger.verbose("Cached schema string. Group: '{}', name: '{}'", new Object[]{schemaGroup, schemaName});
            SimpleResponse schemaIdResponse = new SimpleResponse(response.getRequest(), response.getStatusCode(), response.getHeaders(), (Object)schemaId.getId());
            sink.next((Object)schemaIdResponse);
        });
    }

    void clearCache() {
        this.idCache.clear();
        this.schemaStringCache.clear();
        this.typeParserMap.clear();
    }

    private void resetIfNeeded() {
        if (this.idCache.size() > this.maxSchemaMapSize) {
            this.idCache.clear();
            this.logger.verbose("Cleared schema ID cache.");
        }
        if (this.schemaStringCache.size() > this.maxSchemaMapSize) {
            this.schemaStringCache.clear();
            this.logger.verbose("Cleared schema string cache.");
        }
    }

    private String getSchemaStringCacheKey(String schemaGroup, String schemaName, String schemaString) {
        return schemaGroup + schemaName + schemaString;
    }
}

