/*
 * Decompiled with CFR 0.152.
 */
package com.azure.messaging.servicebus.implementation;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenCredential;
import com.azure.core.credential.TokenRequestContext;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.messaging.servicebus.implementation.ServiceBusConstants;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Base64;
import java.util.Locale;
import java.util.Objects;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import reactor.core.publisher.Mono;

public class ServiceBusSharedKeyCredential
implements TokenCredential {
    private static final String SHARED_ACCESS_SIGNATURE_FORMAT = "SharedAccessSignature sr=%s&sig=%s&se=%s&skn=%s";
    private static final String HASH_ALGORITHM = "HMACSHA256";
    private final ClientLogger logger = new ClientLogger(ServiceBusSharedKeyCredential.class);
    private final String policyName;
    private final Mac hmac;
    private final Duration tokenValidity;

    public ServiceBusSharedKeyCredential(String policyName, String sharedAccessKey) {
        this(policyName, sharedAccessKey, ServiceBusConstants.TOKEN_VALIDITY);
    }

    public ServiceBusSharedKeyCredential(String policyName, String sharedAccessKey, Duration tokenValidity) {
        Objects.requireNonNull(sharedAccessKey, "'sharedAccessKey' cannot be null.");
        this.policyName = Objects.requireNonNull(policyName, "'sharedAccessKey' cannot be null.");
        this.tokenValidity = Objects.requireNonNull(tokenValidity, "'tokenValidity' cannot be null.");
        if (policyName.isEmpty()) {
            throw new IllegalArgumentException("'policyName' cannot be an empty string.");
        }
        if (sharedAccessKey.isEmpty()) {
            throw new IllegalArgumentException("'sharedAccessKey' cannot be an empty string.");
        }
        if (tokenValidity.isZero() || tokenValidity.isNegative()) {
            throw new IllegalArgumentException("'tokenTimeToLive' has to positive and in the order-of seconds");
        }
        try {
            this.hmac = Mac.getInstance(HASH_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw this.logger.logExceptionAsError((RuntimeException)new UnsupportedOperationException(String.format("Unable to create hashing algorithm '%s'", HASH_ALGORITHM), e));
        }
        byte[] sasKeyBytes = sharedAccessKey.getBytes(StandardCharsets.UTF_8);
        SecretKeySpec finalKey = new SecretKeySpec(sasKeyBytes, HASH_ALGORITHM);
        try {
            this.hmac.init(finalKey);
        }
        catch (InvalidKeyException e) {
            throw this.logger.logExceptionAsError((RuntimeException)new IllegalArgumentException("'sharedAccessKey' is an invalid value for the hashing algorithm.", e));
        }
    }

    public Mono<AccessToken> getToken(TokenRequestContext request) {
        if (request.getScopes().size() != 1) {
            throw this.logger.logExceptionAsError((RuntimeException)new IllegalArgumentException("'scopes' should only contain a single argument that is the token audience or resource name."));
        }
        return Mono.fromCallable(() -> this.generateSharedAccessSignature((String)request.getScopes().get(0)));
    }

    private AccessToken generateSharedAccessSignature(String resource) throws UnsupportedEncodingException {
        if (CoreUtils.isNullOrEmpty((CharSequence)resource)) {
            throw this.logger.logExceptionAsError((RuntimeException)new IllegalArgumentException("resource cannot be empty"));
        }
        String utf8Encoding = StandardCharsets.UTF_8.name();
        OffsetDateTime expiresOn = OffsetDateTime.now(ZoneOffset.UTC).plus(this.tokenValidity);
        String expiresOnEpochSeconds = Long.toString(expiresOn.toEpochSecond());
        String audienceUri = URLEncoder.encode(resource, utf8Encoding);
        String secretToSign = audienceUri + "\n" + expiresOnEpochSeconds;
        byte[] signatureBytes = this.hmac.doFinal(secretToSign.getBytes(utf8Encoding));
        String signature = Base64.getEncoder().encodeToString(signatureBytes);
        String token = String.format(Locale.US, SHARED_ACCESS_SIGNATURE_FORMAT, audienceUri, URLEncoder.encode(signature, utf8Encoding), URLEncoder.encode(expiresOnEpochSeconds, utf8Encoding), URLEncoder.encode(this.policyName, utf8Encoding));
        return new AccessToken(token, expiresOn);
    }
}

