// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.mixedreality.remoterendering.models;

import com.azure.core.annotation.Immutable;

import java.time.OffsetDateTime;

/** Holds properties of a conversion. */
@Immutable
public final class AssetConversion {
    private final String id;
    private final AssetConversionOptions options;
    private final String outputAssetUrl;
    private final RemoteRenderingServiceError error;
    private final AssetConversionStatus conversionStatus;
    private final OffsetDateTime creationTime;

    /**
     * Constructs a new AssetConversion object.
     *
     * @param id The id of the conversion supplied when the conversion was created.
     * @param options Options for where to retrieve input files from and where to write output files.
     * @param outputAssetUrl URL of the asset generated by the conversion process. Only present when the status of
     *                       the conversion is 'Succeeded'.
     * @param error The error object containing details about the conversion failure.
     * @param conversionStatus The status of the conversion. Terminal states are CANCELLED, FAILED, or SUCCEEDED.
     * @param creationTime The time when the conversion was created. Date and time in ISO 8601 format.
     */
    public AssetConversion(String id,
                           AssetConversionOptions options,
                           String outputAssetUrl,
                           RemoteRenderingServiceError error,
                           AssetConversionStatus conversionStatus,
                           OffsetDateTime creationTime) {
        this.id = id;
        this.options = options;
        this.outputAssetUrl = outputAssetUrl;
        this.error = error;
        this.conversionStatus = conversionStatus;
        this.creationTime = creationTime;
    }


    /**
     * Get the id property: The id of the conversion supplied when the conversion was created.
     *
     * @return the id value.
     */
    public String getId() {
        return this.id;
    }

    /**
     * Get the conversion options: Options for where to retrieve input files from and where to write output files.
     * Supplied when creating the conversion.
     *
     * @return the conversion options value.
     */
    public AssetConversionOptions getOptions() {
        return this.options;
    }

    /**
     * Get the outputAssetUrl property: URL of the asset generated by the conversion process. Only present when the status of
     * the conversion is 'Succeeded'.
     *
     * @return the outputAssetUrl value.
     */
    public String getOutputAssetUrl() {
        return this.outputAssetUrl;
    }

    /**
     * Get the error property: The error object containing details about the conversion failure.
     *
     * @return the error value.
     */
    public RemoteRenderingServiceError getError() {
        return this.error;
    }

    /**
     * Get the status property: The status of the conversion. Terminal states are CANCELLED, FAILED, or SUCCEEDED.
     *
     * @return the status value.
     */
    public AssetConversionStatus getStatus() {
        return this.conversionStatus;
    }

    /**
     * Get the creationTime property: The time when the conversion was created. Date and time in ISO 8601 format.
     *
     * @return the creationTime value.
     */
    public OffsetDateTime getCreationTime() {
        return this.creationTime;
    }
}
