/*
 * Decompiled with CFR 0.152.
 */
package com.azure.monitor.opentelemetry.autoconfigure.implementation.localstorage;

import com.azure.monitor.opentelemetry.autoconfigure.implementation.localstorage.FileUtil;
import com.azure.monitor.opentelemetry.autoconfigure.implementation.localstorage.LocalFileCache;
import com.azure.monitor.opentelemetry.autoconfigure.implementation.localstorage.LocalStorageStats;
import com.azure.monitor.opentelemetry.autoconfigure.implementation.logging.OperationLogger;
import com.azure.monitor.opentelemetry.autoconfigure.implementation.utils.AzureMonitorMsgId;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import reactor.util.annotation.Nullable;

class LocalFileLoader {
    private static final String TEMPORARY_FILE_EXTENSION = ".tmp";
    private final LocalFileCache localFileCache;
    private final File telemetryFolder;
    private final LocalStorageStats stats;
    private final OperationLogger operationLogger;
    private final OperationLogger updateOperationLogger;

    LocalFileLoader(LocalFileCache localFileCache, File telemetryFolder, LocalStorageStats stats, boolean suppressWarnings) {
        this.localFileCache = localFileCache;
        this.telemetryFolder = telemetryFolder;
        this.stats = stats;
        this.operationLogger = suppressWarnings ? OperationLogger.NOOP : new OperationLogger(LocalFileLoader.class, "Loading telemetry from disk");
        this.updateOperationLogger = suppressWarnings ? OperationLogger.NOOP : new OperationLogger(LocalFileLoader.class, "Updating local telemetry on disk");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    PersistedFile loadTelemetriesFromDisk() {
        File tempFile;
        File fileToBeLoaded = this.localFileCache.poll();
        if (fileToBeLoaded == null) {
            return null;
        }
        try {
            if (!fileToBeLoaded.exists()) {
                return null;
            }
            tempFile = new File(this.telemetryFolder, FileUtil.getBaseName(fileToBeLoaded) + TEMPORARY_FILE_EXTENSION);
            FileUtil.moveFile(fileToBeLoaded, tempFile);
        }
        catch (IOException e) {
            this.operationLogger.recordFailure("Error renaming file: " + fileToBeLoaded.getAbsolutePath(), AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
            this.stats.incrementReadFailureCount();
            return null;
        }
        if (tempFile.length() <= 36L) {
            this.deleteFile(tempFile);
            return null;
        }
        try (DataInputStream dataInputStream = new DataInputStream(Files.newInputStream(tempFile.toPath(), new OpenOption[0]));){
            int version = dataInputStream.readInt();
            if (version == 1) {
                String connectionString = dataInputStream.readUTF();
                int numBytes = dataInputStream.readInt();
                byte[] telemetryBytes = new byte[numBytes];
                dataInputStream.readFully(telemetryBytes);
                this.operationLogger.recordSuccess();
                PersistedFile persistedFile = new PersistedFile(tempFile, connectionString, ByteBuffer.wrap(telemetryBytes));
                return persistedFile;
            }
        }
        catch (IOException e) {
            this.operationLogger.recordFailure("Error reading file: " + tempFile.getAbsolutePath(), e, AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
            this.stats.incrementReadFailureCount();
            return null;
        }
        this.deleteFile(tempFile);
        return null;
    }

    private void deleteFile(File tempFile) {
        if (!FileUtil.deleteFileWithRetries(tempFile)) {
            this.operationLogger.recordFailure("Unable to delete file: " + tempFile.getAbsolutePath(), AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
        }
    }

    void updateProcessedFileStatus(boolean successOrNonRetryableError, File file) {
        if (!file.exists()) {
            this.updateOperationLogger.recordFailure("File no longer exists: " + file.getAbsolutePath(), AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
            return;
        }
        if (successOrNonRetryableError) {
            if (!FileUtil.deleteFileWithRetries(file)) {
                this.updateOperationLogger.recordFailure("Unable to delete file: " + file.getAbsolutePath(), AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
            } else {
                this.updateOperationLogger.recordSuccess();
            }
        } else {
            File sourceFile = new File(this.telemetryFolder, FileUtil.getBaseName(file) + ".trn");
            try {
                FileUtil.moveFile(file, sourceFile);
            }
            catch (IOException e) {
                this.updateOperationLogger.recordFailure("Error renaming file: " + file.getAbsolutePath(), e, AzureMonitorMsgId.DISK_PERSISTENCE_LOADER_ERROR);
                return;
            }
            this.updateOperationLogger.recordSuccess();
            this.localFileCache.addPersistedFile(sourceFile);
        }
    }

    static class PersistedFile {
        final File file;
        final String connectionString;
        final ByteBuffer rawBytes;

        PersistedFile(File file, String connectionString, ByteBuffer byteBuffer) {
            if (connectionString == null) {
                throw new IllegalArgumentException("instrumentation key can not be null.");
            }
            this.file = file;
            this.connectionString = connectionString;
            this.rawBytes = byteBuffer;
        }
    }
}

