/*
 * Decompiled with CFR 0.152.
 */
package com.azure.monitor.opentelemetry.exporter.implementation;

import com.azure.core.util.logging.ClientLogger;
import com.azure.monitor.opentelemetry.exporter.implementation.AiSemanticAttributes;
import com.azure.monitor.opentelemetry.exporter.implementation.MappingsBuilder;
import com.azure.monitor.opentelemetry.exporter.implementation.SemanticAttributes;
import com.azure.monitor.opentelemetry.exporter.implementation.SpanDataMapper;
import com.azure.monitor.opentelemetry.exporter.implementation.builders.AbstractTelemetryBuilder;
import com.azure.monitor.opentelemetry.exporter.implementation.builders.MetricPointBuilder;
import com.azure.monitor.opentelemetry.exporter.implementation.builders.MetricTelemetryBuilder;
import com.azure.monitor.opentelemetry.exporter.implementation.configuration.ConnectionString;
import com.azure.monitor.opentelemetry.exporter.implementation.models.ContextTagKeys;
import com.azure.monitor.opentelemetry.exporter.implementation.models.TelemetryItem;
import com.azure.monitor.opentelemetry.exporter.implementation.preaggregatedmetrics.DependencyExtractor;
import com.azure.monitor.opentelemetry.exporter.implementation.preaggregatedmetrics.RequestExtractor;
import com.azure.monitor.opentelemetry.exporter.implementation.utils.FormattedTime;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.internal.Utils;
import io.opentelemetry.sdk.metrics.data.DoublePointData;
import io.opentelemetry.sdk.metrics.data.HistogramPointData;
import io.opentelemetry.sdk.metrics.data.LongPointData;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.data.MetricDataType;
import io.opentelemetry.sdk.metrics.data.PointData;
import io.opentelemetry.sdk.resources.Resource;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import reactor.util.annotation.Nullable;

public class MetricDataMapper {
    private static final ClientLogger logger = new ClientLogger(MetricDataMapper.class);
    private static final Set<String> OTEL_UNSTABLE_METRICS_TO_EXCLUDE = new HashSet<String>();
    private static final String OTEL_INSTRUMENTATION_NAME_PREFIX = "io.opentelemetry";
    private static final Set<String> OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES = new HashSet<String>(4);
    public static final AttributeKey<String> APPLICATIONINSIGHTS_INTERNAL_METRIC_NAME = AttributeKey.stringKey((String)"applicationinsights.internal.metric_name");
    private final BiConsumer<AbstractTelemetryBuilder, Resource> telemetryInitializer;
    private final boolean captureHttpServer4xxAsError;

    public MetricDataMapper(BiConsumer<AbstractTelemetryBuilder, Resource> telemetryInitializer, boolean captureHttpServer4xxAsError) {
        this.telemetryInitializer = telemetryInitializer;
        this.captureHttpServer4xxAsError = captureHttpServer4xxAsError;
    }

    public void map(MetricData metricData, Consumer<TelemetryItem> consumer) {
        MetricDataType type = metricData.getType();
        if (type == MetricDataType.DOUBLE_SUM || type == MetricDataType.DOUBLE_GAUGE || type == MetricDataType.LONG_SUM || type == MetricDataType.LONG_GAUGE || type == MetricDataType.HISTOGRAM) {
            boolean isPreAggregatedStandardMetric = OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES.contains(metricData.getName());
            if (isPreAggregatedStandardMetric) {
                List<TelemetryItem> preAggregatedStandardMetrics = this.convertOtelMetricToAzureMonitorMetric(metricData, true);
                preAggregatedStandardMetrics.forEach(consumer::accept);
            }
            if (OTEL_UNSTABLE_METRICS_TO_EXCLUDE.contains(metricData.getName()) && metricData.getInstrumentationScopeInfo().getName().startsWith(OTEL_INSTRUMENTATION_NAME_PREFIX)) {
                return;
            }
            List<TelemetryItem> stableOtelMetrics = this.convertOtelMetricToAzureMonitorMetric(metricData, false);
            stableOtelMetrics.forEach(consumer::accept);
        } else {
            logger.warning("metric data type {} is not supported yet.", new Object[]{metricData.getType()});
        }
    }

    private List<TelemetryItem> convertOtelMetricToAzureMonitorMetric(MetricData metricData, boolean isPreAggregatedStandardMetric) {
        ArrayList<TelemetryItem> telemetryItems = new ArrayList<TelemetryItem>();
        for (PointData pointData : metricData.getData().getPoints()) {
            MetricTelemetryBuilder builder = MetricTelemetryBuilder.create();
            this.telemetryInitializer.accept(builder, metricData.getResource());
            builder.setTime(FormattedTime.offSetDateTimeFromEpochNanos(pointData.getEpochNanos()));
            MetricDataMapper.updateMetricPointBuilder(builder, metricData, pointData, this.captureHttpServer4xxAsError, isPreAggregatedStandardMetric);
            telemetryItems.add(builder.build());
        }
        return telemetryItems;
    }

    public static void updateMetricPointBuilder(MetricTelemetryBuilder metricTelemetryBuilder, MetricData metricData, PointData pointData, boolean captureHttpServer4xxAsError, boolean isPreAggregatedStandardMetric) {
        double pointDataValue;
        Utils.checkArgument((metricData != null ? 1 : 0) != 0, (String)"MetricData cannot be null.");
        MetricPointBuilder pointBuilder = new MetricPointBuilder();
        MetricDataType type = metricData.getType();
        switch (type) {
            case LONG_SUM: 
            case LONG_GAUGE: {
                pointDataValue = ((LongPointData)pointData).getValue();
                break;
            }
            case DOUBLE_SUM: 
            case DOUBLE_GAUGE: {
                pointDataValue = ((DoublePointData)pointData).getValue();
                break;
            }
            case HISTOGRAM: {
                long histogramCount = ((HistogramPointData)pointData).getCount();
                if (histogramCount <= Integer.MAX_VALUE && histogramCount >= Integer.MIN_VALUE) {
                    pointBuilder.setCount((int)histogramCount);
                }
                HistogramPointData histogramPointData = (HistogramPointData)pointData;
                double min = histogramPointData.getMin();
                double max = histogramPointData.getMax();
                if (MetricDataMapper.shouldConvertToMilliseconds(metricData.getName(), isPreAggregatedStandardMetric)) {
                    min *= 1000.0;
                    max *= 1000.0;
                }
                pointDataValue = histogramPointData.getSum();
                pointBuilder.setMin(min);
                pointBuilder.setMax(max);
                break;
            }
            default: {
                throw new IllegalArgumentException("metric data type '" + type + "' is not supported yet");
            }
        }
        if (MetricDataMapper.shouldConvertToMilliseconds(metricData.getName(), isPreAggregatedStandardMetric)) {
            pointDataValue *= 1000.0;
        }
        pointBuilder.setValue(pointDataValue);
        String metricName = (String)pointData.getAttributes().get(APPLICATIONINSIGHTS_INTERNAL_METRIC_NAME);
        if (metricName != null) {
            pointBuilder.setName(metricName);
        } else {
            pointBuilder.setName(metricData.getName());
        }
        metricTelemetryBuilder.setMetricPoint(pointBuilder);
        Attributes attributes = pointData.getAttributes();
        if (isPreAggregatedStandardMetric) {
            Long statusCode = SpanDataMapper.getStableOrOldAttribute(attributes, SemanticAttributes.HTTP_RESPONSE_STATUS_CODE, SemanticAttributes.HTTP_STATUS_CODE);
            boolean success = MetricDataMapper.isSuccess(metricData.getName(), statusCode, captureHttpServer4xxAsError);
            Boolean isSynthetic = (Boolean)attributes.get(AiSemanticAttributes.IS_SYNTHETIC);
            attributes.forEach((key, value) -> MetricDataMapper.applyConnectionStringAndRoleNameOverrides(metricTelemetryBuilder, value, key.getKey()));
            if (MetricDataMapper.isServer(metricData.getName())) {
                RequestExtractor.extract(metricTelemetryBuilder, statusCode, success, isSynthetic);
            } else if (MetricDataMapper.isClient(metricData.getName())) {
                int defaultPort;
                String dependencyType;
                if (metricData.getName().startsWith("http")) {
                    dependencyType = "Http";
                    defaultPort = MetricDataMapper.getDefaultPortForHttpScheme(SpanDataMapper.getStableOrOldAttribute(attributes, SemanticAttributes.URL_SCHEME, SemanticAttributes.HTTP_SCHEME));
                } else {
                    dependencyType = (String)attributes.get(SemanticAttributes.RPC_SYSTEM);
                    if (dependencyType == null) {
                        dependencyType = "Unknown";
                    }
                    defaultPort = Integer.MAX_VALUE;
                }
                String target = SpanDataMapper.getTargetOrDefault(attributes, defaultPort, dependencyType);
                DependencyExtractor.extract(metricTelemetryBuilder, statusCode, success, dependencyType, target, isSynthetic);
            }
        } else {
            MappingsBuilder mappingsBuilder = new MappingsBuilder(MappingsBuilder.MappingType.METRIC);
            mappingsBuilder.build().map(attributes, metricTelemetryBuilder);
        }
    }

    private static boolean shouldConvertToMilliseconds(String metricName, boolean isPreAggregatedStandardMetric) {
        return isPreAggregatedStandardMetric && (metricName.equals("http.server.request.duration") || metricName.equals("http.client.request.duration"));
    }

    private static boolean applyConnectionStringAndRoleNameOverrides(AbstractTelemetryBuilder telemetryBuilder, Object value, String key) {
        if (key.equals(AiSemanticAttributes.INTERNAL_CONNECTION_STRING.getKey()) && value instanceof String) {
            telemetryBuilder.setConnectionString(ConnectionString.parse((String)value));
            return true;
        }
        if (key.equals(AiSemanticAttributes.INTERNAL_ROLE_NAME.getKey()) && value instanceof String) {
            telemetryBuilder.addTag(ContextTagKeys.AI_CLOUD_ROLE.toString(), (String)value);
            return true;
        }
        return false;
    }

    private static int getDefaultPortForHttpScheme(@Nullable String httpScheme) {
        if (httpScheme == null) {
            return Integer.MAX_VALUE;
        }
        if (httpScheme.equals("https")) {
            return 443;
        }
        if (httpScheme.equals("http")) {
            return 80;
        }
        return Integer.MAX_VALUE;
    }

    private static boolean isSuccess(String metricName, Long statusCode, boolean captureHttpServer4xxAsError) {
        if (statusCode == null) {
            return true;
        }
        if (MetricDataMapper.isClient(metricName)) {
            return statusCode < 400L;
        }
        if (MetricDataMapper.isServer(metricName)) {
            if (captureHttpServer4xxAsError) {
                return statusCode < 400L;
            }
            return statusCode < 500L;
        }
        return false;
    }

    private static boolean isClient(String metricName) {
        return metricName.contains(".client.");
    }

    private static boolean isServer(String metricName) {
        return metricName.contains(".server.");
    }

    static {
        OTEL_UNSTABLE_METRICS_TO_EXCLUDE.add("rpc.client.duration");
        OTEL_UNSTABLE_METRICS_TO_EXCLUDE.add("rpc.server.duration");
        OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES.add("http.server.request.duration");
        OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES.add("http.client.request.duration");
        OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES.add("rpc.client.duration");
        OTEL_PRE_AGGREGATED_STANDARD_METRIC_NAMES.add("rpc.server.duration");
    }
}

