/*
 * Decompiled with CFR 0.152.
 */
package com.azure.monitor.opentelemetry.exporter.implementation.localstorage;

import com.azure.monitor.opentelemetry.exporter.implementation.localstorage.FileUtil;
import com.azure.monitor.opentelemetry.exporter.implementation.localstorage.LocalFileCache;
import com.azure.monitor.opentelemetry.exporter.implementation.localstorage.LocalStorageStats;
import com.azure.monitor.opentelemetry.exporter.implementation.logging.OperationLogger;
import com.azure.monitor.opentelemetry.exporter.implementation.utils.AzureMonitorMsgId;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.List;

final class LocalFileWriter {
    private static final String PERMANENT_FILE_EXTENSION = ".trn";
    private final long diskPersistenceMaxSizeBytes;
    private final LocalFileCache localFileCache;
    private final File telemetryFolder;
    private final LocalStorageStats stats;
    private final OperationLogger operationLogger;

    LocalFileWriter(int diskPersistenceMaxSizeMb, LocalFileCache localFileCache, File telemetryFolder, LocalStorageStats stats, boolean suppressWarnings) {
        this.telemetryFolder = telemetryFolder;
        this.localFileCache = localFileCache;
        this.stats = stats;
        this.diskPersistenceMaxSizeBytes = (long)diskPersistenceMaxSizeMb * 1024L * 1024L;
        this.operationLogger = suppressWarnings ? OperationLogger.NOOP : new OperationLogger(LocalFileWriter.class, "Writing telemetry to disk (telemetry is discarded on failure)");
    }

    @SuppressFBWarnings(value={"SECPTI"}, justification="The constructed file path cannot be controlled by an end user of the instrumented application")
    void writeToDisk(String connectionString, List<ByteBuffer> buffers) {
        File permanentFile;
        File tempFile;
        long size = LocalFileWriter.getTotalSizeOfPersistedFiles(this.telemetryFolder);
        if (size >= this.diskPersistenceMaxSizeBytes) {
            this.operationLogger.recordFailure("Local persistent storage capacity has been reached. It's currently at (" + size / 1024L + "KB). Telemetry will be lost.", AzureMonitorMsgId.DISK_PERSISTENCE_WRITER_ERROR);
            this.stats.incrementWriteFailureCount();
            return;
        }
        try {
            tempFile = LocalFileWriter.createTempFile(this.telemetryFolder);
        }
        catch (IOException e) {
            this.operationLogger.recordFailure("Error creating file in directory: " + this.telemetryFolder.getAbsolutePath(), e, AzureMonitorMsgId.DISK_PERSISTENCE_WRITER_ERROR);
            this.stats.incrementWriteFailureCount();
            return;
        }
        try {
            LocalFileWriter.write(tempFile, connectionString, buffers);
        }
        catch (IOException e) {
            this.operationLogger.recordFailure("Error writing file: " + tempFile.getAbsolutePath(), e, AzureMonitorMsgId.DISK_PERSISTENCE_WRITER_ERROR);
            this.stats.incrementWriteFailureCount();
            return;
        }
        try {
            permanentFile = new File(this.telemetryFolder, FileUtil.getBaseName(tempFile) + PERMANENT_FILE_EXTENSION);
            FileUtil.moveFile(tempFile, permanentFile);
        }
        catch (IOException e) {
            this.operationLogger.recordFailure("Error renaming file: " + tempFile.getAbsolutePath(), e, AzureMonitorMsgId.DISK_PERSISTENCE_WRITER_ERROR);
            this.stats.incrementWriteFailureCount();
            return;
        }
        this.localFileCache.addPersistedFile(permanentFile);
        this.operationLogger.recordSuccess();
    }

    private static void write(File file, String connectionString, List<ByteBuffer> buffers) throws IOException {
        try (FileOutputStream fileOut = new FileOutputStream(file);
             DataOutputStream dataOut = new DataOutputStream(fileOut);){
            dataOut.writeInt(1);
            dataOut.writeUTF(connectionString);
            int numBytes = buffers.stream().mapToInt(Buffer::remaining).sum();
            dataOut.writeInt(numBytes);
            FileChannel fileChannel = fileOut.getChannel();
            for (ByteBuffer byteBuffer : buffers) {
                fileChannel.write(byteBuffer);
            }
        }
    }

    @SuppressFBWarnings(value={"SECPTI"}, justification="The constructed file path cannot be controlled by an end user of the instrumented application")
    private static File createTempFile(File telemetryFolder) throws IOException {
        String prefix = System.currentTimeMillis() + "-";
        return File.createTempFile(prefix, null, telemetryFolder);
    }

    private static long getTotalSizeOfPersistedFiles(File telemetryFolder) {
        if (!telemetryFolder.exists()) {
            return 0L;
        }
        long sum = 0L;
        List<File> files = FileUtil.listTrnFiles(telemetryFolder);
        for (File file : files) {
            sum += file.length();
        }
        return sum;
    }
}

