// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.monitor.query;

import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceClient;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.Response;
import com.azure.core.util.Context;
import com.azure.monitor.query.models.MetricDefinition;
import com.azure.monitor.query.models.MetricNamespace;
import com.azure.monitor.query.models.MetricsQueryOptions;
import com.azure.monitor.query.models.MetricsQueryResult;

import java.time.OffsetDateTime;
import java.util.List;

/**
 * The synchronous client for querying Azure Monitor metrics.
 *
 * <p><strong>Instantiating a synchronous Metrics query Client</strong></p>
 *
 * <!-- src_embed com.azure.monitor.query.MetricsQueryClient.instantiation -->
 * <pre>
 * MetricsQueryClient metricsQueryClient = new MetricsQueryClientBuilder&#40;&#41;
 *         .credential&#40;tokenCredential&#41;
 *         .buildClient&#40;&#41;;
 * </pre>
 * <!-- end com.azure.monitor.query.MetricsQueryClient.instantiation -->
 */
@ServiceClient(builder = MetricsQueryClientBuilder.class)
public final class MetricsQueryClient {
    private final MetricsQueryAsyncClient asyncClient;

    MetricsQueryClient(MetricsQueryAsyncClient asyncClient) {
        this.asyncClient = asyncClient;
    }

    /**
     * Returns all the Azure Monitor metrics requested for the resource.
     *
     * <p><strong>Query metrics for an Azure resource</strong></p>
     *
     * <!-- src_embed com.azure.monitor.query.MetricsQueryClient.query#String-List -->
     * <pre>
     * MetricsQueryResult response = metricsQueryClient.queryResource&#40;&quot;&#123;resource-id&#125;&quot;,
     *         Arrays.asList&#40;&quot;&#123;metric-1&#125;&quot;, &quot;&#123;metric-2&#125;&quot;&#41;&#41;;
     * for &#40;MetricResult metricResult : response.getMetrics&#40;&#41;&#41; &#123;
     *     System.out.println&#40;&quot;Metric name &quot; + metricResult.getMetricName&#40;&#41;&#41;;
     *
     *     metricResult.getTimeSeries&#40;&#41;.stream&#40;&#41;
     *             .flatMap&#40;timeSeriesElement -&gt; timeSeriesElement.getValues&#40;&#41;.stream&#40;&#41;&#41;
     *             .forEach&#40;metricValue -&gt;
     *                     System.out.println&#40;&quot;Time stamp: &quot; + metricValue.getTimeStamp&#40;&#41; + &quot;; Total:  &quot;
     *                             + metricValue.getTotal&#40;&#41;&#41;&#41;;
     * &#125;
     *
     * </pre>
     * <!-- end com.azure.monitor.query.MetricsQueryClient.query#String-List -->
     *
     * @param resourceUri The resource URI for which the metrics is requested.
     * @param metricsNames The names of the metrics to query.
     * @return A time-series metrics result for the requested metric names.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public MetricsQueryResult queryResource(String resourceUri, List<String> metricsNames) {
        return queryResourceWithResponse(resourceUri, metricsNames, new MetricsQueryOptions(), Context.NONE).getValue();
    }

    /**
     * Returns all the Azure Monitor metrics requested for the resource.
     * @param resourceUri The resource URI for which the metrics is requested.
     * @param metricsNames The names of the metrics to query.
     * @param options Options to filter the query.
     * @param context Additional context that is passed through the Http pipeline during the service call. If no
     * additional context is required, pass {@link Context#NONE} instead.
     * @return A time-series metrics result for the requested metric names.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<MetricsQueryResult> queryResourceWithResponse(String resourceUri, List<String> metricsNames,
                                                                  MetricsQueryOptions options, Context context) {
        return asyncClient.queryResourceWithResponse(resourceUri, metricsNames, options, context).block();
    }

    /**
     * Lists all the metrics namespaces created for the resource URI.
     * @param resourceUri The resource URI for which the metrics namespaces are listed.
     * @param startTime The returned list of metrics namespaces are created after the specified start time.
     * @return A {@link PagedIterable paged collection} of metrics namespaces.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<MetricNamespace> listMetricNamespaces(String resourceUri, OffsetDateTime startTime) {
        return listMetricNamespaces(resourceUri, startTime, Context.NONE);
    }

    /**
     * Lists all the metrics namespaces created for the resource URI.
     * @param resourceUri The resource URI for which the metrics namespaces are listed.
     * @param startTime The returned list of metrics namespaces are created after the specified start time.
     * @param context Additional context that is passed through the Http pipeline during the service call. If no
     * additional context is required, pass {@link Context#NONE} instead.
     * @return A {@link PagedIterable paged collection} of metrics namespaces.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<MetricNamespace> listMetricNamespaces(String resourceUri, OffsetDateTime startTime,
                                                               Context context) {
        return new PagedIterable<>(asyncClient.listMetricNamespaces(resourceUri, startTime, context));
    }

    /**
     * Lists all the metrics definitions created for the resource URI.
     * @param resourceUri The resource URI for which the metrics definitions are listed.
     * @return A {@link PagedIterable paged collection} of metrics definitions.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<MetricDefinition> listMetricDefinitions(String resourceUri) {
        return listMetricDefinitions(resourceUri, null, Context.NONE);
    }

    /**
     * Lists all the metrics definitions created for the resource URI.
     * @param resourceUri The resource URI for which the metrics definitions are listed.
     * @param metricsNamespace The metrics namespace to which the listed metrics definitions belong.
     * @param context Additional context that is passed through the Http pipeline during the service call. If no
     * additional context is required, pass {@link Context#NONE} instead.
     * @return A {@link PagedIterable paged collection} of metrics definitions.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<MetricDefinition> listMetricDefinitions(String resourceUri, String metricsNamespace,
                                                                 Context context) {
        return new PagedIterable<>(asyncClient.listMetricDefinitions(resourceUri, metricsNamespace, context));
    }
}
