// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.security.keyvault.certificates;

import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceClient;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.exception.HttpRequestException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.Response;
import com.azure.core.util.Context;
import com.azure.core.util.polling.SyncPoller;
import com.azure.security.keyvault.certificates.models.CertificateOperation;
import com.azure.security.keyvault.certificates.models.CertificatePolicy;
import com.azure.security.keyvault.certificates.models.DeletedCertificate;
import com.azure.security.keyvault.certificates.models.CertificateContact;
import com.azure.security.keyvault.certificates.models.CertificateIssuer;
import com.azure.security.keyvault.certificates.models.IssuerProperties;
import com.azure.security.keyvault.certificates.models.MergeCertificateOptions;
import com.azure.security.keyvault.certificates.models.CertificateProperties;
import com.azure.security.keyvault.certificates.models.KeyVaultCertificate;
import com.azure.security.keyvault.certificates.models.KeyVaultCertificateWithPolicy;
import com.azure.security.keyvault.certificates.models.CertificatePolicyAction;
import com.azure.security.keyvault.certificates.models.LifetimeAction;
import com.azure.security.keyvault.certificates.models.ImportCertificateOptions;

import java.util.List;
import java.util.Map;
import java.util.Objects;


/**
 * The CertificateClient provides synchronous methods to manage {@link KeyVaultCertificate certifcates} in the Azure Key Vault. The client
 * supports creating, retrieving, updating, merging, deleting, purging, backing up, restoring and listing the
 * {@link KeyVaultCertificate certificates}. The client also supports listing {@link DeletedCertificate deleted certificates} for
 * a soft-delete enabled Azure Key Vault.
 *
 * <p>The client further allows creating, retrieving, updating, deleting and listing the {@link CertificateIssuer certificate issuers}. The client also supports
 * creating, listing and deleting {@link CertificateContact certificate contacts}</p>
 *
 * <p><strong>Samples to construct the sync client</strong></p>
 *
 * <pre>
 * CertificateClient certificateClient = new CertificateClientBuilder&#40;&#41;
 *     .credential&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;
 *     .vaultUrl&#40;&quot;https:&#47;&#47;myvault.vault.azure.net&#47;&quot;&#41;
 *     .httpLogOptions&#40;new HttpLogOptions&#40;&#41;.setLogLevel&#40;HttpLogDetailLevel.BODY_AND_HEADERS&#41;&#41;
 *     .buildClient&#40;&#41;;
 * </pre>
 *
 * @see CertificateClientBuilder
 * @see PagedIterable
 */
@ServiceClient(builder = CertificateClientBuilder.class, serviceInterfaces = CertificateService.class)
public final class CertificateClient {
    private final CertificateAsyncClient client;

    /**
     * Creates a CertificateClient that uses {@code pipeline} to service requests
     *
     * @param client The {@link CertificateAsyncClient} that the client routes its request through.
     */
    CertificateClient(CertificateAsyncClient client) {
        this.client = client;
    }

    /**
     * Get the vault endpoint url to which service requests are sent to.
     * @return the vault endpoint url
     */
    public String getVaultUrl() {
        return client.getVaultUrl();
    }

    /**
     * Creates a new certificate. If this is the first version, the certificate resource is created. This operation requires
     * the certificates/create permission.
     *
     * <p>Create certificate is a long running operation. It indefinitely waits for the create certificate operation to complete on service side.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Create certificate is a long running operation. The createCertificate indefinitely waits for the operation to complete and
     * returns its last status. The details of the last certificate operation status are printed when a response is received</p>
     *
     * <pre>
     * CertificatePolicy certificatePolicyPkcsSelf = new CertificatePolicy&#40;&quot;Self&quot;,
     *     &quot;CN=SelfSignedJavaPkcs12&quot;&#41;;
     * SyncPoller&lt;CertificateOperation, KeyVaultCertificateWithPolicy&gt; certPoller = certificateClient
     *     .beginCreateCertificate&#40;&quot;certificateName&quot;, certificatePolicyPkcsSelf, true, new HashMap&lt;&gt;&#40;&#41;&#41;;
     * certPoller.waitUntil&#40;LongRunningOperationStatus.SUCCESSFULLY_COMPLETED&#41;;
     * KeyVaultCertificate cert = certPoller.getFinalResult&#40;&#41;;
     * System.out.printf&#40;&quot;Certificate created with name %s&quot;, cert.getName&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to be created.
     * @param policy The policy of the certificate to be created.
     * @param isEnabled The enabled status of the certificate.
     * @param tags The application specific metadata to set.
     * @throws ResourceModifiedException when invalid certificate policy configuration is provided.
     * @return A {@link SyncPoller} to poll on the create certificate operation status.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public SyncPoller<CertificateOperation, KeyVaultCertificateWithPolicy> beginCreateCertificate(String certificateName, CertificatePolicy policy, Boolean isEnabled, Map<String, String> tags) {
        return client.beginCreateCertificate(certificateName, policy, isEnabled, tags).getSyncPoller();
    }

    /**
     * Creates a new certificate. If this is the first version, the certificate resource is created. This operation requires
     * the certificates/create permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Create certificate is a long running operation. The createCertificate indefinitely waits for the operation to complete and
     * returns its last status. The details of the last certificate operation status are printed when a response is received</p>
     *
     * <pre>
     * CertificatePolicy certificatePolicy = new CertificatePolicy&#40;&quot;Self&quot;,
     *     &quot;CN=SelfSignedJavaPkcs12&quot;&#41;;
     * SyncPoller&lt;CertificateOperation, KeyVaultCertificateWithPolicy&gt; certificatePoller = certificateClient
     *     .beginCreateCertificate&#40;&quot;certificateName&quot;, certificatePolicy&#41;;
     * certificatePoller.waitUntil&#40;LongRunningOperationStatus.SUCCESSFULLY_COMPLETED&#41;;
     * KeyVaultCertificate certificate = certificatePoller.getFinalResult&#40;&#41;;
     * System.out.printf&#40;&quot;Certificate created with name %s&quot;, certificate.getName&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to be created.
     * @param policy The policy of the certificate to be created.
     * @throws ResourceModifiedException when invalid certificate policy configuration is provided.
     * @return A {@link SyncPoller} to poll on the create certificate operation status.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public SyncPoller<CertificateOperation, KeyVaultCertificateWithPolicy> beginCreateCertificate(String certificateName, CertificatePolicy policy) {
        return client.beginCreateCertificate(certificateName, policy).getSyncPoller();
    }

    /**
     * Gets a pending {@link CertificateOperation} from the key vault. This operation requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Geta a pending certificate operation. The {@link SyncPoller poller} allows users to automatically poll on the certificate
     * operation status.</p>
     *
     * <pre>
     * SyncPoller&lt;CertificateOperation, KeyVaultCertificateWithPolicy&gt; certPoller = certificateClient
     *     .getCertificateOperation&#40;&quot;certificateName&quot;&#41;;
     * certPoller.waitUntil&#40;LongRunningOperationStatus.SUCCESSFULLY_COMPLETED&#41;;
     * KeyVaultCertificate cert = certPoller.getFinalResult&#40;&#41;;
     * System.out.printf&#40;&quot;Certificate created with name %s&quot;, cert.getName&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @throws ResourceNotFoundException when a certificate operation for a certificate with {@code certificateName} doesn't exist.
     * @return A {@link SyncPoller} to poll on the certificate operation status.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public SyncPoller<CertificateOperation, KeyVaultCertificateWithPolicy> getCertificateOperation(String certificateName) {
        return client.getCertificateOperation(certificateName).getSyncPoller();
    }

    /**
     * Gets information about the latest version of the specified certificate. This operation requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the certificate in the key vault. Prints out the returned certificate details when a response has been received.</p>
     *
     * <pre>
     * KeyVaultCertificateWithPolicy certificate = certificateClient.getCertificate&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Received certificate with name %s and version %s and secret id&quot;,
     *     certificate.getProperties&#40;&#41;.getName&#40;&#41;,
     *     certificate.getProperties&#40;&#41;.getVersion&#40;&#41;, certificate.getSecretId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to retrieve, cannot be null
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return The requested {@link KeyVaultCertificateWithPolicy certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public KeyVaultCertificateWithPolicy getCertificate(String certificateName) {
        return getCertificateWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Gets information about the latest version of the specified certificate. This operation requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the certificate in the key vault. Prints out the returned certificate details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;KeyVaultCertificateWithPolicy&gt; certificateWithResponse = certificateClient
     *     .getCertificateWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Received certificate with name %s and version %s and secret id&quot;,
     *     certificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     certificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getVersion&#40;&#41;, certificate.getSecretId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to retrieve, cannot be null
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the requested {@link KeyVaultCertificateWithPolicy certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<KeyVaultCertificateWithPolicy> getCertificateWithResponse(String certificateName, Context context) {
        return client.getCertificateWithResponse(certificateName, "", context).block();
    }

    /**
     * Gets information about the latest version of the specified certificate. This operation requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the certificate in the key vault. Prints out the returned certificate details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;KeyVaultCertificate&gt; returnedCertificateWithResponse = certificateClient
     *     .getCertificateVersionWithResponse&#40;&quot;certificateName&quot;, &quot;certificateVersion&quot;,
     *         new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Received certificate with name %s and version %s and secret id&quot;,
     *     returnedCertificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     returnedCertificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getVersion&#40;&#41;,
     *     returnedCertificateWithResponse.getValue&#40;&#41;.getSecretId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to retrieve, cannot be null
     * @param version The version of the certificate to retrieve. If this is an empty String or null then latest version of the certificate is retrieved.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the requested {@link KeyVaultCertificate certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<KeyVaultCertificate> getCertificateVersionWithResponse(String certificateName, String version, Context context) {
        return client.getCertificateVersionWithResponse(certificateName, version, context).block();
    }

    /**
     * Gets information about the specified version of the specified certificate. This operation requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the certificate in the key vault. Prints out the returned certificate details when a response has been received.</p>
     *
     * <pre>
     * KeyVaultCertificate returnedCertificate = certificateClient.getCertificateVersion&#40;&quot;certificateName&quot;,
     *     &quot;certificateVersion&quot;&#41;;
     * System.out.printf&#40;&quot;Received certificate with name %s and version %s and secret id&quot;,
     *     returnedCertificate.getProperties&#40;&#41;.getName&#40;&#41;, returnedCertificate.getProperties&#40;&#41;.getVersion&#40;&#41;,
     *     returnedCertificate.getSecretId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to retrieve, cannot be null
     * @param version The version of the certificate to retrieve. If this is an empty String or null then latest version of the certificate is retrieved.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return The requested {@link KeyVaultCertificate certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public KeyVaultCertificate getCertificateVersion(String certificateName, String version) {
        return getCertificateVersionWithResponse(certificateName, version, Context.NONE).getValue();
    }

    /**
     * Updates the specified attributes associated with the specified certificate. The update operation changes specified attributes of an existing
     * stored certificate and attributes that are not specified in the request are left unchanged. This operation requires the certificates/update permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets latest version of the certificate, changes its tags and enabled status and then updates it in the Azure Key Vault. Prints out the
     * returned certificate details when a response has been received.</p>
     *
     * <pre>
     * KeyVaultCertificate certificate = certificateClient.getCertificate&#40;&quot;certificateName&quot;&#41;;
     * &#47;&#47; Update certificate enabled status
     * certificate.getProperties&#40;&#41;.setEnabled&#40;false&#41;;
     * KeyVaultCertificate updatedCertificate = certificateClient.updateCertificateProperties&#40;certificate.getProperties&#40;&#41;&#41;;
     * System.out.printf&#40;&quot;Updated Certificate with name %s and enabled status %s&quot;,
     *     updatedCertificate.getProperties&#40;&#41;.getName&#40;&#41;, updatedCertificate.getProperties&#40;&#41;.isEnabled&#40;&#41;&#41;;
     * </pre>
     *
     * @param properties The {@link CertificateProperties} object with updated properties.
     * @throws NullPointerException if {@code certificate} is {@code null}.
     * @throws ResourceNotFoundException when a certificate with {@link CertificateProperties#getName() certificateName} and {@link CertificateProperties#getVersion() version} doesn't exist in the key vault.
     * @throws HttpRequestException if {@link CertificateProperties#getName() certificateName} or {@link CertificateProperties#getVersion() version} is empty string.
     * @return The {@link CertificateProperties updated certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public KeyVaultCertificate updateCertificateProperties(CertificateProperties properties) {
        return updateCertificatePropertiesWithResponse(properties, Context.NONE).getValue();
    }

    /**
     * Updates the specified attributes associated with the specified certificate. The update operation changes specified attributes of an existing
     * stored certificate and attributes that are not specified in the request are left unchanged. This operation requires the certificates/update permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets latest version of the certificate, changes its tags and enabled status and then updates it in the Azure Key Vault. Prints out the
     * returned certificate details when a response has been received.</p>
     *
     * <pre>
     * KeyVaultCertificate certificateToUpdate = certificateClient.getCertificate&#40;&quot;certificateName&quot;&#41;;
     * &#47;&#47; Update certificate enabled status
     * certificateToUpdate.getProperties&#40;&#41;.setEnabled&#40;false&#41;;
     * Response&lt;KeyVaultCertificate&gt; updatedCertificateResponse = certificateClient.
     *     updateCertificatePropertiesWithResponse&#40;certificateToUpdate.getProperties&#40;&#41;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Updated Certificate with name %s and enabled status %s&quot;,
     *     updatedCertificateResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     updatedCertificateResponse.getValue&#40;&#41;.getProperties&#40;&#41;.isEnabled&#40;&#41;&#41;;
     * </pre>
     *
     * @param properties The {@link CertificateProperties} object with updated properties.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws NullPointerException if {@code certificate} is {@code null}.
     * @throws ResourceNotFoundException when a certificate with {@link CertificateProperties#getName() certificateName} and {@link CertificateProperties#getVersion() version} doesn't exist in the key vault.
     * @throws HttpRequestException if {@link CertificateProperties#getName() certificateName} or {@link CertificateProperties#getVersion() version} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateProperties updated certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<KeyVaultCertificate> updateCertificatePropertiesWithResponse(CertificateProperties properties, Context context) {
        return client.updateCertificatePropertiesWithResponse(properties, context).block();
    }


    /**
     * Deletes a certificate from a specified key vault. All the versions of the certificate along with its associated policy
     * get deleted. If soft-delete is enabled on the key vault then the certificate is placed in the deleted state and requires to be
     * purged for permanent deletion else the certificate is permanently deleted. The delete operation applies to any certificate stored in
     * Azure Key Vault but it cannot be applied to an individual version of a certificate. This operation requires the certificates/delete permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the certificate in the Azure Key Vault. Prints out the
     * deleted certificate details when a response has been received.</p>
     *
     * <pre>
     * SyncPoller&lt;DeletedCertificate, Void&gt; deleteCertificatePoller =
     *     certificateClient.beginDeleteCertificate&#40;&quot;certificateName&quot;&#41;;
     * &#47;&#47; Deleted Certificate is accessible as soon as polling beings.
     * PollResponse&lt;DeletedCertificate&gt; pollResponse = deleteCertificatePoller.poll&#40;&#41;;
     * System.out.printf&#40;&quot;Deleted certitifcate with name %s and recovery id %s&quot;, pollResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     pollResponse.getValue&#40;&#41;.getRecoveryId&#40;&#41;&#41;;
     * deleteCertificatePoller.waitForCompletion&#40;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate to be deleted.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link SyncPoller} to poll on and retrieve {@link DeletedCertificate deleted certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public SyncPoller<DeletedCertificate, Void> beginDeleteCertificate(String certificateName) {
        return client.beginDeleteCertificate(certificateName).getSyncPoller();
    }

    /**
     * Retrieves information about the specified deleted certificate. The GetDeletedCertificate operation  is applicable for soft-delete
     * enabled vaults and additionally retrieves deleted certificate's attributes, such as retention interval, scheduled permanent deletion and the current deletion recovery level. This operation
     * requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Gets the deleted certificate from the key vault enabled for soft-delete. Prints out the
     * deleted certificate details when a response has been received.</p>
     *
     * <pre>
     * DeletedCertificate deletedCertificate = certificateClient.getDeletedCertificate&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Deleted certificate with name %s and recovery id %s&quot;, deletedCertificate.getName&#40;&#41;,
     *     deletedCertificate.getRecoveryId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the deleted certificate.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return The {@link DeletedCertificate deleted certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public DeletedCertificate getDeletedCertificate(String certificateName) {
        return getDeletedCertificateWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Retrieves information about the specified deleted certificate. The GetDeletedCertificate operation  is applicable for soft-delete
     * enabled vaults and additionally retrieves deleted certificate's attributes, such as retention interval, scheduled permanent deletion and the current deletion recovery level. This operation
     * requires the certificates/get permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Gets the deleted certificate from the key vault enabled for soft-delete. Prints out the
     * deleted certificate details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;DeletedCertificate&gt; deletedCertificateWithResponse = certificateClient
     *     .getDeletedCertificateWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Deleted certificate with name %s and recovery id %s&quot;,
     *     deletedCertificateWithResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     deletedCertificateWithResponse.getValue&#40;&#41;.getRecoveryId&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the deleted certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link DeletedCertificate deleted certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<DeletedCertificate> getDeletedCertificateWithResponse(String certificateName, Context context) {
        return client.getDeletedCertificateWithResponse(certificateName, context).block();
    }

    /**
     * Permanently deletes the specified deleted certificate without possibility for recovery. The Purge Deleted Certificate operation is applicable for
     * soft-delete enabled vaults and is not available if the recovery level does not specify 'Purgeable'. This operation requires the certificate/purge permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Purges the deleted certificate from the key vault enabled for soft-delete. Prints out the
     * status code from the server response when a response has been received.</p>

     * <pre>
     * certificateClient.purgeDeletedCertificate&#40;&quot;certificateName&quot;&#41;;
     * </pre>
     *
     * @param certificateName The name of the deleted certificate.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public void purgeDeletedCertificate(String certificateName) {
        purgeDeletedCertificateWithResponse(certificateName, Context.NONE);
    }

    /**
     * Permanently deletes the specified deleted certificate without possibility for recovery. The Purge Deleted Certificate operation is applicable for
     * soft-delete enabled vaults and is not available if the recovery level does not specify 'Purgeable'. This operation requires the certificate/purge permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Purges the deleted certificate from the key vault enabled for soft-delete. Prints out the
     * status code from the server response when a response has been received.</p>

     * <pre>
     * Response&lt;Void&gt; purgeResponse = certificateClient.purgeDeletedCertificateWithResponse&#40;&quot;certificateName&quot;,
     *     new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Purged Deleted certificate with status %d %n&quot;, purgeResponse.getStatusCode&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the deleted certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A response containing status code and HTTP headers.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<Void> purgeDeletedCertificateWithResponse(String certificateName, Context context) {
        return client.purgeDeletedCertificateWithResponse(certificateName, context).block();
    }

    /**
     * Recovers the deleted certificate back to its current version under /certificates and can only be performed on a soft-delete enabled vault.
     * The RecoverDeletedCertificate operation performs the reversal of the Delete operation and must be issued during the retention interval
     * (available in the deleted certificate's attributes). This operation requires the certificates/recover permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Recovers the deleted certificate from the key vault enabled for soft-delete. Prints out the
     * recovered certificate details when a response has been received.</p>

     * <pre>
     * SyncPoller&lt;KeyVaultCertificateWithPolicy, Void&gt; recoverCertPoller = certificateClient
     *     .beginRecoverDeletedCertificate&#40;&quot;deletedCertificateName&quot;&#41;;
     * &#47;&#47; Recovered certificate is accessible as soon as polling beings
     * PollResponse&lt;KeyVaultCertificateWithPolicy&gt; pollResponse = recoverCertPoller.poll&#40;&#41;;
     * System.out.printf&#40;&quot; Recovered Deleted certificate with name %s and id %s&quot;, pollResponse.getValue&#40;&#41;
     *     .getProperties&#40;&#41;.getName&#40;&#41;, pollResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getId&#40;&#41;&#41;;
     * recoverCertPoller.waitForCompletion&#40;&#41;;
     * </pre>
     *
     * @param certificateName The name of the deleted certificate to be recovered.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the certificate vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link SyncPoller} to poll on and retrieve {@link KeyVaultCertificateWithPolicy recovered certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public SyncPoller<KeyVaultCertificateWithPolicy, Void> beginRecoverDeletedCertificate(String certificateName) {
        return client.beginRecoverDeletedCertificate(certificateName).getSyncPoller();
    }

    /**
     * Requests that a backup of the specified certificate be downloaded to the client. All versions of the certificate will
     * be downloaded. This operation requires the certificates/backup permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Backs up the certificate from the key vault. Prints out the
     * length of the certificate's backup byte array returned in the response.</p>
     *
     * <pre>
     * byte[] certificateBackup = certificateClient.backupCertificate&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Backed up certificate with back up blob length %d&quot;, certificateBackup.length&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return The backed up certificate blob.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public byte[] backupCertificate(String certificateName) {
        return backupCertificateWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Requests that a backup of the specified certificate be downloaded to the client. All versions of the certificate will
     * be downloaded. This operation requires the certificates/backup permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Backs up the certificate from the key vault. Prints out the
     * length of the certificate's backup byte array returned in the response.</p>
     *
     * <pre>
     * Response&lt;byte[]&gt; certificateBackupWithResponse = certificateClient
     *     .backupCertificateWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Backed up certificate with back up blob length %d&quot;,
     *     certificateBackupWithResponse.getValue&#40;&#41;.length&#41;;
     * </pre>
     *
     * @param certificateName The certificateName of the certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the backed up certificate blob.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<byte[]> backupCertificateWithResponse(String certificateName, Context context) {
        return client.backupCertificateWithResponse(certificateName, context).block();
    }

    /**
     * Restores a backed up certificate to the vault. All the versions of the certificate are restored to the vault. This operation
     * requires the certificates/restore permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Restores the certificate in the key vault from its backup. Prints out the restored certificate
     * details when a response has been received.</p>
     *
     * <pre>
     * byte[] certificateBackupBlob = &#123;&#125;;
     * KeyVaultCertificate certificate = certificateClient.restoreCertificateBackup&#40;certificateBackupBlob&#41;;
     * System.out.printf&#40;&quot; Restored certificate with name %s and id %s&quot;,
     *     certificate.getProperties&#40;&#41;.getName&#40;&#41;, certificate.getProperties&#40;&#41;.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param backup The backup blob associated with the certificate.
     * @throws ResourceModifiedException when {@code backup} blob is malformed.
     * @return The {@link KeyVaultCertificate restored certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public KeyVaultCertificateWithPolicy restoreCertificateBackup(byte[] backup) {
        return restoreCertificateBackupWithResponse(backup, Context.NONE).getValue();
    }

    /**
     * Restores a backed up certificate to the vault. All the versions of the certificate are restored to the vault. This operation
     * requires the certificates/restore permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Restores the certificate in the key vault from its backup. Prints out the restored certificate
     * details when a response has been received.</p>
     *
     * <pre>
     * byte[] certificateBackupBlobArray = &#123;&#125;;
     * Response&lt;KeyVaultCertificateWithPolicy&gt; certificateResponse = certificateClient
     *     .restoreCertificateBackupWithResponse&#40;certificateBackupBlobArray, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot; Restored certificate with name %s and id %s&quot;,
     *     certificateResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     certificateResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param backup The backup blob associated with the certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceModifiedException when {@code backup} blob is malformed.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultCertificate restored certificate}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<KeyVaultCertificateWithPolicy> restoreCertificateBackupWithResponse(byte[] backup, Context context) {
        return client.restoreCertificateBackupWithResponse(backup, context).block();
    }

    /**
     * List certificates in a the key vault. Retrieves the set of certificates resources in the key vault and the individual
     * certificate response in the iterable is represented by {@link CertificateProperties} as only the certificate identifier, thumbprint,
     * attributes and tags are provided in the response. The policy and individual certificate versions are not listed in
     * the response. This operation requires the certificates/list permission.
     *
     * <p>It is possible to get certificates with all the properties excluding the policy from this information. Loop over the {@link CertificateProperties} and
     * call {@link CertificateClient#getCertificateVersion(String, String)} . This will return the {@link KeyVaultCertificate certificate}
     * with all its properties excluding the policy.</p>
     *
     * <pre>
     * for &#40;CertificateProperties certificateProperties : certificateClient.listPropertiesOfCertificates&#40;&#41;&#41; &#123;
     *     KeyVaultCertificate certificateWithAllProperties = certificateClient
     *         .getCertificateVersion&#40;certificateProperties.getName&#40;&#41;, certificateProperties.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received certificate with name %s and secret id %s&quot;,
     *         certificateWithAllProperties.getProperties&#40;&#41;.getName&#40;&#41;,
     *         certificateWithAllProperties.getSecretId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @return A {@link PagedIterable} containing {@link CertificateProperties certificate} for all the certificates in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateProperties> listPropertiesOfCertificates() {
        return new PagedIterable<>(client.listPropertiesOfCertificates(false, Context.NONE));
    }

    /**
     * List certificates in a the key vault. Retrieves the set of certificates resources in the key vault and the individual
     * certificate response in the iterable is represented by {@link CertificateProperties} as only the certificate identifier, thumbprint,
     * attributes and tags are provided in the response. The policy and individual certificate versions are not listed in
     * the response. This operation requires the certificates/list permission.
     *
     * <p>It is possible to get certificates with all the properties excluding the policy from this information. Loop over the {@link CertificateProperties} and
     * call {@link CertificateClient#getCertificateVersion(String, String)} . This will return the {@link KeyVaultCertificate certificate}
     * with all its properties excluding the policy.</p>
     *
     * <pre>
     * for &#40;CertificateProperties certificateProperties : certificateClient
     *     .listPropertiesOfCertificates&#40;true, new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     KeyVaultCertificate certificateWithAllProperties = certificateClient
     *         .getCertificateVersion&#40;certificateProperties.getName&#40;&#41;, certificateProperties.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received certificate with name %s and secret id %s&quot;,
     *         certificateWithAllProperties.getProperties&#40;&#41;.getName&#40;&#41;,
     *         certificateWithAllProperties.getSecretId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param includePending indicate if pending certificates should be included in the results.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link PagedIterable} containing {@link CertificateProperties certificate} for all the certificates in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateProperties> listPropertiesOfCertificates(boolean includePending, Context context) {
        return new PagedIterable<>(client.listPropertiesOfCertificates(includePending, context));
    }

    /**
     * Lists the {@link DeletedCertificate deleted certificates} in the key vault currently available for recovery. This operation includes
     * deletion-specific information and is applicable for vaults enabled for soft-delete. This operation requires the
     * {@code certificates/get/list} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the deleted certificates in the key vault. Prints out the
     * recovery id of each deleted certificate when a response has been received.</p>
     *
     * <pre>
     * for &#40;DeletedCertificate deletedCertificate : certificateClient.listDeletedCertificates&#40;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted certificate's recovery Id %s&quot;, deletedCertificate.getRecoveryId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @return A {@link PagedIterable} containing all of the {@link DeletedCertificate deleted certificates} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<DeletedCertificate> listDeletedCertificates() {
        return listDeletedCertificates(false, Context.NONE);
    }


    /**
     * Lists the {@link DeletedCertificate deleted certificates} in the key vault currently available for recovery. This operation includes
     * deletion-specific information and is applicable for vaults enabled for soft-delete. This operation requires the
     * {@code certificates/get/list} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the deleted certificates in the key vault. Prints out the
     * recovery id of each deleted certificate when a response has been received.</p>
     *
     * <pre>
     * for &#40;DeletedCertificate deletedCertificate : certificateClient
     *     .listDeletedCertificates&#40;true, new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted certificate's recovery Id %s&quot;, deletedCertificate.getRecoveryId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param includePending indicate if pending deleted certificates should be included in the results.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link PagedIterable} containing all of the {@link DeletedCertificate deleted certificates} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<DeletedCertificate> listDeletedCertificates(boolean includePending, Context context) {
        return new PagedIterable<>(client.listDeletedCertificates(includePending, context));
    }

    /**
     * List all versions of the specified certificate. The individual certificate response in the iterable is represented by {@link CertificateProperties}
     * as only the certificate identifier, thumbprint, attributes and tags are provided in the response. The policy is not listed in
     * the response. This operation requires the certificates/list permission.
     *
     * <p>It is possible to get the certificates with properties excluding the policy for all the versions from this information. Loop over the {@link CertificateProperties} and
     * call {@link CertificateClient#getCertificateVersion(String, String)}. This will return the {@link KeyVaultCertificate certificate}
     * with all its properties excluding the policy.</p>
     *
     * <pre>
     * for &#40;CertificateProperties certificateProperties : certificateClient
     *     .listPropertiesOfCertificateVersions&#40;&quot;certificateName&quot;&#41;&#41; &#123;
     *     KeyVaultCertificate certificateWithAllProperites  = certificateClient
     *         .getCertificateVersion&#40;certificateProperties.getName&#40;&#41;, certificateProperties.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received certificate's version with name %s, version %s and secret id %s&quot;,
     *         certificateWithAllProperites.getProperties&#40;&#41;.getName&#40;&#41;,
     *         certificateWithAllProperites.getProperties&#40;&#41;.getVersion&#40;&#41;, certificateWithAllProperites.getSecretId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link PagedIterable} containing {@link CertificateProperties certificate} of all the versions of the specified certificate in the vault. Paged Iterable is empty if certificate with {@code certificateName} does not exist in key vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateProperties> listPropertiesOfCertificateVersions(String certificateName) {
        return listPropertiesOfCertificateVersions(certificateName, Context.NONE);
    }

    /**
     * List all versions of the specified certificate. The individual certificate response in the iterable is represented by {@link CertificateProperties}
     * as only the certificate identifier, thumbprint, attributes and tags are provided in the response. The policy is not listed in
     * the response. This operation requires the certificates/list permission.
     *
     * <p>It is possible to get the certificates with properties excluding the policy for all the versions from this information. Loop over the {@link CertificateProperties} and
     * call {@link CertificateClient#getCertificateVersion(String, String)}. This will return the {@link KeyVaultCertificate certificate}
     * with all its properties excluding the policy.</p>
     *
     * <pre>
     * for &#40;CertificateProperties certificateProperties : certificateClient
     *     .listPropertiesOfCertificateVersions&#40;&quot;certificateName&quot;&#41;&#41; &#123;
     *     KeyVaultCertificate certificateWithAllProperites  = certificateClient
     *         .getCertificateVersion&#40;certificateProperties.getName&#40;&#41;, certificateProperties.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received certificate's version with name %s, version %s and secret id %s&quot;,
     *         certificateWithAllProperites.getProperties&#40;&#41;.getName&#40;&#41;,
     *         certificateWithAllProperites.getProperties&#40;&#41;.getVersion&#40;&#41;, certificateWithAllProperites.getSecretId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate with {@code certificateName} is empty string.
     * @return A {@link PagedIterable} containing {@link CertificateProperties certificate} of all the versions of the specified certificate in the vault. Iterable is empty if certificate with {@code certificateName} does not exist in key vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateProperties> listPropertiesOfCertificateVersions(String certificateName, Context context) {
        return new PagedIterable<>(client.listPropertiesOfCertificateVersions(certificateName, context));
    }

    /**
     * Retrieves the policy of the specified certificate in the key vault. This operation requires the {@code certificates/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the policy of a certirifcate in the key vault. Prints out the
     * returned certificate policy details when a response has been received.</p>
     *
     * <pre>
     * CertificatePolicy policy = certificateClient.getCertificatePolicy&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Received policy with subject name %s&quot;, policy.getSubject&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate whose policy is to be retrieved, cannot be null
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return The requested {@link CertificatePolicy certificate policy}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificatePolicy getCertificatePolicy(String certificateName) {
        return getCertificatePolicyWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Retrieves the policy of the specified certificate in the key vault. This operation requires the {@code certificates/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the policy of a certirifcate in the key vault. Prints out the
     * returned certificate policy details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;CertificatePolicy&gt; returnedPolicyWithResponse = certificateClient.getCertificatePolicyWithResponse&#40;
     *     &quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Received policy with subject name %s&quot;,
     *     returnedPolicyWithResponse.getValue&#40;&#41;.getSubject&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate whose policy is to be retrieved, cannot be null
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the requested {@link CertificatePolicy certificate policy}.
     */
    public Response<CertificatePolicy> getCertificatePolicyWithResponse(String certificateName, Context context) {
        return client.getCertificatePolicyWithResponse(certificateName, context).block();
    }

    /**
     * Updates the policy for a certificate. The update operation changes specified attributes of the certificate policy and attributes
     * that are not specified in the request are left unchanged. This operation requires the {@code certificates/update} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the certificate policy, changes its properties and then updates it in the Azure Key Vault. Prints out the
     * returned policy details when a response has been received.</p>
     *
     * <pre>
     * CertificatePolicy certificatePolicy = certificateClient.getCertificatePolicy&#40;&quot;certificateName&quot;&#41;;
     * &#47;&#47;Update the certificate policy cert transparency property.
     * certificatePolicy.setCertificateTransparent&#40;true&#41;;
     * CertificatePolicy updatedCertPolicy = certificateClient.updateCertificatePolicy&#40;&quot;certificateName&quot;,
     *     certificatePolicy&#41;;
     * System.out.printf&#40;&quot;Updated Certificate Policy transparency status %s&quot;,
     *     updatedCertPolicy.isCertificateTransparent&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate whose policy is to be updated.
     * @param policy The certificate policy to be updated.
     * @throws NullPointerException if {@code policy} is {@code null}.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string or if {@code policy} is invalid.
     * @return The updated {@link CertificatePolicy certificate policy}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificatePolicy updateCertificatePolicy(String certificateName, CertificatePolicy policy) {
        return updateCertificatePolicyWithResponse(certificateName, policy, Context.NONE).getValue();
    }

    /**
     * Updates the policy for a certificate. The update operation changes specified attributes of the certificate policy and attributes
     * that are not specified in the request are left unchanged. This operation requires the {@code certificates/update} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the certificate policy, changes its properties and then updates it in the Azure Key Vault. Prints out the
     * returned policy details when a response has been received.</p>
     *
     * <pre>
     * CertificatePolicy certificatePolicyToUpdate = certificateClient.getCertificatePolicy&#40;&quot;certificateName&quot;&#41;;
     * &#47;&#47;Update the certificate policy cert transparency property.
     * certificatePolicyToUpdate.setCertificateTransparent&#40;true&#41;;
     * Response&lt;CertificatePolicy&gt; updatedCertPolicyWithResponse = certificateClient
     *     .updateCertificatePolicyWithResponse&#40;&quot;certificateName&quot;, certificatePolicyToUpdate,
     *         new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Updated Certificate Policy transparency status %s&quot;, updatedCertPolicyWithResponse
     *     .getValue&#40;&#41;.isCertificateTransparent&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The certificateName of the certificate whose policy is to be updated.
     * @param policy The certificate policy to be updated.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws NullPointerException if {@code policy} is {@code null}.
     * @throws ResourceNotFoundException when a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code certificateName} is empty string or if {@code policy} is invalid.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the updated {@link CertificatePolicy certificate policy}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificatePolicy> updateCertificatePolicyWithResponse(String certificateName, CertificatePolicy policy, Context context) {
        return client.updateCertificatePolicyWithResponse(certificateName, policy, context).block();
    }

    /**
     * Creates the specified certificate issuer. The SetCertificateIssuer operation updates the specified certificate issuer if it
     * already exists or adds it if doesn't exist. This operation requires the certificates/setissuers permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new certificate issuer in the key vault. Prints out the created certificate issuer details when a
     * response has been received.</p>
     *
     * <pre>
     * CertificateIssuer issuerToCreate = new CertificateIssuer&#40;&quot;myissuer&quot;, &quot;myProvider&quot;&#41;
     *     .setAccountId&#40;&quot;testAccount&quot;&#41;
     *     .setAdministratorContacts&#40;Arrays.asList&#40;new AdministratorContact&#40;&#41;.setFirstName&#40;&quot;test&quot;&#41;.setLastName&#40;&quot;name&quot;&#41;
     *         .setEmail&#40;&quot;test&#123;@literal @&#125;example.com&quot;&#41;&#41;&#41;;
     * CertificateIssuer returnedIssuer = certificateClient.createIssuer&#40;issuerToCreate&#41;;
     * System.out.printf&#40;&quot;Created Issuer with name %s provider %s&quot;, returnedIssuer.getName&#40;&#41;,
     *     returnedIssuer.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuer The configuration of the certificate issuer to be created.
     * @throws ResourceModifiedException when invalid certificate issuer {@code issuer} configuration is provided.
     * @throws HttpRequestException when a certificate issuer with {@link CertificateIssuer#getName() name} is empty string.
     * @return The created {@link CertificateIssuer certificate issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateIssuer createIssuer(CertificateIssuer issuer) {
        return createIssuerWithResponse(issuer, Context.NONE).getValue();
    }

    /**
     * Creates the specified certificate issuer. The SetCertificateIssuer operation updates the specified certificate issuer if it
     * already exists or adds it if doesn't exist. This operation requires the certificates/setissuers permission.

     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new certificate issuer in the key vault. Prints out the created certificate
     * issuer details when a response has been received.</p>
     *
     * <pre>
     * CertificateIssuer issuer = new CertificateIssuer&#40;&quot;issuerName&quot;, &quot;myProvider&quot;&#41;
     *     .setAccountId&#40;&quot;testAccount&quot;&#41;
     *     .setAdministratorContacts&#40;Arrays.asList&#40;new AdministratorContact&#40;&#41;.setFirstName&#40;&quot;test&quot;&#41;.setLastName&#40;&quot;name&quot;&#41;
     *         .setEmail&#40;&quot;test&#123;@literal @&#125;example.com&quot;&#41;&#41;&#41;;
     * Response&lt;CertificateIssuer&gt; issuerResponse = certificateClient.createIssuerWithResponse&#40;issuer,
     *     new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Created Issuer with name %s provider %s&quot;, issuerResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     issuerResponse.getValue&#40;&#41;.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuer The configuration of the certificate issuer to be created.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceModifiedException when invalid certificate issuer {@code issuer} configuration is provided.
     * @throws HttpRequestException when a certificate issuer with {@link CertificateIssuer#getName() name} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the created {@link CertificateIssuer certificate issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateIssuer> createIssuerWithResponse(CertificateIssuer issuer, Context context) {
        return client.createIssuerWithResponse(issuer, context).block();
    }

    /**
     * Retrieves the specified certificate issuer from the key vault. This operation requires the certificates/manageissuers/getissuers permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the specificed certifcate issuer in the key vault. Prints out the returned certificate issuer details when
     * a response has been received.</p>
     *
     * <pre>
     * Response&lt;CertificateIssuer&gt; issuerResponse = certificateClient.getIssuerWithResponse&#40;&quot;issuerName&quot;,
     *     new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Retrieved issuer with name %s and prodier %s&quot;, issuerResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     issuerResponse.getValue&#40;&#41;.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuerName The name of the certificate issuer to retrieve, cannot be null
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate issuer with {@code issuerName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code issuerName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the requested {@link CertificateIssuer certificate issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateIssuer> getIssuerWithResponse(String issuerName, Context context) {
        return client.getIssuerWithResponse(issuerName, context).block();
    }

    /**
     * Retrieves the specified certificate issuer from the key vault. This operation requires the certificates/manageissuers/getissuers permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the specified certificate issuer in the key vault. Prints out the returned certificate issuer details
     * when a response has been received.</p>
     *
     * <pre>
     * CertificateIssuer returnedIssuer = certificateClient.getIssuer&#40;&quot;issuerName&quot;&#41;;
     * System.out.printf&#40;&quot;Retrieved issuer with name %s and prodier %s&quot;, returnedIssuer.getName&#40;&#41;,
     *     returnedIssuer.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuerName The name of the certificate issuer to retrieve, cannot be null
     * @throws ResourceNotFoundException when a certificate issuer with {@code issuerName} doesn't exist in the key vault.
     * @throws HttpRequestException if {@code issuerName} is empty string.
     * @return The requested {@link CertificateIssuer certificate issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateIssuer getIssuer(String issuerName) {
        return getIssuerWithResponse(issuerName, Context.NONE).getValue();
    }

    /**
     * Deletes the specified certificate issuer. The DeleteCertificateIssuer operation permanently removes the specified certificate
     * issuer from the key vault. This operation requires the {@code certificates/manageissuers/deleteissuers permission}.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the certificate issuer in the Azure Key Vault. Prints out the
     * deleted certificate details when a response has been received.</p>
     *
     * <pre>
     * CertificateIssuer deletedIssuer = certificateClient.deleteIssuer&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Deleted certificate issuer with name %s and provider id %s&quot;, deletedIssuer.getName&#40;&#41;,
     *     deletedIssuer.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuerName The name of the certificate issuer to be deleted.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate issuer with {@code issuerName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate issuer with {@code issuerName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateIssuer deleted issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateIssuer> deleteIssuerWithResponse(String issuerName, Context context) {
        return client.deleteIssuerWithResponse(issuerName, context).block();
    }

    /**
     * Deletes the specified certificate issuer. The DeleteCertificateIssuer operation permanently removes the specified certificate
     * issuer from the key vault. This operation requires the {@code certificates/manageissuers/deleteissuers permission}.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the certificate issuer in the Azure Key Vault. Prints out the deleted certificate details when a
     * response has been received.</p>
     *
     * <pre>
     * Response&lt;CertificateIssuer&gt; deletedIssuerWithResponse = certificateClient.
     *     deleteIssuerWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Deleted certificate issuer with name %s and provider id %s&quot;,
     *     deletedIssuerWithResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     deletedIssuerWithResponse.getValue&#40;&#41;.getProvider&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuerName The name of the certificate issuer to be deleted.
     * @throws ResourceNotFoundException when a certificate issuer with {@code issuerName} doesn't exist in the key vault.
     * @throws HttpRequestException when a certificate issuer with {@code issuerName} is empty string.
     * @return The {@link CertificateIssuer deleted issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateIssuer deleteIssuer(String issuerName) {
        return deleteIssuerWithResponse(issuerName, Context.NONE).getValue();
    }

    /**
     * List all the certificate issuers resources in the key vault. The individual certificate issuer response in the iterable is represented by {@link IssuerProperties}
     * as only the certificate issuer identifier and provider are provided in the response. This operation requires the
     * {@code certificates/manageissuers/getissuers} permission.
     *
     * <p>It is possible to get the certificate issuer with all of its properties from this information. Loop over the {@link IssuerProperties issuerProperties} and
     * call {@link CertificateClient#getIssuer(String)} . This will return the {@link CertificateIssuer issuer}
     * with all its properties.</p>.
     *
     * <pre>
     * for &#40;IssuerProperties issuer : certificateClient.listPropertiesOfIssuers&#40;&#41;&#41; &#123;
     *     CertificateIssuer retrievedIssuer = certificateClient.getIssuer&#40;issuer.getName&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received issuer with name %s and provider %s&quot;, retrievedIssuer.getName&#40;&#41;,
     *         retrievedIssuer.getProvider&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @return A {@link PagedIterable} containing all of the {@link IssuerProperties certificate issuers} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<IssuerProperties> listPropertiesOfIssuers() {
        return listPropertiesOfIssuers(Context.NONE);
    }

    /**
     * List all the certificate issuers resources in the key vault. The individual certificate issuer response in the iterable is represented by {@link IssuerProperties}
     * as only the certificate issuer identifier and provider are provided in the response. This operation requires the
     * {@code certificates/manageissuers/getissuers} permission.
     *
     * <p>It is possible to get the certificate issuer with all of its properties from this information. Loop over the {@link IssuerProperties issuerProperties} and
     * call {@link CertificateClient#getIssuer(String)}. This will return the {@link CertificateIssuer issuer}
     * with all its properties.</p>.
     *
     * <pre>
     * for &#40;IssuerProperties issuer : certificateClient.listPropertiesOfIssuers&#40;new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     CertificateIssuer retrievedIssuer = certificateClient.getIssuer&#40;issuer.getName&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received issuer with name %s and provider %s&quot;, retrievedIssuer.getName&#40;&#41;,
     *         retrievedIssuer.getProvider&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link PagedIterable} containing all of the {@link IssuerProperties certificate issuers} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<IssuerProperties> listPropertiesOfIssuers(Context context) {
        return new PagedIterable<>(client.listPropertiesOfIssuers(context));
    }

    /**
     * Updates the specified certificate issuer. The UpdateCertificateIssuer operation updates the specified attributes of
     * the certificate issuer entity. This operation requires the certificates/setissuers permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the certificate issuer, changes its attributes/properties then updates it in the Azure Key Vault. Prints out the
     * returned certificate issuer details when a response has been received.</p>
     *
     * <pre>
     * CertificateIssuer returnedIssuer = certificateClient.getIssuer&#40;&quot;issuerName&quot;&#41;;
     * returnedIssuer.setAccountId&#40;&quot;newAccountId&quot;&#41;;
     * CertificateIssuer updatedIssuer = certificateClient.updateIssuer&#40;returnedIssuer&#41;;
     * System.out.printf&#40;&quot;Updated issuer with name %s, provider %s and account Id %s&quot;, updatedIssuer.getName&#40;&#41;,
     *     updatedIssuer.getProvider&#40;&#41;, updatedIssuer.getAccountId&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuer The {@link CertificateIssuer issuer} with updated properties.
     * @throws NullPointerException if {@code issuer} is {@code null}.
     * @throws ResourceNotFoundException when a certificate issuer with {@link CertificateIssuer#getName() name} doesn't exist in the key vault.
     * @throws HttpRequestException if {@link CertificateIssuer#getName() name} is empty string.
     * @return The {@link CertificateIssuer updated issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateIssuer updateIssuer(CertificateIssuer issuer) {
        return updateIssuerWithResponse(issuer, Context.NONE).getValue();
    }

    /**
     * Updates the specified certificate issuer. The UpdateCertificateIssuer operation updates the specified attributes of
     * the certificate issuer entity. This operation requires the certificates/setissuers permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the certificate issuer, changes its attributes/properties then updates it in the Azure Key Vault. Prints out the
     * returned certificate issuer details when a response has been received.</p>
     *
     * <pre>
     * CertificateIssuer issuer = certificateClient.getIssuer&#40;&quot;issuerName&quot;&#41;;
     * returnedIssuer.setAccountId&#40;&quot;newAccountId&quot;&#41;;
     * Response&lt;CertificateIssuer&gt; updatedIssuerWithResponse = certificateClient.updateIssuerWithResponse&#40;issuer,
     *     new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Updated issuer with name %s, provider %s and account Id %s&quot;,
     *     updatedIssuerWithResponse.getValue&#40;&#41;.getName&#40;&#41;,
     *     updatedIssuerWithResponse.getValue&#40;&#41;.getProvider&#40;&#41;,
     *     updatedIssuerWithResponse.getValue&#40;&#41;.getAccountId&#40;&#41;&#41;;
     * </pre>
     *
     * @param issuer The {@link CertificateIssuer issuer} with updated properties.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws NullPointerException if {@code issuer} is {@code null}.
     * @throws ResourceNotFoundException when a certificate issuer with {@link CertificateIssuer#getName() name} doesn't exist in the key vault.
     * @throws HttpRequestException if {@link CertificateIssuer#getName() name} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateIssuer updated issuer}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateIssuer> updateIssuerWithResponse(CertificateIssuer issuer, Context context) {
        return client.updateIssuerWithResponse(issuer, context).block();
    }


    /**
     * Sets the certificate contacts on the key vault. This operation requires the {@code certificates/managecontacts} permission.
     *
     *<p>The {@link LifetimeAction} of type {@link CertificatePolicyAction#EMAIL_CONTACTS} set on a {@link CertificatePolicy} emails the contacts set on the vault when triggered.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Sets the certificate contacts in the Azure Key Vault. Prints out the returned contacts details.</p>
     *
     * <pre>
     * CertificateContact contactToAdd = new CertificateContact&#40;&#41;.setName&#40;&quot;user&quot;&#41;.setEmail&#40;&quot;useremail&#123;@literal @&#125;example.com&quot;&#41;;
     * for &#40;CertificateContact contact : certificateClient.setContacts&#40;Collections.singletonList&#40;contactToAdd&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Added contact with name %s and email %s to key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param contacts The list of contacts to set on the vault.
     * @throws HttpRequestException when a contact information provided is invalid/incomplete.
     * @return A {@link PagedIterable} containing all of the {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> setContacts(List<CertificateContact> contacts) {
        return setContacts(contacts, Context.NONE);
    }

    /**
     * Sets the certificate contacts on the key vault. This operation requires the {@code certificates/managecontacts} permission.
     *
     *<p>The {@link LifetimeAction} of type {@link CertificatePolicyAction#EMAIL_CONTACTS} set on a {@link CertificatePolicy} emails the contacts set on the vault when triggered.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Sets the certificate contacts in the Azure Key Vault. Prints out the returned contacts details.</p>
     *
     * <pre>
     * CertificateContact sampleContact = new CertificateContact&#40;&#41;.setName&#40;&quot;user&quot;&#41;.setEmail&#40;&quot;useremail&#123;@literal @&#125;example.com&quot;&#41;;
     * for &#40;CertificateContact contact : certificateClient.setContacts&#40;Collections.singletonList&#40;sampleContact&#41;,
     *     new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Added contact with name %s and email %s to key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param contacts The list of contacts to set on the vault.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws HttpRequestException when a contact information provided is invalid/incomplete.
     * @return A {@link PagedIterable} containing all of the {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> setContacts(List<CertificateContact> contacts, Context context) {
        return new PagedIterable<>(client.setContacts(contacts, context));
    }

    /**
     * Lists the certificate contacts in the key vault. This operation requires the certificates/managecontacts permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the certificate contacts in the Azure Key Vault. Prints out the returned contacts details in the response.</p>
     *
     * <pre>
     * for &#40;CertificateContact contact : certificateClient.listContacts&#40;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Added contact with name %s and email %s to key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @return A {@link PagedIterable} containing all of the {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> listContacts() {
        return listContacts(Context.NONE);
    }


    /**
     * Lists the certificate contacts in the key vault. This operation requires the certificates/managecontacts permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the certificate contacts in the Azure Key Vault. Prints out the returned contacts details in the response.</p>
     *
     * <pre>
     * for &#40;CertificateContact contact : certificateClient.listContacts&#40;new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Added contact with name %s and email %s to key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link PagedIterable} containing all of the {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> listContacts(Context context) {
        return new PagedIterable<>(client.listContacts(context));
    }

    /**
     * Deletes the certificate contacts in the key vault. This operation requires the {@code certificates/managecontacts} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the certificate contacts in the Azure Key Vault. Subscribes to the call and prints out the
     * deleted contacts details.</p>
     *
     * <pre>
     * for &#40;CertificateContact contact : certificateClient.deleteContacts&#40;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted contact with name %s and email %s from key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @return A {@link PagedIterable} containing all of the deleted {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> deleteContacts() {
        return deleteContacts(Context.NONE);
    }

    /**
     * Deletes the certificate contacts in the key vault. This operation requires the {@code certificates/managecontacts} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the certificate contacts in the Azure Key Vault. Prints out the deleted contacts details in the response.</p>
     *
     * <pre>
     * for &#40;CertificateContact contact : certificateClient.deleteContacts&#40;new Context&#40;key1, value1&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted contact with name %s and email %s from key vault&quot;, contact.getName&#40;&#41;,
     *         contact.getEmail&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link PagedIterable} containing all of the deleted {@link CertificateContact certificate contacts} in the vault.
     */
    @ServiceMethod(returns = ReturnType.COLLECTION)
    public PagedIterable<CertificateContact> deleteContacts(Context context) {
        return new PagedIterable<>(client.deleteContacts(context));
    }

    /**
     * Deletes the creation operation for the specified certificate that is in the process of being created. The certificate is
     * no longer created. This operation requires the {@code certificates/update permission}.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Triggers certificate creation and then deletes the certificate creation operation in the Azure Key Vault. Subscribes to the call and prints out the
     * deleted certificate operation details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;CertificateOperation&gt; deletedCertificateOperationWithResponse = certificateClient
     *     .deleteCertificateOperationWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Deleted Certificate Operation's last status %s&quot;,
     *     deletedCertificateOperationWithResponse.getValue&#40;&#41;.getStatus&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @throws ResourceNotFoundException when a certificate operation for a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when the {@code certificateName} is empty string.
     * @return The deleted {@link CertificateOperation certificate operation}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateOperation deleteCertificateOperation(String certificateName) {
        return deleteCertificateOperationWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Deletes the creation operation for the specified certificate that is in the process of being created. The certificate is
     * no longer created. This operation requires the {@code certificates/update permission}.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Triggers certificate creation and then deletes the certificate creation operation in the Azure Key Vault. Subscribes to the call and prints out the
     * deleted certificate operation details when a response has been received.</p>
     *
     * <pre>
     * CertificateOperation deletedCertificateOperation = certificateClient
     *     .deleteCertificateOperation&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Deleted Certificate Operation's last status %s&quot;, deletedCertificateOperation.getStatus&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate operation for a certificate with {@code certificateName} doesn't exist in the key vault.
     * @throws HttpRequestException when the {@code certificateName} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateOperation deleted certificate operation}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateOperation> deleteCertificateOperationWithResponse(String certificateName, Context context) {
        return client.deleteCertificateOperationWithResponse(certificateName, context).block();
    }
    /**
     * Cancels a certificate creation operation that is already in progress. This operation requires the {@code certificates/update} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Triggers certificate creation and then cancels the certificate creation operation in the Azure Key Vault. Subscribes to the call and prints out the
     * updated certificate operation details when a response has been received.</p>
     *
     * <pre>
     * CertificateOperation certificateOperation = certificateClient
     *     .cancelCertificateOperation&#40;&quot;certificateName&quot;&#41;;
     * System.out.printf&#40;&quot;Certificate Operation status %s&quot;, certificateOperation.getStatus&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate which is in the process of being created.
     * @throws ResourceNotFoundException when a certificate operation for a certificate with {@code name} doesn't exist in the key vault.
     * @throws HttpRequestException when the {@code name} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateOperation cancelled certificate operation}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public CertificateOperation cancelCertificateOperation(String certificateName) {
        return cancelCertificateOperationWithResponse(certificateName, Context.NONE).getValue();
    }

    /**
     * Cancels a certificate creation operation that is already in progress. This operation requires the {@code certificates/update} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Triggers certificate creation and then cancels the certificate creation operation in the Azure Key Vault. Subscribes to the call and prints out the
     * updated certificate operation details when a response has been received.</p>
     *
     * <pre>
     * Response&lt;CertificateOperation&gt; certificateOperationWithResponse = certificateClient
     *     .cancelCertificateOperationWithResponse&#40;&quot;certificateName&quot;, new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Certificate Operation status %s&quot;, certificateOperationWithResponse.getValue&#40;&#41;.getStatus&#40;&#41;&#41;;
     * </pre>
     *
     * @param certificateName The name of the certificate which is in the process of being created.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws ResourceNotFoundException when a certificate operation for a certificate with {@code name} doesn't exist in the key vault.
     * @throws HttpRequestException when the {@code name} is empty string.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link CertificateOperation cancelled certificate operation}.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<CertificateOperation> cancelCertificateOperationWithResponse(String certificateName, Context context) {
        return client.cancelCertificateOperationWithResponse(certificateName, context).block();
    }

    /**
     * Merges a certificate or a certificate chain with a key pair currently available in the service. This operation requires
     * the {@code certificates/create} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Merges a certificate with a kay pair available in the service.</p>
     *
     * <pre>
     * List&lt;byte[]&gt; x509CertificatesToMerge = new ArrayList&lt;&gt;&#40;&#41;;
     * MergeCertificateOptions config =
     *     new MergeCertificateOptions&#40;&quot;certificateName&quot;, x509CertificatesToMerge&#41;
     *         .setEnabled&#40;false&#41;;
     * KeyVaultCertificate mergedCertificate = certificateClient.mergeCertificate&#40;config&#41;;
     * System.out.printf&#40;&quot;Received Certificate with name %s and key id %s&quot;,
     *     mergedCertificate.getProperties&#40;&#41;.getName&#40;&#41;, mergedCertificate.getKeyId&#40;&#41;&#41;;
     * </pre>
     *
     * @param mergeCertificateOptions the merge certificate configuration holding the x509 certificates.
     * @throws NullPointerException when {@code mergeCertificateOptions} is null.
     * @throws HttpRequestException if {@code mergeCertificateOptions} is invalid/corrupt.
     * @return The merged certificate.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public KeyVaultCertificateWithPolicy mergeCertificate(MergeCertificateOptions mergeCertificateOptions) {
        return mergeCertificateWithResponse(mergeCertificateOptions, Context.NONE).getValue();
    }

    /**
     * Merges a certificate or a certificate chain with a key pair currently available in the service. This operation requires
     * the {@code certificates/create} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Merges a certificate with a kay pair available in the service.</p>
     *
     * <pre>
     * List&lt;byte[]&gt; x509CertsToMerge = new ArrayList&lt;&gt;&#40;&#41;;
     * MergeCertificateOptions mergeConfig =
     *     new MergeCertificateOptions&#40;&quot;certificateName&quot;, x509CertsToMerge&#41;
     *         .setEnabled&#40;false&#41;;
     * Response&lt;KeyVaultCertificateWithPolicy&gt; mergedCertificateWithResponse =
     *     certificateClient.mergeCertificateWithResponse&#40;mergeConfig, new Context&#40;key2, value2&#41;&#41;;
     * System.out.printf&#40;&quot;Received Certificate with name %s and key id %s&quot;,
     *     mergedCertificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     mergedCertificateWithResponse.getValue&#40;&#41;.getKeyId&#40;&#41;&#41;;
     * </pre>
     *
     * @param mergeCertificateOptions the merge certificate configuration holding the x509 certificates.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws NullPointerException when {@code mergeCertificateOptions} is null.
     * @throws HttpRequestException if {@code mergeCertificateOptions} is invalid/corrupt.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the merged certificate.
     */
    @ServiceMethod(returns = ReturnType.SINGLE)
    public Response<KeyVaultCertificateWithPolicy> mergeCertificateWithResponse(MergeCertificateOptions mergeCertificateOptions, Context context) {
        Objects.requireNonNull(mergeCertificateOptions, "'mergeCertificateOptions' cannot be null.");
        return client.mergeCertificateWithResponse(mergeCertificateOptions, context).block();
    }

    /**
     * Imports a pre-existing certificate to the key vault. The specified certificate must be in PFX or PEM format,
     * and must contain the private key as well as the x509 certificates. This operation requires the {@code certificates/import} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Imports a certificate into the key vault.</p>
     *
     * <pre>
     * byte[] certificateToImport = new byte[100];
     * ImportCertificateOptions config =
     *     new ImportCertificateOptions&#40;&quot;certificateName&quot;, certificateToImport&#41;.setEnabled&#40;false&#41;;
     * KeyVaultCertificate importedCertificate = certificateClient.importCertificate&#40;config&#41;;
     * System.out.printf&#40;&quot;Received Certificate with name %s and key id %s&quot;,
     *     importedCertificate.getProperties&#40;&#41;.getName&#40;&#41;, importedCertificate.getKeyId&#40;&#41;&#41;;
     * </pre>
     *
     * @param importCertificateOptions The details of the certificate to import to the key vault
     * @throws HttpRequestException when the {@code importCertificateOptions} are invalid.
     * @return the {@link KeyVaultCertificateWithPolicy imported certificate}.
     */
    public KeyVaultCertificateWithPolicy importCertificate(ImportCertificateOptions importCertificateOptions) {
        return importCertificateWithResponse(importCertificateOptions, Context.NONE).getValue();
    }

    /**
     * Imports a pre-existing certificate to the key vault. The specified certificate must be in PFX or PEM format,
     * and must contain the private key as well as the x509 certificates. This operation requires the {@code certificates/import} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p> Imports a certificate into the key vault.</p>
     *
     * <pre>
     * byte[] certToImport = new byte[100];
     * ImportCertificateOptions importCertificateOptions  =
     *     new ImportCertificateOptions&#40;&quot;certificateName&quot;, certToImport&#41;.setEnabled&#40;false&#41;;
     * Response&lt;KeyVaultCertificateWithPolicy&gt; importedCertificateWithResponse =
     *     certificateClient.importCertificateWithResponse&#40;importCertificateOptions, new Context&#40;key2, value2&#41;&#41;;
     * System.out.printf&#40;&quot;Received Certificate with name %s and key id %s&quot;,
     *     importedCertificateWithResponse.getValue&#40;&#41;.getProperties&#40;&#41;.getName&#40;&#41;,
     *     importedCertificateWithResponse.getValue&#40;&#41;.getKeyId&#40;&#41;&#41;;
     * </pre>
     *
     * @param importCertificateOptions The details of the certificate to import to the key vault
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @throws HttpRequestException when the {@code importCertificateOptions} are invalid.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultCertificateWithPolicy imported certificate}.
     */
    public Response<KeyVaultCertificateWithPolicy> importCertificateWithResponse(ImportCertificateOptions importCertificateOptions, Context context) {
        return client.importCertificateWithResponse(importCertificateOptions, context).block();
    }
}
