/*
 * Decompiled with CFR 0.152.
 */
package com.azure.security.keyvault.jca.implementation;

import com.azure.security.keyvault.jca.KeyVaultPrivateKey;
import com.azure.security.keyvault.jca.implementation.model.CertificateBundle;
import com.azure.security.keyvault.jca.implementation.model.CertificateItem;
import com.azure.security.keyvault.jca.implementation.model.CertificateListResult;
import com.azure.security.keyvault.jca.implementation.model.CertificatePolicy;
import com.azure.security.keyvault.jca.implementation.model.KeyProperties;
import com.azure.security.keyvault.jca.implementation.model.SecretBundle;
import com.azure.security.keyvault.jca.implementation.model.SignResult;
import com.azure.security.keyvault.jca.implementation.utils.AccessTokenUtil;
import com.azure.security.keyvault.jca.implementation.utils.HttpUtil;
import com.azure.security.keyvault.jca.implementation.utils.JsonConverterUtil;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.net.URLEncoder;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

public class KeyVaultClient {
    public static final String KEY_VAULT_BASE_URI_GLOBAL = "https://vault.azure.net";
    public static final String KEY_VAULT_BASE_URI_CN = "https://vault.azure.cn";
    public static final String KEY_VAULT_BASE_URI_US = "https://vault.usgovcloudapi.net";
    public static final String KEY_VAULT_BASE_URI_DE = "https://vault.microsoftazure.de";
    public static final String AAD_LOGIN_URI_GLOBAL = "https://login.microsoftonline.com/";
    public static final String AAD_LOGIN_URI_CN = "https://login.partner.microsoftonline.cn/";
    public static final String AAD_LOGIN_URI_US = "https://login.microsoftonline.us/";
    public static final String AAD_LOGIN_URI_DE = "https://login.microsoftonline.de/";
    private static final Logger LOGGER = Logger.getLogger(KeyVaultClient.class.getName());
    private static final String HTTPS_PREFIX = "https://";
    private static final String API_VERSION_POSTFIX = "?api-version=7.1";
    private final String keyVaultBaseUri;
    private final String keyVaultUrl;
    private final String aadAuthenticationUrl;
    private final String tenantId;
    private final String clientId;
    private final String clientSecret;
    private String managedIdentity;

    public static String getAADLoginURIByKeyVaultBaseUri(String keyVaultBaseUri) {
        String aadAuthenticationUrl;
        switch (keyVaultBaseUri) {
            case "https://vault.azure.net": {
                aadAuthenticationUrl = AAD_LOGIN_URI_GLOBAL;
                break;
            }
            case "https://vault.azure.cn": {
                aadAuthenticationUrl = AAD_LOGIN_URI_CN;
                break;
            }
            case "https://vault.usgovcloudapi.net": {
                aadAuthenticationUrl = AAD_LOGIN_URI_US;
                break;
            }
            case "https://vault.microsoftazure.de": {
                aadAuthenticationUrl = AAD_LOGIN_URI_DE;
                break;
            }
            default: {
                throw new IllegalArgumentException("Property of azure.keyvault.uri is illegal.");
            }
        }
        return aadAuthenticationUrl;
    }

    KeyVaultClient(String keyVaultUri, String managedIdentity) {
        this(keyVaultUri, null, null, null, managedIdentity);
    }

    public KeyVaultClient(String keyVaultUri, String tenantId, String clientId, String clientSecret) {
        this(keyVaultUri, tenantId, clientId, clientSecret, null);
    }

    public KeyVaultClient(String keyVaultUri, String tenantId, String clientId, String clientSecret, String managedIdentity) {
        LOGGER.log(Level.INFO, "Using Azure Key Vault: {0}", keyVaultUri);
        if (!keyVaultUri.endsWith("/")) {
            keyVaultUri = keyVaultUri + "/";
        }
        this.keyVaultUrl = keyVaultUri;
        String domainNameSuffix = Optional.of(keyVaultUri).map(uri -> uri.split("\\.", 2)[1]).map(suffix -> suffix.substring(0, suffix.length() - 1)).orElse(null);
        this.keyVaultBaseUri = HTTPS_PREFIX + domainNameSuffix;
        this.aadAuthenticationUrl = KeyVaultClient.getAADLoginURIByKeyVaultBaseUri(this.keyVaultBaseUri);
        this.tenantId = tenantId;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.managedIdentity = managedIdentity;
    }

    public static KeyVaultClient createKeyVaultClientBySystemProperty() {
        String keyVaultUri = System.getProperty("azure.keyvault.uri");
        String tenantId = System.getProperty("azure.keyvault.tenant-id");
        String clientId = System.getProperty("azure.keyvault.client-id");
        String clientSecret = System.getProperty("azure.keyvault.client-secret");
        String managedIdentity = System.getProperty("azure.keyvault.managed-identity");
        return new KeyVaultClient(keyVaultUri, tenantId, clientId, clientSecret, managedIdentity);
    }

    private String getAccessToken() {
        LOGGER.entering("KeyVaultClient", "getAccessToken");
        String accessToken = null;
        try {
            String resource = URLEncoder.encode(this.keyVaultBaseUri, "UTF-8");
            if (this.managedIdentity != null) {
                this.managedIdentity = URLEncoder.encode(this.managedIdentity, "UTF-8");
            }
            accessToken = this.tenantId != null && this.clientId != null && this.clientSecret != null ? AccessTokenUtil.getAccessToken(resource, this.aadAuthenticationUrl, this.tenantId, this.clientId, this.clientSecret) : AccessTokenUtil.getAccessToken(resource, this.managedIdentity);
        }
        catch (Throwable throwable) {
            LOGGER.log(Level.WARNING, "Unsupported encoding or missing Httpclient", throwable);
        }
        LOGGER.exiting("KeyVaultClient", "getAccessToken", accessToken);
        return accessToken;
    }

    public List<String> getAliases() {
        ArrayList<String> result = new ArrayList<String>();
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String url = String.format("%scertificates%s", this.keyVaultUrl, API_VERSION_POSTFIX);
        while (url != null && url.length() != 0) {
            String response = HttpUtil.get(url, headers);
            CertificateListResult certificateListResult = null;
            if (response != null) {
                certificateListResult = (CertificateListResult)JsonConverterUtil.fromJson(response, CertificateListResult.class);
            }
            if (certificateListResult != null) {
                url = certificateListResult.getNextLink();
                for (CertificateItem certificateItem : certificateListResult.getValue()) {
                    String id = certificateItem.getId();
                    String alias = id.substring(id.indexOf("certificates") + "certificates".length() + 1);
                    result.add(alias);
                }
                continue;
            }
            url = null;
        }
        return result;
    }

    private CertificateBundle getCertificateBundle(String alias) {
        CertificateBundle result = null;
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String url = String.format("%scertificates/%s%s", this.keyVaultUrl, alias, API_VERSION_POSTFIX);
        String response = HttpUtil.get(url, headers);
        if (response != null) {
            result = (CertificateBundle)JsonConverterUtil.fromJson(response, CertificateBundle.class);
        }
        return result;
    }

    public Certificate getCertificate(String alias) {
        String certificateString;
        LOGGER.entering("KeyVaultClient", "getCertificate", alias);
        LOGGER.log(Level.INFO, "Getting certificate for alias: {0}", alias);
        X509Certificate certificate = null;
        CertificateBundle certificateBundle = this.getCertificateBundle(alias);
        if (certificateBundle != null && (certificateString = certificateBundle.getCer()) != null) {
            try {
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                certificate = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(Base64.getDecoder().decode(certificateString)));
            }
            catch (CertificateException ce) {
                LOGGER.log(Level.WARNING, "Certificate error", ce);
            }
        }
        LOGGER.exiting("KeyVaultClient", "getCertificate", certificate);
        return certificate;
    }

    public Key getKey(String alias, char[] password) {
        LOGGER.entering("KeyVaultClient", "getKey", new Object[]{alias, password});
        LOGGER.log(Level.INFO, "Getting key for alias: {0}", alias);
        CertificateBundle certificateBundle = this.getCertificateBundle(alias);
        boolean isExportable = Optional.ofNullable(certificateBundle).map(CertificateBundle::getPolicy).map(CertificatePolicy::getKeyProperties).map(KeyProperties::isExportable).orElse(false);
        String keyType = Optional.ofNullable(certificateBundle).map(CertificateBundle::getPolicy).map(CertificatePolicy::getKeyProperties).map(KeyProperties::getKty).orElse(null);
        if (!isExportable) {
            LOGGER.exiting("KeyVaultClient", "getKey", null);
            return Optional.ofNullable(certificateBundle).map(CertificateBundle::getKid).map(kid -> new KeyVaultPrivateKey(keyType, (String)kid)).orElse(null);
        }
        String certificateSecretUri = certificateBundle.getSid();
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String body = HttpUtil.get(certificateSecretUri + API_VERSION_POSTFIX, headers);
        if (body == null) {
            LOGGER.exiting("KeyVaultClient", "getKey", null);
            return null;
        }
        Key key = null;
        SecretBundle secretBundle = (SecretBundle)JsonConverterUtil.fromJson(body, SecretBundle.class);
        String contentType = secretBundle.getContentType();
        if ("application/x-pkcs12".equals(contentType)) {
            try {
                KeyStore keyStore = KeyStore.getInstance("PKCS12");
                keyStore.load(new ByteArrayInputStream(Base64.getDecoder().decode(secretBundle.getValue())), "".toCharArray());
                alias = keyStore.aliases().nextElement();
                key = keyStore.getKey(alias, "".toCharArray());
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException ex) {
                LOGGER.log(Level.WARNING, "Unable to decode key", ex);
            }
        } else if ("application/x-pem-file".equals(contentType)) {
            try {
                key = this.createPrivateKeyFromPem(secretBundle.getValue(), keyType);
            }
            catch (IOException | IllegalArgumentException | NoSuchAlgorithmException | InvalidKeySpecException ex) {
                LOGGER.log(Level.WARNING, "Unable to decode key", ex);
            }
        }
        LOGGER.exiting("KeyVaultClient", "getKey", key);
        return key;
    }

    public byte[] getSignedWithPrivateKey(String digestName, String digestValue, String keyId) {
        SignResult result = null;
        String bodyString = String.format("{\"alg\": \"" + digestName + "\", \"value\": \"%s\"}", digestValue);
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String url = String.format("%s/sign%s", keyId, API_VERSION_POSTFIX);
        String response = HttpUtil.post(url, headers, bodyString, "application/json");
        if (response != null) {
            result = (SignResult)JsonConverterUtil.fromJson(response, SignResult.class);
        }
        if (result != null) {
            return Base64.getUrlDecoder().decode(result.getValue());
        }
        return new byte[0];
    }

    private PrivateKey createPrivateKeyFromPem(String pemString, String keyType) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        StringBuilder builder = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new StringReader(pemString));){
            String line = reader.readLine();
            if (line == null || !line.contains("BEGIN PRIVATE KEY")) {
                throw new IllegalArgumentException("No PRIVATE KEY found");
            }
            line = "";
            while (line != null) {
                if (line.contains("END PRIVATE KEY")) {
                    break;
                }
                builder.append(line);
                line = reader.readLine();
            }
        }
        byte[] bytes = Base64.getDecoder().decode(builder.toString());
        PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(bytes);
        KeyFactory factory = KeyFactory.getInstance(keyType);
        return factory.generatePrivate(spec);
    }

    String getKeyVaultBaseUri() {
        return this.keyVaultBaseUri;
    }

    String getAadAuthenticationUrl() {
        return this.aadAuthenticationUrl;
    }
}

