// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.spring.autoconfigure.condition.aad;

import com.azure.spring.aad.AADApplicationType;
import com.azure.spring.autoconfigure.aad.AADAuthenticationProperties;
import org.springframework.boot.autoconfigure.condition.ConditionMessage;
import org.springframework.boot.autoconfigure.condition.ConditionOutcome;
import org.springframework.boot.autoconfigure.condition.SpringBootCondition;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.util.StringUtils;

import java.util.Optional;

import static com.azure.spring.aad.AADApplicationType.RESOURCE_SERVER;

/**
 * Web application, web resource server or all in scenario condition.
 */
public final class ClientRegistrationCondition extends SpringBootCondition {

    @Override
    public ConditionOutcome getMatchOutcome(ConditionContext context, AnnotatedTypeMetadata metadata) {
        ConditionMessage.Builder message = ConditionMessage.forCondition("AAD Application Client Condition");
        AADAuthenticationProperties properties =
            Binder.get(context.getEnvironment())
                  .bind("azure.activedirectory", AADAuthenticationProperties.class)
                  .orElse(null);
        if (properties == null) {
            return ConditionOutcome.noMatch(
                message.notAvailable("AAD authorization properties(azure.activedirectory" + ".xxx)"));
        }

        if (!StringUtils.hasText(properties.getClientId())) {
            return ConditionOutcome.noMatch(message.didNotFind("azure.activedirectory.client-id").atAll());
        }

        // Bind properties will not execute AADAuthenticationProperties#afterPropertiesSet()
        AADApplicationType applicationType = Optional.ofNullable(properties.getApplicationType())
                                                     .orElseGet(AADApplicationType::inferApplicationTypeByDependencies);
        if (applicationType == null || applicationType == RESOURCE_SERVER) {
            return ConditionOutcome.noMatch(
                message.because("Resource server does not need client registration."));
        }
        return ConditionOutcome.match(
            message.foundExactly("azure.activedirectory.application-type=" + applicationType));
    }
}
