// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.spring.cloud.autoconfigure.jms.properties;

import com.azure.spring.cloud.core.implementation.connectionstring.ServiceBusConnectionString;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.jms.JmsPoolConnectionFactoryProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.NestedConfigurationProperty;
import org.springframework.jms.support.QosSettings;
import org.springframework.util.StringUtils;

import java.time.Duration;

/**
 * {@link ConfigurationProperties} for configuring Azure Service Bus JMS.
 */
@ConfigurationProperties(prefix = AzureServiceBusJmsProperties.PREFIX)
public class AzureServiceBusJmsProperties implements InitializingBean {

    public static final String PREFIX = "spring.jms.servicebus";

    private static final String DEFAULT_REMOTE_URL = "amqp://localhost:5672";

    private static final String AMQP_URI_FORMAT = "amqps://%s?amqp.idleTimeout=%d";
    /**
     * Connection string to connect to a Service Bus namespace.
     */
    private String connectionString;
    /**
     * Service Bus topic client ID. Only works for the bean of topicJmsListenerContainerFactory.
     */
    private String topicClientId;
    /**
     * Connection idle timeout duration.
     */
    private Duration idleTimeout = Duration.ofMinutes(30);
    /**
     * Pricing tier for a Service Bus namespace.
     */
    private String pricingTier;

    private final Listener listener = new Listener();

    private final PrefetchPolicy prefetchPolicy = new PrefetchPolicy();

    @NestedConfigurationProperty
    private final JmsPoolConnectionFactoryProperties pool = new JmsPoolConnectionFactoryProperties();

    /**
     * URL of the AMQP broker. Auto-generated by default.
     */
    private String remoteUrl = DEFAULT_REMOTE_URL;

    /**
     * Login user of the AMQP broker.
     */
    private String username;

    /**
     * Login password of the AMQP broker.
     */
    private String password;

    /**
     * Get the URL of the AMQP broker.
     * @return the URL of the AMQP broker.
     */
    public String getRemoteUrl() {
        return remoteUrl;
    }

    /**
     * Set the URL of the AMQP broker.
     * @param remoteUrl the URL of the AMQP broker.
     */
    public void setRemoteUrl(String remoteUrl) {
        this.remoteUrl = remoteUrl;
    }

    /**
     * Get the login user of the AMQP broker.
     * @return the login user of the AMQP broker.
     */
    public String getUsername() {
        return username;
    }

    /**
     * Set the login user of the AMQP broker.
     * @param username the login user of the AMQP broker.
     */
    public void setUsername(String username) {
        this.username = username;
    }

    /**
     * Get the login password of the AMQP broker.
     * @return the login password of the AMQP broker.
     */
    public String getPassword() {
        return password;
    }

    /**
     * Set the login password of the AMQP broker.
     * @param password the login password of the AMQP broker.
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * The properties for a pooled connection factory.
     * @return the properties for a pooled connection factory.
     */
    public JmsPoolConnectionFactoryProperties getPool() {
        return pool;
    }

    /**
     * Get the connection string to connect to a Service Bus namesapce.
     * @return the connection string to connect to a Service Bus namesapce.
     */
    public String getConnectionString() {
        return connectionString;
    }

    /**
     * Set the connection string to connect to a Service Bus namesapce.
     * @param connectionString the connection string to connect to a Service Bus namesapce.
     */
    public void setConnectionString(String connectionString) {
        this.connectionString = connectionString;
    }

    /**
     * Get the Service Bus topic client ID.
     * @return the Service Bus topic client ID.
     */
    public String getTopicClientId() {
        return topicClientId;
    }

    /**
     * Set the Service Bus topic client ID.
     * @param topicClientId the Service Bus topic client ID.
     */
    public void setTopicClientId(String topicClientId) {
        this.topicClientId = topicClientId;
    }

    /**
     * Get the pricing tier for a Service Bus namespace.
     * @return the pricing tier for a Service Bus namespace.
     */
    public String getPricingTier() {
        return this.pricingTier;
    }

    /**
     * Set the pricing tier for a Service Bus namespace.
     * @param pricingTier the pricing tier for a Service Bus namespace.
     */
    public void setPricingTier(String pricingTier) {
        this.pricingTier = pricingTier;
    }

    /**
     * Get the connection idle timeout duration.
     * @return the connection idle timeout duration.
     */
    public Duration getIdleTimeout() {
        return idleTimeout;
    }

    /**
     * Set the connection idle timeout duration.
     * @param idleTimeout the connection idle timeout duration.
     */
    public void setIdleTimeout(Duration idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    /**
     * Get the listener related properties.
     * @return the listener related properties.
     */
    public Listener getListener() {
        return listener;
    }

    /**
     * Get the prefetch policy related properties.
     * @return the prefetch policy related properties.
     */
    public PrefetchPolicy getPrefetchPolicy() {
        return prefetchPolicy;
    }

    /**
     * Validate spring.jms.servicebus related properties.
     *
     * @throws IllegalArgumentException If connectionString is empty.
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        if (!StringUtils.hasText(connectionString)) {
            throw new IllegalArgumentException("'spring.jms.servicebus.connection-string' should be provided");
        }

        if (null == pricingTier || !pricingTier.matches("(?i)premium|standard|basic")) {
            throw new IllegalArgumentException("'spring.jms.servicebus.pricing-tier' is not valid");
        }

        ServiceBusConnectionString serviceBusConnectionString = new ServiceBusConnectionString(connectionString);
        String host = serviceBusConnectionString.getEndpointUri().getHost();

        this.remoteUrl = String.format(AMQP_URI_FORMAT, host, idleTimeout.toMillis());
        this.username = serviceBusConnectionString.getSharedAccessKeyName();
        this.password = serviceBusConnectionString.getSharedAccessKey();
    }

    /**
     * Properties to configure {@link org.apache.qpid.jms.policy.JmsDefaultPrefetchPolicy} for {@link
     * org.apache.qpid.jms.JmsConnectionFactory} .
     */
    public static class PrefetchPolicy {

        /**
         * Fallback value for prefetch option in this Service Bus namespace.
         */
        private int all = 0;
        /**
         * The number of prefetch for durable topic.
         */
        private int durableTopicPrefetch = 0;
        /**
         * The number of prefetch for queue browser.
         */
        private int queueBrowserPrefetch = 0;
        /**
         * The number of prefetch for queue.
         */
        private int queuePrefetch = 0;
        /**
         * The number of prefetch for topic.
         */
        private int topicPrefetch = 0;

        public int getAll() {
            return Math.max(all, 0);
        }

        public void setAll(int all) {
            this.all = all;
        }

        public int getDurableTopicPrefetch() {
            return durableTopicPrefetch > 0 ? durableTopicPrefetch : getAll();
        }

        public void setDurableTopicPrefetch(int durableTopicPrefetch) {
            this.durableTopicPrefetch = durableTopicPrefetch;
        }

        public int getQueueBrowserPrefetch() {
            return queueBrowserPrefetch > 0 ? queueBrowserPrefetch : getAll();
        }

        public void setQueueBrowserPrefetch(int queueBrowserPrefetch) {
            this.queueBrowserPrefetch = queueBrowserPrefetch;
        }

        public int getQueuePrefetch() {
            return queuePrefetch > 0 ? queuePrefetch : getAll();
        }

        public void setQueuePrefetch(int queuePrefetch) {
            this.queuePrefetch = queuePrefetch;
        }

        public int getTopicPrefetch() {
            return topicPrefetch > 0 ? topicPrefetch : getAll();
        }

        public void setTopicPrefetch(int topicPrefetch) {
            this.topicPrefetch = topicPrefetch;
        }
    }

    /**
     * Properties to configure {@link org.springframework.jms.annotation.JmsListener} for {@link
     * org.springframework.jms.config.AbstractJmsListenerContainerFactory}.
     */
    public static class Listener {

        /**
         * Whether the reply destination type is topic. Only works for the bean of topicJmsListenerContainerFactory.
         */
        private Boolean replyPubSubDomain;

        /**
         * The QosSettings to use when sending a reply.
         */
        private QosSettings replyQosSettings;

        /**
         * Whether to make the subscription durable. Only works for the bean of topicJmsListenerContainerFactory.
         */
        private Boolean subscriptionDurable = Boolean.TRUE;

        /**
         * Whether to make the subscription shared. Only works for the bean of topicJmsListenerContainerFactory.
         */
        private Boolean subscriptionShared;

        /**
         * The phase in which this container should be started and stopped.
         */
        private Integer phase;

        public Boolean isReplyPubSubDomain() {
            return replyPubSubDomain;
        }

        public void setReplyPubSubDomain(Boolean replyPubSubDomain) {
            this.replyPubSubDomain = replyPubSubDomain;
        }

        public QosSettings getReplyQosSettings() {
            return replyQosSettings;
        }

        public void setReplyQosSettings(QosSettings replyQosSettings) {
            this.replyQosSettings = replyQosSettings;
        }

        public Boolean isSubscriptionDurable() {
            return subscriptionDurable;
        }

        public void setSubscriptionDurable(Boolean subscriptionDurable) {
            this.subscriptionDurable = subscriptionDurable;
        }

        public Boolean isSubscriptionShared() {
            return subscriptionShared;
        }

        public void setSubscriptionShared(Boolean subscriptionShared) {
            this.subscriptionShared = subscriptionShared;
        }

        public Integer getPhase() {
            return phase;
        }

        public void setPhase(Integer phase) {
            this.phase = phase;
        }
    }
}
