/*
 * Decompiled with CFR 0.152.
 */
package com.azure.tools.codesnippetplugin.implementation;

import com.azure.tools.codesnippetplugin.implementation.SnippetDictionary;
import com.azure.tools.codesnippetplugin.implementation.SnippetOperationResult;
import com.azure.tools.codesnippetplugin.implementation.VerifyResult;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;

public final class SnippetReplacer {
    static final Pattern SNIPPET_DEF_BEGIN = Pattern.compile("\\s*\\/\\/\\s*BEGIN\\:\\s+([a-zA-Z0-9\\.\\#\\-\\_]*)\\s*");
    static final Pattern SNIPPET_DEF_END = Pattern.compile("\\s*\\/\\/\\s*END\\:\\s+([a-zA-Z0-9\\.\\#\\-\\_]*)\\s*");
    static final Pattern SNIPPET_SRC_CALL_BEGIN = Pattern.compile("(\\s*)\\*?\\s*<!--\\s+src_embed\\s+([a-zA-Z0-9\\.\\#\\-\\_]*)\\s*-->");
    static final Pattern SNIPPET_SRC_CALL_END = Pattern.compile("(\\s*)\\*?\\s*<!--\\s+end\\s+([a-zA-Z0-9\\.\\#\\-\\_]*)\\s*-->");
    static final Pattern SNIPPET_README_CALL_BEGIN = Pattern.compile("```(\\s*)?java\\s+([a-zA-Z0-9\\.\\#\\-\\_]*)\\s*");
    static final Pattern SNIPPET_README_CALL_END = Pattern.compile("```");
    static final Pattern WHITESPACE_EXTRACTION = Pattern.compile("(\\s*)(.*)");
    static final Pattern END_OF_LINE_WHITESPACES = Pattern.compile("[\\s]+$");
    private static final Map<String, String> REPLACEMENT_SET = new LinkedHashMap<String, String>(){
        {
            this.put("&", "&amp;");
            this.put("\"", "&quot;");
            this.put(">", "&gt;");
            this.put("<", "&lt;");
            this.put("@", "&#64;");
            this.put("{", "&#123;");
            this.put("}", "&#125;");
            this.put("(", "&#40;");
            this.put(")", "&#41;");
            this.put("/", "&#47;");
            this.put("\\", "&#92;");
        }
    };

    public static void verifyCodesnippets(Path codesnippetRootDirectory, String codesnippetGlob, Path sourcesRootDirectory, String sourcesGlob, boolean includeSources, Path readmePath, boolean includeReadme, int maxLineLength, Log logger) throws IOException, MojoExecutionException {
        if (!includeSources && !includeReadme) {
            logger.debug((CharSequence)"Neither sources or README were included. No codesnippet updating will be done.");
            return;
        }
        List<Path> codesnippetFiles = SnippetReplacer.globFiles(codesnippetRootDirectory, codesnippetGlob, false);
        List<Object> sourceFiles = Collections.emptyList();
        if (includeSources) {
            sourceFiles = SnippetReplacer.globFiles(sourcesRootDirectory, sourcesGlob, true);
        }
        ArrayList snippetsNeedingUpdate = new ArrayList();
        ArrayList<VerifyResult> badSnippetCalls = new ArrayList<VerifyResult>();
        Map<String, List<String>> foundSnippets = SnippetReplacer.getAllSnippets(codesnippetFiles);
        if (includeSources) {
            for (Path path : sourceFiles) {
                SnippetOperationResult<List<VerifyResult>> sourcesResult = SnippetReplacer.verifySnippets(path, SNIPPET_SRC_CALL_BEGIN, SNIPPET_SRC_CALL_END, foundSnippets, "<pre>", "</pre>", 1, "* ", false);
                snippetsNeedingUpdate.addAll((Collection)sourcesResult.result);
                badSnippetCalls.addAll(sourcesResult.errorList);
            }
        }
        if (includeReadme) {
            SnippetOperationResult<List<VerifyResult>> readmeResult = SnippetReplacer.verifySnippets(readmePath, SNIPPET_README_CALL_BEGIN, SNIPPET_README_CALL_END, foundSnippets, "", "", 0, "", true);
            snippetsNeedingUpdate.addAll((Collection)readmeResult.result);
            badSnippetCalls.addAll(readmeResult.errorList);
        }
        if (snippetsNeedingUpdate.size() > 0 || badSnippetCalls.size() > 0) {
            for (VerifyResult verifyResult : snippetsNeedingUpdate) {
                logger.error((CharSequence)String.format("SnippetId %s needs update in file %s.", verifyResult.snippetWithIssues, verifyResult.readmeLocation));
            }
            for (VerifyResult verifyResult : badSnippetCalls) {
                logger.error((CharSequence)String.format("Unable to locate snippet with Id of %s. Reference in %s", verifyResult.snippetWithIssues, verifyResult.readmeLocation));
            }
            throw new MojoExecutionException("Snippet-Replacer has encountered errors, check above output for details.");
        }
    }

    public static void updateCodesnippets(Path codesnippetRootDirectory, String codesnippetGlob, Path sourcesRootDirectory, String sourcesGlob, boolean includeSources, Path readmePath, boolean includeReadme, int maxLineLength, Log logger) throws IOException, MojoExecutionException {
        if (!includeSources && !includeReadme) {
            logger.debug((CharSequence)"Neither sources or README were included. No codesnippet updating will be done.");
            return;
        }
        List<Path> codesnippetFiles = SnippetReplacer.globFiles(codesnippetRootDirectory, codesnippetGlob, false);
        List<Object> sourceFiles = Collections.emptyList();
        if (includeSources) {
            sourceFiles = SnippetReplacer.globFiles(sourcesRootDirectory, sourcesGlob, true);
        }
        Map<String, List<String>> foundSnippets = SnippetReplacer.getAllSnippets(codesnippetFiles);
        ArrayList<VerifyResult> badSnippetCalls = new ArrayList<VerifyResult>();
        if (includeSources) {
            for (Path path : sourceFiles) {
                badSnippetCalls.addAll(SnippetReplacer.updateSourceCodeSnippets(path, foundSnippets));
            }
        }
        if (includeReadme) {
            badSnippetCalls.addAll(SnippetReplacer.updateReadmeCodesnippets(readmePath, foundSnippets));
        }
        if (badSnippetCalls.size() > 0) {
            for (VerifyResult verifyResult : badSnippetCalls) {
                logger.error((CharSequence)String.format("Unable to locate snippet with Id of %s. Reference in %s", verifyResult.snippetWithIssues, verifyResult.readmeLocation.toString()));
            }
            throw new MojoExecutionException("Discovered snippets in need of updating. Please run this plugin in update mode and commit the changes.");
        }
    }

    static List<VerifyResult> updateReadmeCodesnippets(Path file, Map<String, List<String>> snippetMap) throws IOException {
        SnippetOperationResult<StringBuilder> opResult = SnippetReplacer.updateSnippets(file, SNIPPET_README_CALL_BEGIN, SNIPPET_README_CALL_END, snippetMap, "", "", 0, "", true);
        if (opResult.result != null) {
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);){
                writer.write(((StringBuilder)opResult.result).toString());
            }
        }
        return opResult.errorList;
    }

    static List<VerifyResult> updateSourceCodeSnippets(Path file, Map<String, List<String>> snippetMap) throws IOException {
        SnippetOperationResult<StringBuilder> opResult = SnippetReplacer.updateSnippets(file, SNIPPET_SRC_CALL_BEGIN, SNIPPET_SRC_CALL_END, snippetMap, "<pre>", "</pre>", 1, "* ", false);
        if (opResult.result != null) {
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);){
                writer.write(((StringBuilder)opResult.result).toString());
            }
        }
        return opResult.errorList;
    }

    static SnippetOperationResult<StringBuilder> updateSnippets(Path file, Pattern beginRegex, Pattern endRegex, Map<String, List<String>> snippetMap, String preFence, String postFence, int prefixGroupNum, String additionalLinePrefix, boolean disableEscape) throws IOException {
        List<String> lines = Files.readAllLines(file, StandardCharsets.UTF_8);
        ArrayList<VerifyResult> badSnippetCalls = new ArrayList<VerifyResult>();
        StringBuilder modifiedLines = new StringBuilder();
        boolean inSnippet = false;
        boolean needsAmend = false;
        String lineSep = System.lineSeparator();
        String currentSnippetId = "";
        for (String line : lines) {
            Matcher begin = beginRegex.matcher(line);
            Matcher end = endRegex.matcher(line);
            if (begin.matches()) {
                modifiedLines.append(line).append(lineSep);
                currentSnippetId = begin.group(2);
                inSnippet = true;
                continue;
            }
            if (end.matches()) {
                if (inSnippet) {
                    if (!snippetMap.containsKey(currentSnippetId)) {
                        badSnippetCalls.add(new VerifyResult(file, currentSnippetId));
                        needsAmend = true;
                        inSnippet = false;
                        continue;
                    }
                    List<String> newSnippets = snippetMap.get(currentSnippetId);
                    ArrayList<String> modifiedSnippets = new ArrayList<String>();
                    String linePrefix = SnippetReplacer.prefixFunction(end, prefixGroupNum, additionalLinePrefix);
                    for (String snippet : SnippetReplacer.respaceLines(newSnippets)) {
                        String moddedSnippet = disableEscape ? snippet : SnippetReplacer.escapeString(snippet);
                        modifiedSnippets.add(moddedSnippet.length() == 0 ? END_OF_LINE_WHITESPACES.matcher(linePrefix).replaceAll("") + lineSep : linePrefix + moddedSnippet + lineSep);
                    }
                    if (preFence != null && preFence.length() > 0) {
                        modifiedLines.append(linePrefix).append(preFence).append(lineSep);
                    }
                    modifiedSnippets.forEach(modifiedLines::append);
                    if (postFence != null && postFence.length() > 0) {
                        modifiedLines.append(linePrefix).append(postFence).append(lineSep);
                    }
                    modifiedLines.append(line).append(lineSep);
                    needsAmend = true;
                    inSnippet = false;
                    continue;
                }
                modifiedLines.append(line).append(lineSep);
                continue;
            }
            if (inSnippet) continue;
            modifiedLines.append(line).append(lineSep);
        }
        if (needsAmend) {
            return new SnippetOperationResult<StringBuilder>(modifiedLines, badSnippetCalls);
        }
        return new SnippetOperationResult<Object>(null, badSnippetCalls);
    }

    static SnippetOperationResult<List<VerifyResult>> verifySnippets(Path file, Pattern beginRegex, Pattern endRegex, Map<String, List<String>> snippetMap, String preFence, String postFence, int prefixGroupNum, String additionalLinePrefix, boolean disableEscape) throws IOException {
        List<String> lines = Files.readAllLines(file, StandardCharsets.UTF_8);
        boolean inSnippet = false;
        String lineSep = System.lineSeparator();
        ArrayList<String> currentSnippetSet = null;
        ArrayList<VerifyResult> foundIssues = new ArrayList<VerifyResult>();
        ArrayList<VerifyResult> badSnippetCalls = new ArrayList<VerifyResult>();
        String currentSnippetId = "";
        for (String line : lines) {
            Matcher begin = beginRegex.matcher(line);
            Matcher end = endRegex.matcher(line);
            if (begin.matches()) {
                currentSnippetId = begin.group(2);
                inSnippet = true;
                currentSnippetSet = new ArrayList<String>();
                continue;
            }
            if (end.matches()) {
                if (!inSnippet) continue;
                if (!snippetMap.containsKey(currentSnippetId)) {
                    badSnippetCalls.add(new VerifyResult(file, currentSnippetId));
                    inSnippet = false;
                    currentSnippetSet = null;
                    continue;
                }
                List<String> newSnippets = snippetMap.get(currentSnippetId);
                ArrayList<String> modifiedSnippets = new ArrayList<String>();
                String linePrefix = SnippetReplacer.prefixFunction(end, prefixGroupNum, additionalLinePrefix);
                for (String snippet : SnippetReplacer.respaceLines(newSnippets)) {
                    String moddedSnippet = disableEscape ? snippet : SnippetReplacer.escapeString(snippet);
                    modifiedSnippets.add(moddedSnippet.length() == 0 ? END_OF_LINE_WHITESPACES.matcher(linePrefix).replaceAll("") + lineSep : linePrefix + moddedSnippet + lineSep);
                }
                Collections.sort(modifiedSnippets);
                Collections.sort(currentSnippetSet);
                if (!modifiedSnippets.equals(currentSnippetSet)) {
                    foundIssues.add(new VerifyResult(file, currentSnippetId));
                }
                inSnippet = false;
                currentSnippetSet = null;
                continue;
            }
            if (!inSnippet) continue;
            if (preFence.length() > 0 && postFence.length() > 0) {
                if (line.contains(preFence) || line.contains(postFence)) continue;
                currentSnippetSet.add(line + lineSep);
                continue;
            }
            currentSnippetSet.add(line + lineSep);
        }
        return new SnippetOperationResult<List<VerifyResult>>(foundIssues, badSnippetCalls);
    }

    static Map<String, List<String>> getAllSnippets(List<Path> snippetSources) throws IOException, MojoExecutionException {
        HashMap<String, List<String>> locatedSnippets = new HashMap<String, List<String>>();
        ArrayList<VerifyResult> detectedIssues = new ArrayList<VerifyResult>();
        for (Path samplePath : snippetSources) {
            List<String> fileContent = Files.readAllLines(samplePath, StandardCharsets.UTF_8);
            HashMap<String, List<String>> tempSnippetMap = new HashMap<String, List<String>>();
            SnippetDictionary snippetReader = new SnippetDictionary();
            for (String line : fileContent) {
                Matcher begin = SNIPPET_DEF_BEGIN.matcher(line);
                Matcher end = SNIPPET_DEF_END.matcher(line);
                if (begin.matches()) {
                    String id_beginning = begin.group(1);
                    snippetReader.beginSnippet(id_beginning);
                    continue;
                }
                if (end.matches()) {
                    String id_ending = end.group(1);
                    List<String> snippetContent = snippetReader.finalizeSnippet(id_ending);
                    if (!tempSnippetMap.containsKey(id_ending)) {
                        tempSnippetMap.put(id_ending, snippetContent);
                        continue;
                    }
                    detectedIssues.add(new VerifyResult(samplePath, id_ending));
                    continue;
                }
                if (!snippetReader.isActive()) continue;
                snippetReader.processLine(line);
            }
            for (String snippetId : tempSnippetMap.keySet()) {
                if (!locatedSnippets.containsKey(snippetId)) {
                    locatedSnippets.put(snippetId, (List)tempSnippetMap.get(snippetId));
                    continue;
                }
                detectedIssues.add(new VerifyResult(samplePath, snippetId));
            }
        }
        if (detectedIssues.size() > 0) {
            throw new MojoExecutionException("Duplicate Snippet Definitions Detected. " + System.lineSeparator() + SnippetReplacer.getErrorString(detectedIssues));
        }
        return locatedSnippets;
    }

    static String getErrorString(List<VerifyResult> errors) {
        StringBuilder results = new StringBuilder();
        for (VerifyResult result : errors) {
            results.append("Duplicate snippetId ").append(result.snippetWithIssues).append(" detected in ").append(result.readmeLocation).append(".").append(System.lineSeparator());
        }
        return results.toString();
    }

    static List<String> respaceLines(List<String> snippetText) {
        String minWhitespace = null;
        ArrayList<String> modifiedStrings = new ArrayList<String>();
        for (String snippetLine : snippetText) {
            Matcher leadSpaceMatch;
            if (snippetLine.trim().length() == 0 || !(leadSpaceMatch = WHITESPACE_EXTRACTION.matcher(snippetLine)).matches()) continue;
            String leadSpace = leadSpaceMatch.group(1);
            if (minWhitespace != null && leadSpace.length() >= minWhitespace.length()) continue;
            minWhitespace = leadSpace;
        }
        if (minWhitespace != null) {
            for (String snippetLine : snippetText) {
                modifiedStrings.add(snippetLine.replaceFirst(minWhitespace, ""));
            }
        }
        return modifiedStrings;
    }

    static String prefixFunction(Matcher match, int groupNum, String additionalPrefix) {
        if (match == null || groupNum < 1) {
            return "";
        }
        return match.group(groupNum) + additionalPrefix;
    }

    static String escapeString(String target) {
        if (target != null && target.trim().length() > 0) {
            for (String key : REPLACEMENT_SET.keySet()) {
                target = target.replace(key, REPLACEMENT_SET.get(key));
            }
        }
        return target;
    }

    private static List<Path> globFiles(Path rootFolder, String glob, boolean validate) throws IOException, MojoExecutionException {
        if (rootFolder == null || !rootFolder.toFile().isDirectory()) {
            if (validate) {
                throw new MojoExecutionException(String.format("Expected '%s' to be a directory but it wasn't.", rootFolder));
            }
            return new ArrayList<Path>();
        }
        final ArrayList<Path> locatedPaths = new ArrayList<Path>();
        final PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher("glob:" + glob);
        Files.walkFileTree(rootFolder, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (pathMatcher.matches(file)) {
                    locatedPaths.add(file);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                return FileVisitResult.CONTINUE;
            }
        });
        return locatedPaths;
    }

    private SnippetReplacer() {
    }
}

