/*
 * Copyright 2017 Baidu, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

package com.baidu.aip.imageprocess;

import com.baidu.aip.client.BaseClient;
import com.baidu.aip.error.AipError;
import com.baidu.aip.http.AipRequest;
import com.baidu.aip.http.EBodyFormat;
import com.baidu.aip.util.Base64Util;
import com.baidu.aip.util.Util;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;

public class AipImageProcess extends BaseClient {

    public AipImageProcess(String appId, String apiKey, String secretKey) {
        super(appId, apiKey, secretKey);
    }

    /**
     * 图像无损放大接口   
     * 输入一张图片，可以在尽量保持图像质量的条件下，将图像在长宽方向各放大两倍。
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject imageQualityEnhance(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.IMAGE_QUALITY_ENHANCE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像无损放大接口
     * 输入一张图片，可以在尽量保持图像质量的条件下，将图像在长宽方向各放大两倍。
     *
     * @param image - 本地图片路径
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject imageQualityEnhance(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return imageQualityEnhance(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像去雾接口   
     * 对浓雾天气下拍摄，导致细节无法辨认的图像进行去雾处理，还原更清晰真实的图像。
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject dehaze(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.DEHAZE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像去雾接口
     * 对浓雾天气下拍摄，导致细节无法辨认的图像进行去雾处理，还原更清晰真实的图像。
     *
     * @param image - 本地图片路径
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject dehaze(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return dehaze(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像对比度增强接口   
     * 调整过暗或者过亮图像的对比度，使图像更加鲜明。
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject contrastEnhance(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.CONTRAST_ENHANCE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像对比度增强接口
     * 调整过暗或者过亮图像的对比度，使图像更加鲜明。
     *
     * @param image - 本地图片路径
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject contrastEnhance(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return contrastEnhance(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 黑白图像上色接口   
     * 智能识别黑白图像内容并填充色彩，使黑白图像变得鲜活。
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject colourize(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.COLOURIZE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 黑白图像上色接口
     * 智能识别黑白图像内容并填充色彩，使黑白图像变得鲜活。
     *
     * @param image - 本地图片路径
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject colourize(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return colourize(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 拉伸图像恢复接口   
     * 自动识别过度拉伸的图像，将图像内容恢复成正常比例。
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject stretchRestore(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.STRETCH_RESTORE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 拉伸图像恢复接口
     * 自动识别过度拉伸的图像，将图像内容恢复成正常比例。
     *
     * @param image - 本地图片路径
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject stretchRestore(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return stretchRestore(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像风格转换
     *
     * @param image - 图像地址
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */

    public JSONObject styleTrans(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return styleTrans(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像风格转换
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject styleTrans(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.STYLETRANS);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像修复
     *
     * @param image - 图像地址
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject inpaintingByMask(String image, List<JSONObject> list, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return inpaintingByMask(data, list, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像修复
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject inpaintingByMask(byte[] image, List<JSONObject> list, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        request.setBodyFormat(EBodyFormat.RAW_JSON);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        request.addBody("rectangle", list);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.INPAINTING);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像清晰度增强
     *
     * @param image - 图像地址
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject imageDefinitionEnhance(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return imageDefinitionEnhance(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像清晰度增强
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject imageDefinitionEnhance(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.IMAGEDEFINITIONENHANCE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 人像动漫化
     *
     * @param image - 图像地址
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject selfieAnime(String image, HashMap<String, Object> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return selfieAnime(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 人像动漫化
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject selfieAnime(byte[] image, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.SELFIEANIME);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 天空分割
     *
     * @param image - 图像地址
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject skySeg(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return skySeg(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 天空分割
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject skySeg(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.SKYSEG);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像色彩增强
     *
     * @param image - 图片位置
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject colorEnhance(String image, HashMap<String, String> options) {
        try {
            byte[] data = Util.readFileByBytes(image);
            return colorEnhance(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像色彩增强
     *
     * @param image - 二进制图像数据
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject colorEnhance(byte[] image, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.COLOR_ENHANCE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像色彩增强
     *
     * @param url - 图片url
     * @param options - 可选参数对象，key: value都为string类型
     * options - options列表:
     * @return JSONObject
     */
    public JSONObject colorEnhanceUrl(String url, HashMap<String, String> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        request.addBody("url", url);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.COLOR_ENHANCE);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图片去摩尔纹
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/ql4wdlnc0
     */
    public JSONObject removeMoireV1(String imagePath, HashMap<String, Object> options) {
        try {
            byte[] data = Util.readFileByBytes(imagePath);
            return removeMoireV1(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图片去摩尔纹
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/ql4wdlnc0
     */
    public JSONObject removeMoireV1(byte[] image, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.REMOVE_MOIRE_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图片去摩尔纹 - url
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/ql4wdlnc0
     */
    public JSONObject removeMoireV1Url(String url, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);

        request.addBody("url", url);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.REMOVE_MOIRE_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图片去摩尔纹 - pdf
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/ql4wdlnc0
     */
    public JSONObject removeMoireV1Pdf(String pdfPath, HashMap<String, Object> options) {
        try {
            byte[] data = Util.readFileByBytes(pdfPath);
            return removeMoireV1Pdf(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图片去摩尔纹 - pdf
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/ql4wdlnc0
     */
    public JSONObject removeMoireV1Pdf(byte[] pdf, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(pdf);
        request.addBody("pdf_file", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.REMOVE_MOIRE_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像风格自定义
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/al50vf6bq
     */
    public JSONObject customizeStylizationV1(String imagePath, HashMap<String, Object> options) {
        try {
            byte[] data = Util.readFileByBytes(imagePath);
            return customizeStylizationV1(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像风格自定义
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/al50vf6bq
     */
    public JSONObject customizeStylizationV1(byte[] image, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.CUSTOMIZE_STYLIZATION_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像风格自定义 - url
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/al50vf6bq
     */
    public JSONObject customizeStylizationV1Url(String url, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);

        request.addBody("url", url);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.CUSTOMIZE_STYLIZATION_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 文档图片去底纹
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Nl6os53ab
     */
    public JSONObject docRepairV1(String imagePath, HashMap<String, Object> options) {
        try {
            byte[] data = Util.readFileByBytes(imagePath);
            return docRepairV1(data, options);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 文档图片去底纹
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Nl6os53ab
     */
    public JSONObject docRepairV1(byte[] image, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.DOC_REPAIR_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 文档图片去底纹 - url
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Nl6os53ab
     */
    public JSONObject docRepairV1Url(String url, HashMap<String, Object> options) {
        AipRequest request = new AipRequest();
        preOperation(request);

        request.addBody("url", url);
        if (options != null) {
            request.addBody(options);
        }
        request.setUri(ImageProcessConsts.DOC_REPAIR_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像去噪
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Tl78sby7g
     */
    public JSONObject denoiseV1(String imagePath, int option) {
        try {
            byte[] data = Util.readFileByBytes(imagePath);
            return denoiseV1(data, option);
        } catch (IOException e) {
            e.printStackTrace();
            return AipError.IMAGE_READ_ERROR.toJsonResult();
        }
    }

    /**
     * 图像去噪
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Tl78sby7g
     */
    public JSONObject denoiseV1(byte[] image, int option) {
        AipRequest request = new AipRequest();
        preOperation(request);
        String base64Content = Base64Util.encode(image);
        request.addBody("image", base64Content);
        request.addBody("option", option);
        request.setUri(ImageProcessConsts.DENOISE_V1);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像去噪 - url
     * 接口使用说明文档: https://ai.baidu.com/ai-doc/IMAGEPROCESS/Tl78sby7g
     */
    public JSONObject denoiseV1Url(String url, int option) {
        AipRequest request = new AipRequest();
        preOperation(request);
        request.addBody("url", url);
        request.addBody("option", option);
        request.setUri(ImageProcessConsts.DENOISE_V1);
        postOperation(request);
        return requestServer(request);
    }

}