/*
 * Copyright (C) 2016 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.bns;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.util.CollectionUtils;

import com.baidu.driver4j.bns.Group;
import com.baidu.driver4j.bns.Instance;

/**
 * 支持缓存service实例的group
 * Created by chenxingbo on 2016/7/15.
 */
public class BdrpBnsGroup {
    //1分钟后即使结果错误也更新
    private static Long VERIFICATION_TIME = 60 * 1000L;
    //组的信息
    private Group group;
    //group 的instance实例
    private Map<String, List<Instance>> serviceInstances = new HashMap<String, List<Instance>>();

    private Map<String, Map.Entry<Long, List<Instance>>> verificationServiceInstances = new ConcurrentHashMap<String,
            Map.Entry<Long, List<Instance>>>();

    public BdrpBnsGroup(Group group, Map<String, List<Instance>> serviceInstances) {
        this.group = group;
        this.serviceInstances = serviceInstances;
    }

    /**
     * 更新group和serviceInstance，并获取更新后的对象
     */
    public BdrpBnsGroup updateAndGet(Group group, Map<String, List<Instance>> serviceInstances) {
        //默认全部更新group信息
        BdrpBnsGroup newGroup = new BdrpBnsGroup(this.group, this.serviceInstances);
        newGroup.verificationServiceInstances = this.verificationServiceInstances;
        newGroup.update(group, serviceInstances);
        return newGroup;
    }

    private void update(Group group, Map<String, List<Instance>> serviceInstances) {
        this.group = group;
        //instances返回空，则不更新
        if (this.serviceInstances == null) {
            this.serviceInstances = new HashMap<String, List<Instance>>();
        }
        updateServiceInstances(CollectionUtils.isEmpty(serviceInstances) ? this.serviceInstances : serviceInstances);
    }

    /**
     * 修改 group的instance实例信息
     */
    private void updateServiceInstances(Map<String, List<Instance>> serviceInstances) {
        Map<String, List<Instance>> tmpServiceInstanceMap = new HashMap<String, List<Instance>>();
        for (Map.Entry<String, List<Instance>> serviceInstanceEntry : serviceInstances.entrySet()) {

            if (group == null || group.getSerivceNames() == null || (!group.getSerivceNames()
                    .contains(serviceInstanceEntry.getKey()))) {
                continue;
            }
            List<Instance> originInstanceList = this.serviceInstances.get(serviceInstanceEntry.getKey());

            if (needUpdateInstance(serviceInstanceEntry.getKey(), originInstanceList,
                    serviceInstanceEntry.getValue())) {
                tmpServiceInstanceMap.put(serviceInstanceEntry.getKey(), serviceInstanceEntry.getValue());
            }
        }
        this.serviceInstances = tmpServiceInstanceMap;
    }

    /**
     * 判断是否需要更新service下的instance
     * 1. “BNS 故障 = 没返回结果 || (机器数少了50%以上 && 少的机器不是同一个物理机房)，50%是
     * （本次get_instance_by_services返回的机器数量/本地缓存中机器数量）得出的，这个数值跟status状态没关系。
     */
    private boolean needUpdateInstance(String serviceName, List<Instance> originInstanceList, List<Instance>
            newInstanceList) {
        if (CollectionUtils.isEmpty(originInstanceList)) {
            return true;
        }
        if (newInstanceList == null) {
            newInstanceList = new ArrayList<Instance>();
        }

        if (originInstanceList.size() > (newInstanceList.size() * 2)) {
            Map.Entry<Long, List<Instance>> verificationInstancePair = verificationServiceInstances.get(serviceName);
            if (!((verificationInstancePair != null
                           && objectEquals(verificationInstancePair.getValue(), newInstanceList)
                           && (System.currentTimeMillis() - verificationInstancePair.getKey()) >= VERIFICATION_TIME))) {
                verificationServiceInstances.put(serviceName,
                        new AbstractMap.SimpleEntry<Long, List<Instance>>(System.currentTimeMillis(), newInstanceList));
                //一个service_name下如果 节点数量少了50%以上，那么就等1分钟再看，如果1分钟内节点仍有变化，就不更新本地缓存；
                return false;
            }
            //一个service_name下如果 节点数量少了50%以上，那么就等1分钟再看，如果还是少了这些，就更新本地缓存；
        }
        verificationServiceInstances.remove(serviceName);
        //一个service_name下如果节点数量没有少50%以上（节点增加，或者少的量小于50%） 就直接更新本地缓存
        return true;
    }

    private boolean objectEquals(Object var0, Object var1) {
        return var0 == null ? var1 == null : var0.equals(var1);
    }

    public Group getGroup() {
        return group;
    }

    public Map<String, List<Instance>> getServiceInstances() {
        return serviceInstances;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof BdrpBnsGroup)) {
            return false;
        }

        BdrpBnsGroup that = (BdrpBnsGroup) o;

        //TODO 目前group没有实现equals方法，需要自己定义(等待bns客户端更新)
        //        if (group != null ? !group.equals(that.group) : that.group != null) {
        //            return false;
        //        }

        if (!groupEquals(group, that.group)) {
            return false;
        }
        return serviceInstances != null ? serviceInstances.equals(that.serviceInstances)
                : that.serviceInstances == null;

    }

    private boolean groupEquals(Group thisGroup, Group oGroup) {
        if (thisGroup == oGroup) {
            return true;
        }
        if (thisGroup.getThreshold() != oGroup.getThreshold()) {
            return false;
        }
        if (thisGroup.getThresholdPercent() != oGroup.getThresholdPercent()) {
            return false;
        }
        if (!objectEquals(thisGroup.getConf(), oGroup.getConf())) {
            return false;
        }
        if (!objectEquals(thisGroup.getName(), oGroup.getName())) {
            return false;
        }
        if (!objectEquals(thisGroup.getSerivceNames(), oGroup.getSerivceNames())) {
            return false;
        }
        if (!objectEquals(thisGroup.getNodePath(), oGroup.getNodePath())) {
            return false;
        }
        return true;
    }
}
