/*
 * Copyright (C) 2016 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.bns;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.driver4j.bdrp.node.BnsGroupSupportsNodeManager;
import com.baidu.driver4j.bdrp.node.selector.RandomNodeSelector;
import com.baidu.driver4j.bdrp.node.selector.RoundRobbinNodeSelector;

/**
 * 监控bns group配置的变化，并回调更新node manager
 * Created by chenxingbo on 2016/7/21.
 */
public class BnsGroupConfUpdater implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(BnsGroupConfUpdater.class);
    //线程
    private Thread bnsGroupConfUpdateThread = new Thread(this);
    //bns group 配置变化检测的间隔时间（ms）
    private int bnsGroupConfUpdateInterval = 1000;
    //线程是否关闭
    private volatile boolean closed = false;
    //node manager
    private BnsGroupSupportsNodeManager nodeManager;

    public BnsGroupConfUpdater(BnsGroupSupportsNodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    @Override
    public void run() {
        long time;
        long timeToWait; // 计算后的sleep时间
        while (!closed) {
            try {
                time = System.currentTimeMillis();
                boolean needForceRefesh = updateConf();
                if (needForceRefesh) {
                    synchronized (nodeManager) {
                        nodeManager.refeshNodes(null, true);
                    }
                }
                timeToWait = System.currentTimeMillis() - time;
                if (timeToWait >= bnsGroupConfUpdateInterval) {
                    continue; // already large than interval time
                }
                try {
                    Thread.sleep(bnsGroupConfUpdateInterval - timeToWait);
                } catch (Exception e) {
                    logger.debug(e.getMessage(), e.getMessage(), e.getCause());
                }
            } catch (Exception e) {
                logger.warn("BnsGroupConfUpdater update error : {}", e);
            }
        }
    }

    /**
     * @return boolean 是否需要强制更新nodeManager，若true，则强制更新nodeManager所有节点，若为false
     */
    public boolean updateConf() {
        boolean needForceRefesh = false;
        GroupIDCInstanceQuery groupQueryer = nodeManager.getBnsGroupInstanceQuery();
        BnsGroupConf bnsGroupConf = groupQueryer.getBnsGroupConf(nodeManager.getBnsService());
        if (bnsGroupConf == null) {
            return false;
        }
        //均衡策略：SuperStrategy->Balance为均衡策略，只需支持两种即可：Random(随机)、RoundRobin(轮询)。
        if (bnsGroupConf.getSuperStrategy() != null) {
            String balanceStrategy = bnsGroupConf.getSuperStrategy().get("Balance");
            if (balanceStrategy != null && !"".equals(balanceStrategy.trim())) {
                if ("Random".equals(balanceStrategy)
                        && !(nodeManager.getNodeSelector() instanceof RandomNodeSelector)) {
                    nodeManager.setNodeSelector(new RandomNodeSelector());
                } else if ("RoundRobin".equals(balanceStrategy)
                        && !(nodeManager.getNodeSelector() instanceof RoundRobbinNodeSelector)) {
                    nodeManager.setNodeSelector(new RoundRobbinNodeSelector());
                }
            }
        }
        int timeout = 0;
        //设置连接超时时间
        //        if (bnsGroupConf.getServiceCtimeout() != null && bnsGroupConf.getServiceCtimeout() > 0) {
        //            timeout = bnsGroupConf.getServiceCtimeout();
        //        }
        if (bnsGroupConf.getServiceRtimeout() != null && bnsGroupConf.getServiceRtimeout() > timeout) {
            timeout = bnsGroupConf.getServiceRtimeout();
        }
        if (bnsGroupConf.getServiceWtimeout() != null && bnsGroupConf.getServiceWtimeout() > timeout) {
            timeout = bnsGroupConf.getServiceWtimeout();
        }
        if (timeout > 0 && timeout != nodeManager.getTimeout()) {
            nodeManager.setTimeout(timeout);
            needForceRefesh = true;
        }

        //设置失败重试次数
        if (bnsGroupConf.getServiceRetry() != null && bnsGroupConf.getServiceRetry() >= 0) {
            nodeManager.setRetryCountOnError(bnsGroupConf.getServiceRetry());
        }
        //暂不考虑读超时和写超时

        return needForceRefesh;
    }

    public void stop() {
        closed = true;
    }

    public void start() {
        bnsGroupConfUpdateThread.start();
    }

    public void setBnsGroupConfUpdateInterval(int bnsGroupConfUpdateInterval) {
        this.bnsGroupConfUpdateInterval = bnsGroupConfUpdateInterval;
    }
}
