/*
 * Copyright (C) 2016 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.bns;

import com.baidu.driver4j.bns.*;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * bns group 基于 idc 的查询
 * Created by chenxingbo on 2016/7/20.
 */
public class GroupIDCInstanceQuery implements InstanceQuery {
    private static final Logger LOGGER = LoggerFactory.getLogger(GroupIDCInstanceQuery.class);
    //指定的idc下没有service时，使用的默认idc
    private static final String DEFAULT_IDC = "default";
    //service实例中 IDC信息的分隔符
    private static final String IDC_SPERATOR = ".";
    //IDC信息
    private String idc;
    private BNSQueryAgentProxy bnsQueryAgentProxy;

    private final Map<String, BdrpBnsGroup> localGroupsCache = new ConcurrentHashMap<String, BdrpBnsGroup>();

    protected GroupIDCInstanceQuery(String bnsServerUrl, String idc) {
        BNSQueryProxy bnsQueryProxy = BNSQueryProxy.proxy(bnsServerUrl);
        bnsQueryAgentProxy = BNSQueryAgentProxy.proxy(bnsQueryProxy);
        this.idc = idc;
    }

    public static GroupIDCInstanceQuery proxy(String bnsServerUrl, String idc) {
        return new GroupIDCInstanceQuery(bnsServerUrl, idc);
    }

    /**
     * 通过bns group name 和idc获取instance
     */
    @Override
    public List<Instance> queryInstances(String bnsGroupName) throws BNSException {
        BdrpBnsGroup bdrpBnsGroup = localGroupsCache.get(bnsGroupName);
        BdrpBnsGroup newBdrpBnsGroup = getBnsGroup(bnsGroupName);
        if (bdrpBnsGroup != null) {
            newBdrpBnsGroup =
                    bdrpBnsGroup.updateAndGet(newBdrpBnsGroup.getGroup(), newBdrpBnsGroup.getServiceInstances());
        }
        if (bdrpBnsGroup == null || (newBdrpBnsGroup != null && !newBdrpBnsGroup.equals(bdrpBnsGroup))) {
            localGroupsCache.put(bnsGroupName, newBdrpBnsGroup);
        }
        return getIdcInstanceFromBnsGroup(newBdrpBnsGroup);
    }

    public BdrpBnsGroup getBnsGroup(String bnsGroupName) {
        Group group = bnsQueryAgentProxy.getBnsQueryProxy().getServiceGroup(bnsGroupName);
        Map<String, List<Instance>> groupServiceInstances =
                getGroupServiceInstances(group == null ? null : group.getSerivceNames());
        return new BdrpBnsGroup(group, groupServiceInstances);
    }

    /**
     * 获取serviceNames对应的instances
     */
    private Map<String, List<Instance>> getGroupServiceInstances(Set<String> serviceNames) {
        if (CollectionUtils.isEmpty(serviceNames)) {
            return null;
        }
        Map<String, List<Instance>> groupServiceInstances = new HashMap<String, List<Instance>>();
        for (String serviceName : serviceNames) {
            List<Instance> serviceInstances = BnsHelper.queryValidInstances(bnsQueryAgentProxy, serviceName);
            if (!CollectionUtils.isEmpty(serviceInstances)) {
                groupServiceInstances.put(serviceName, serviceInstances);
            }
        }
        return groupServiceInstances;
    }

    /**
     * 获取idc对应的bns实例
     * 取的顺序idc(prefer->backup) | default(prefer->backup)
     */
    private List<Instance> getIdcInstanceFromBnsGroup(BdrpBnsGroup bdrpBnsGroup) {
        if (bdrpBnsGroup == null || bdrpBnsGroup.getGroup() == null) {
            return null;
        }
        /*获取bdrp group 配置信息*/
        Group group = bdrpBnsGroup.getGroup();
        BnsGroupConf bnsGroupConfObj = getBnsGroupConf(group);
        Assert.notNull(bnsGroupConfObj, "没有获取到bdrp的bns group配置信息!");
        Assert.notNull(bnsGroupConfObj.getIdcMap(), "没有获取到bdrp的bns group配置中idc_map信息!");
        /*获取idc对应的idcmap信息*/
        BnsGroupConf.IdcMapInfo idcMapInfo = bnsGroupConfObj.getIdcMap().get(idc);
        if (idcMapInfo == null) {
            idcMapInfo = bnsGroupConfObj.getIdcMap().get(DEFAULT_IDC);
        }
        /*获取实例信息*/
        List<Instance> instances = getGroupIdcServices(bdrpBnsGroup, idcMapInfo);
        if (CollectionUtils.isEmpty(instances)) {
            LOGGER.warn("没有获取到bdrp的bns group下指定idc的bns实例,group conf 为:" + group == null ? "" : group.getConf());
        }
        return instances;
    }

    public BnsGroupConf getBnsGroupConf(Group group) {
        String groupConf = group.getConf();
        return json2Object(groupConf);
    }

    public BnsGroupConf getBnsGroupConf(String bnsGroupName) {
        BdrpBnsGroup bdrpBnsGroup = localGroupsCache.get(bnsGroupName);
        if (bdrpBnsGroup == null || bdrpBnsGroup.getGroup() == null) {
            return null;
        }
        String groupConf = bdrpBnsGroup.getGroup().getConf();
        return json2Object(groupConf);
    }

    /**
     * 获取group中指定idc的bns实例信息(通过group下的service列表+idcMapInfo)
     */
    private List<Instance> getGroupIdcServices(BdrpBnsGroup bdrpBnsGroup, BnsGroupConf.IdcMapInfo idcMapInfo) {
        Set<String> groupServiceNameList = bdrpBnsGroup.getGroup().getSerivceNames();
        if (idcMapInfo == null || CollectionUtils.isEmpty(groupServiceNameList)) {
            return null;
        }
        //获取prefer对应的idc instances
        Set<String> preferServiceNames = getIdcServiceNames(groupServiceNameList, idcMapInfo.getPrefer());
        List<Instance> preferInstances = getInstanceByServices(preferServiceNames, bdrpBnsGroup.getServiceInstances());
        if (!CollectionUtils.isEmpty(preferInstances)) {
            return preferInstances;
        }
        //获取backup对应的idc instances
        Set<String> backupServiceNames = getIdcServiceNames(groupServiceNameList, idcMapInfo.getBackup());
        return getInstanceByServices(backupServiceNames, bdrpBnsGroup.getServiceInstances());
    }

    /**
     * 通过service列表所有的instance列表
     */
    private List<Instance> getInstanceByServices(Set<String> serviceNames,
                                                 Map<String, List<Instance>> groupServiceInstances) {
        if (CollectionUtils.isEmpty(serviceNames) || groupServiceInstances == null) {
            return null;
        }
        List<Instance> instances = new ArrayList<Instance>();
        for (String serviceName : serviceNames) {
            List<Instance> serviceInstances = groupServiceInstances.get(serviceName);
            if (!CollectionUtils.isEmpty(serviceInstances)) {
                instances.addAll(serviceInstances);
            }
        }
        return instances;
    }

    /**
     * 获取idc对应的bns service 的set
     */
    private Set<String> getIdcServiceNames(Set<String> groupSerivceNameList, Set<String> idcs) {
        Set<String> idcServiceNames = new HashSet<String>();
        if (!(CollectionUtils.isEmpty(idcs) || CollectionUtils.isEmpty(groupSerivceNameList))) {
            outer:
            for (String groupServiceName : groupSerivceNameList) {
                for (String idc : idcs) {
                    if (groupServiceName.endsWith(IDC_SPERATOR + idc)) {
                        idcServiceNames.add(groupServiceName);
                        continue outer;
                    }
                }
            }
        }
        return idcServiceNames;
    }

    /**
     * json转换成bdrp的bns配置
     */
    private BnsGroupConf json2Object(String json) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapper.configure(SerializationFeature.INDENT_OUTPUT, false);
        objectMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
        objectMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
        objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        objectMapper.enable(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY);
        try {
            Map<String, Object> groupConf = objectMapper.readValue(json, Map.class);
            if (groupConf == null) {
                return null;
            }
            Object groupServicesConf = groupConf.get("services");
            if (groupServicesConf == null || !(groupServicesConf instanceof Map)) {
                return null;
            }
            Collection<Object> bnsGroupConfValue = ((Map) groupServicesConf).values();
            if (bnsGroupConfValue == null || bnsGroupConfValue.isEmpty()) {
                return null;
            }
            return objectMapper.readValue(objectMapper.writeValueAsString(bnsGroupConfValue.toArray()[0]), BnsGroupConf
                    .class);
        } catch (IOException e) {
            LOGGER.error("Json Transfer BdrpBnsGroupConf Exception Occurred!", e);
        }
        return null;
    }


    public void setConnectTimeout(int connectTimeout) {
        bnsQueryAgentProxy.setConnectTimeout(connectTimeout);
    }

    public void setReadTimeout(int readTimeout) {
        bnsQueryAgentProxy.setReadTimeout(readTimeout);
    }
}
