/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.bns;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.driver4j.bdrp.node.NodeManager;
import com.baidu.driver4j.bns.BNSCallback;
import com.baidu.driver4j.bns.Instance;

/**
 * BNS节点更新回调.
 *
 * @author dingxuefeng
 */
public class NodeBnsCallback implements BNSCallback {

    /** The Constant logger. */
    private static final Logger logger = LoggerFactory.getLogger(NodeBnsCallback.class);

    /** The node manager. */
    private NodeManager nodeManager;

    /** The service names. */
    private List<String> serviceNames = new ArrayList<String>();

    /**
     * 空构造方法.
     */
    public NodeBnsCallback() {
    }

    /**
     * 传入NodeManager的构造方法.
     *
     * @param nodeManager 将要刷新的节点管理器
     */
    public NodeBnsCallback(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    /**
     * 传入NodeManager和BNS名字的构造方法.
     *
     * @param nodeManager 将要刷新的节点管理器
     * @param serviceName BNS名称
     */
    public NodeBnsCallback(NodeManager nodeManager, String serviceName) {
        this.nodeManager = nodeManager;
        List<String> serviceNames = new ArrayList<String>();
        serviceNames.add(serviceName);
        this.serviceNames = serviceNames;
    }

    /**
     * 获取本Callback适用的服务名称.
     *
     * @return BNS服务名称
     */
    @Override
    public List<String> getServiceNames() {
        return serviceNames;
    }

    /**
     * 具体回调逻辑，更新NodeManager里的节点列表.
     *
     * @param s BNS客户端传入的BNS服务名
     * @param list BNS实例列表
     */
    @Override
    public void callback(String s, List<Instance> list) {
        if (nodeManager == null) {
            logger.warn("nodeManager is NULL. Ignore!");
            return;
        }
        if (list == null || list.isEmpty()) {
            logger.warn("There's an empty list while refreshing the BNS nodes. Ignore!");
            return;
		}
		if (null == s || !this.serviceNames.contains(s)) {
            logger.warn("The given service name {} is NOT valid. Ignore!", s);
            return;
        }

        StringBuilder nodeList = new StringBuilder();
        for (Instance instance : list) {
            nodeList.append(instance.getIp()).append(':').append(instance.getPort()).append(';');
        }
        synchronized(nodeManager) {
            nodeManager.refreshNodes(nodeList.toString());
        }
    }

    /**
     * Sets the node manager.
     *
     * @param nodeManager the new node manager
     */
    public void setNodeManager(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    /**
     * Sets the service names.
     *
     * @param serviceNames the new service names
     */
    public void setServiceNames(List<String> serviceNames) {
        this.serviceNames = serviceNames;
    }
}