/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.client;

import java.lang.reflect.Proxy;

import com.baidu.driver4j.bdrp.bns.BnsHelper;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;

import com.baidu.driver4j.bdrp.node.BnsGroupSupportsNodeManager;
import com.baidu.driver4j.bdrp.node.BnsSupportsNodeManager;
import com.baidu.driver4j.bdrp.node.NodeManager;

/**
 * BDRP客户端的创建工厂类，主要用于创建BdrpClient的代理实例
 *
 * @author dingxuefeng
 */
public class BdrpClientFactory implements InitializingBean, DisposableBean, FactoryBean {

    /**
     * 是否保持BdrpClient为单例
     */
    private boolean singleton = true;
    private BdrpClient client;

    private String nodes = "";
    private String idc = "default";

    private NodeManager nodeManager;

    /**
     * 控制是否开启BNS callback功能
     */
    private boolean enableBnsCallback = true;

    /**
     * 初始化NodeManager和BNS回调
     */
    public void init()  {
        // if already use setNodeManager method to set node manager
        if (nodeManager != null) {
            return;
        }

        Assert.hasText(nodes, "property 'nodes' is blank.");
        if (BnsHelper.isBnsGroup(nodes)) {
            nodeManager = new BnsGroupSupportsNodeManager();
            ((BnsGroupSupportsNodeManager) nodeManager).setIdc(idc);
            ((BnsGroupSupportsNodeManager) nodeManager).setEnableCallback(enableBnsCallback);
        } else if (BnsHelper.isBnsService(nodes)) {
            BnsSupportsNodeManager eNodeManager = new BnsSupportsNodeManager();
            eNodeManager.setEnableCallback(enableBnsCallback);
            nodeManager = eNodeManager;
        } else {
            nodeManager = new NodeManager();
        }
        nodeManager.setNodes(nodes);
        nodeManager.init();
    }

    /**
     * Spring初始化方法
     *
     * @see #init()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        init();
    }

    /**
     * 获取一个BdrpClient实例，根据配置选择创建一个新的或者是返回单例
     *
     * @return BdrpClient实例
     */
    public BdrpClient getBdrpClient() {
        if (!isSingleton()) {
            return build();
        }
        if (client == null) {
            synchronized(this) {
                if (client == null) {
                    client = build();
                }
            }
        }
        return client;
    }

    /**
     * 获取BdrpClient实例
     *
     * @see #getBdrpClient()
     */
    @Override
    public Object getObject() throws Exception {
        return getBdrpClient();
    }

    /**
     * 返回BdrpClient类型
     */
    @Override
    public Class getObjectType() {
        return BdrpClient.class;
    }

    /**
     * 销毁当前工厂相关实例
     */
    @Override
    public void destroy() throws Exception {
        if (nodeManager != null) {
            nodeManager.destroy();
        }
    }

    /**
     * 创建一个新的BdrpClient
     *
     * @return 针对当前配置的新BdrpClient
     */
    protected BdrpClient build() {
        JedisInvocationHandler handler = new JedisInvocationHandler();
        handler.setNodeManager(nodeManager);
        return (BdrpClient) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[] {BdrpClient.class},
                handler);
    }

    public NodeManager getNodeManager() {
        return nodeManager;
    }

    public void setNodeManager(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    @Override
    public boolean isSingleton() {
        return singleton;
    }

    public void setSingleton(boolean singleton) {
        this.singleton = singleton;
    }

    public String getNodes() {
        return nodes;
    }

    public void setNodes(String nodes) {
        this.nodes = nodes;
    }

    public void setIdc(String idc) {
        this.idc = idc;
    }

    public void setEnableBnsCallback(boolean enableBnsCallback) {
        this.enableBnsCallback = enableBnsCallback;
    }
}
