/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */

package com.baidu.driver4j.bdrp.client;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import com.baidu.driver4j.bdrp.node.BdrpNode;
import com.baidu.driver4j.bdrp.node.BdrpUtils;
import com.baidu.driver4j.bdrp.node.NodeManager;

import redis.clients.jedis.Jedis;

/**
 * Jedis实现的代理对象，负责将各个方法都进行增强
 *
 * @author dingxuefeng
 */
public class JedisInvocationHandler implements InvocationHandler {
    private static final Logger logger = LoggerFactory.getLogger(JedisInvocationHandler.class);

    private NodeManager nodeManager;

    /**
     * 代理各个Jedis操作，主要是在方法执行前后增加了对连接池的操作
     *
     * @param proxy   代理对象
     * @param method  调用的方法
     * @param objects 调用传入的参数
     * @return 方法返回值
     * @throws Throwable 执行时可能抛出的异常
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] objects) throws Throwable {
        Assert.notNull(nodeManager, "NodeManager should NOT be null!");

        Object ret = processSpecialMethods(method, objects);
        if (ret != null) {
            return ret;
        }

        int maxInvokeCount = (nodeManager.getRetryCountOnError() >= 0) ? (nodeManager.getRetryCountOnError() + 1) : 1;
        Throwable throwable = null;
        for (int i = 0; i < maxInvokeCount; i++) {
            throwable = null;
            Jedis jedis = null;
            BdrpNode node = nodeManager.getNode();
            Assert.notNull(node, "Can't obtain a property BdrpNode!");
            try {
                jedis = node.getJedis();
                ret = method.invoke(jedis, objects);
                node.returnJedis(jedis);
                break;
            } catch (Throwable t) {
                if (jedis != null) {
                    node.returnBrokenJedis(jedis);
                }
                nodeManager.refreshDisabledNodes();
                logger.error("Exception occurred while invoking Jedis operation: " + method.getName(), t);
                throwable = t;
            }
        }

        if (throwable != null) {
            throw throwable;
        }
        return ret;
    }

    /**
     * 代理一些特定的方法
     */
    protected Object processSpecialMethods(Method method, Object[] objects) {
        if ("getJedisFromPool".equals(method.getName())) {
            BdrpNode node = nodeManager.getNode();
            Assert.notNull(node, "Can't obtain a property BdrpNode!");
            return node.getJedis();
        }
        if ("returnJedisToPool".equals(method.getName())) {
            Assert.isTrue(objects != null && objects.length == 1);
            Jedis jedis = (Jedis) objects[0];
            String nodeName = jedis.getClient().getHost() + ":" + jedis.getClient().getPort();
            BdrpNode node = nodeManager.getNode(nodeName);
            if (node == null) {
                return "OK";
            }
            if (BdrpUtils.validateJedis(jedis)) {
                node.returnJedis(jedis);
            } else {
                node.returnBrokenJedis(jedis);
                nodeManager.refreshDisabledNodes();
            }
            return "OK";
        }

        return null;
    }

    public void setNodeManager(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }
}
