/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.driver4j.bdrp.node.health.HealthCriteria;
import com.baidu.driver4j.bdrp.node.health.HealthStatus;
import com.baidu.driver4j.bdrp.node.pool.BdrpPool;

import redis.clients.jedis.Jedis;

/**
 * BDRP节点对象
 *
 * @author dingxuefeng
 */
public class BdrpNode {
    private static final Logger logger = LoggerFactory.getLogger(BdrpNode.class);

    private String nodeKey;
    private HealthStatus healthStatus;
    private BdrpPool bdrpPool;

    /**
     * 创建BDRP节点
     *
     * @param nodeKey        一般是IP:PORT形式的字符串
     * @param healthCriteria 健康检查条件
     * @param bdrpPool       BDRP连接池
     */
    public BdrpNode(String nodeKey, HealthCriteria healthCriteria, BdrpPool bdrpPool) {
        this.nodeKey = nodeKey;
        this.healthStatus = new HealthStatus(healthCriteria);
        this.bdrpPool = bdrpPool;
    }

    /**
     * 归还Jedis连接，记录一次调用成功
     *
     * @param jedis 归还的Jedis
     */
    public void returnJedis(Jedis jedis) {
        if (jedis != null && bdrpPool != null) {
            bdrpPool.returnResource(jedis);
            healthStatus.recordSuccess();
        }
    }

    /**
     * 归还有问题的Jedis连接，记录一次调用失败
     *
     * @param jedis 需要关闭的Jedis
     */
    public void returnBrokenJedis(Jedis jedis) {
        logger.warn("Return broken jedis to node: {}", nodeKey);
        if (jedis != null && bdrpPool != null) {
            bdrpPool.returnBrokenResource(jedis);
            healthStatus.recordFailure();
        }
    }

    /**
     * 获取Jedis连接
     *
     * @return 如果连接池不为空则从连接池中获取Jedis
     */
    public Jedis getJedis() {
        if (bdrpPool == null) {
            return null;
        }
        return bdrpPool.getResource();
    }

    /**
     * 销毁节点，关闭连接池
     */
    public void destroy() {
        try {
            bdrpPool.destroy();
        } catch (Exception e) {
            logger.error("Exception occurred while destroying bdrpPool.", e);
        }
    }

    public boolean isHealth() {
        return healthStatus.isHealth();
    }

    public String getNodeKey() {
        return nodeKey;
    }

    public void setNodeKey(String nodeKey) {
        this.nodeKey = nodeKey;
    }

    public HealthStatus getHealthStatus() {
        return healthStatus;
    }
}
