/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node;

import com.baidu.driver4j.bdrp.bns.BnsHelper;
import com.baidu.driver4j.bdrp.bns.NodeBnsCallback;
import com.baidu.driver4j.bns.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;

/**
 * A BNS supports class for {@link NodeManager} which provide BNS change
 * callback feature.
 *
 * @author xiemalin
 */
public class BnsSupportsNodeManager extends NodeManager {
    private static final Logger logger = LoggerFactory.getLogger(BnsSupportsNodeManager.class);
    private BNSQueryAgentProxy bnsQueryProxy;

    /**
     * add call back supports
     */
    protected boolean enableCallback = true;

    private NodeBnsCallback nodeBnsCallback;
    protected BNSCallbackProxy bnsCallbackProxy;
    protected String bnsServer = "http://bns.noah.baidu.com/webfoot/index.php";
    protected String bnsService;
    protected int bnsReadTimeout = 10000;
    protected int bnsConnectTimeout = 10000;
    protected int bnsUpdateInterval = 5000;

    /**
     * 初始化BNS相关的一些配置
     */
    @Override
    public void init() {
        bnsQueryProxy = BNSQueryAgentProxy.proxy(bnsServer);
        initBnsQueryProxyTimeOut(bnsQueryProxy);
        List<Instance> instances = getInitedInstances();
        Assert.notNull(instances, bnsService + "'s instances can not be acquired.");
        logger.info("Add {} nodes to this BnsSupportNodeManager.", instances.size());
        setNodeList(instances);

        if (!enableCallback) {
            return;
        }
        if (nodeBnsCallback == null) {
            nodeBnsCallback = new NodeBnsCallback();
        }

        nodeBnsCallback.setNodeManager(this);
        nodeBnsCallback.setServiceNames(Collections.singletonList(bnsService));

        if (bnsCallbackProxy == null) {
            bnsCallbackProxy = createBNSCallbackProxy();
            bnsCallbackProxy.setInterval(bnsUpdateInterval);
        }
        bnsCallbackProxy.start();
        bnsCallbackProxy.register(nodeBnsCallback);

        super.init();
    }

    protected BNSCallbackProxy createBNSCallbackProxy() {
        return new BNSCallbackProxy(new InstanceQuery() {
            @Override
            public List<Instance> queryInstances(String serviceName) throws BNSException {
                return BnsHelper.queryValidInstances(bnsQueryProxy, serviceName);
            }
        });
    }

    protected void initBnsQueryProxyTimeOut(BNSQueryAgentProxy bnsQueryProxy) {
        bnsQueryProxy.setConnectTimeout(bnsConnectTimeout);
        bnsQueryProxy.setReadTimeout(bnsReadTimeout);
    }

    protected List<Instance> getInitedInstances() {
        return BnsHelper.queryValidInstances(bnsQueryProxy, bnsService);
    }

    /**
     * 销毁当前实例
     */
    @Override
    public void destroy() throws Exception {
        if (bnsCallbackProxy != null) {
            bnsCallbackProxy.stop();
        }
        super.destroy();
    }

    private void setNodeList(List<Instance> instances) {
        StringBuilder nodeList = new StringBuilder();
        for (Instance instance : instances) {
            nodeList.append(instance.getIp()).append(':').append(instance.getPort()).append(";");
        }
        super.setNodes(nodeList.toString());
    }

    public void setEnableCallback(boolean enableCallback) {
        this.enableCallback = enableCallback;
    }

    public void setNodeBnsCallback(NodeBnsCallback nodeBnsCallback) {
        this.nodeBnsCallback = nodeBnsCallback;
    }

    public void setBnsCallbackProxy(BNSCallbackProxy bnsCallbackProxy) {
        this.bnsCallbackProxy = bnsCallbackProxy;
    }

    @Override
    public void setNodes(String nodes) {
        this.bnsService = nodes;
    }

    public String getBnsService() {
        return bnsService;
    }

    public void setBnsServer(String bnsServer) {
        this.bnsServer = bnsServer;
    }

    public void setBnsReadTimeout(int bnsReadTimeout) {
        this.bnsReadTimeout = bnsReadTimeout;
    }

    public void setBnsConnectTimeout(int bnsConnectTimeout) {
        this.bnsConnectTimeout = bnsConnectTimeout;
    }

    public void setBnsUpdateInterval(int bnsUpdateInterval) {
        this.bnsUpdateInterval = bnsUpdateInterval;
    }
}