/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.baidu.driver4j.bdrp.node.health.HealthChecker;
import com.baidu.driver4j.bdrp.node.health.HealthCriteria;
import com.baidu.driver4j.bdrp.node.pool.BdrpPool;
import com.baidu.driver4j.bdrp.node.selector.NodeSelector;
import com.baidu.driver4j.bdrp.node.selector.RandomNodeSelector;

import redis.clients.jedis.Protocol;

/**
 * BDRP节点管理器
 *
 * @author dingxuefeng
 */
public class NodeManager extends GenericObjectPoolConfig implements InitializingBean, DisposableBean {
    private static final Logger logger = LoggerFactory.getLogger(NodeManager.class);

    // 存储节点信息的MAP
    private Map<String, BdrpNode> nodeMap = new HashMap<String, BdrpNode>();
    // 存储禁用节点键名
    private Set<String> disabledNodeSet = new HashSet<String>();
    // 是IP:PORT;IP:PORT的形式
    private String nodes = "";

    private HealthChecker healthChecker;
    private HealthCriteria healthCriteria = new HealthCriteria();
    private NodeSelector nodeSelector = new RandomNodeSelector();

    private int timeout = Protocol.DEFAULT_TIMEOUT;
    private int retryCountOnError = 0;

    public NodeManager() {
        //设置idle时校验连接
        setTestWhileIdle(true);
        //示idle object evitor两次扫描之间要sleep的毫秒数,默认1分钟
        setTimeBetweenEvictionRunsMillis(60 * 1000);
        //表示一个对象至少停留在idle状态的最短时间,默认5分钟
        setMinEvictableIdleTimeMillis(5 * 60 * 1000);
        setNumTestsPerEvictionRun(5);
    }

    /**
     * 主要是设置设置检查器
     */
    public void init() {
        if (healthCriteria == null) {
            healthCriteria = new HealthCriteria();
        }
        healthChecker = new HealthChecker();
        healthChecker.setCheckInterval(healthCriteria.getTimeUnit().toSeconds(healthCriteria.getPeriod()));
        healthChecker.setNodeManager(this);
        healthChecker.startChecking();
        refreshNodes(nodes);
    }

    /**
     * Spring用的初始化方法
     *
     * @see #init()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(nodeSelector);
        init();
    }

    public void setNodeSelector(NodeSelector nodeSelector) {
        this.nodeSelector = nodeSelector;
    }

    /**
     * 根据KEY信息取得对应的节点
     *
     * @return 从nodeMap里获取对应的节点
     */
    public BdrpNode getNode() {
        String key = nodeSelector.selectNodeName(this);
        logger.info("Select a node: {}", key);
        if (!StringUtils.hasText(key)) {
            return null;
        }
        return nodeMap.get(key);
    }

    /**
     * 通过ip:port获取node
     */
    public BdrpNode getNode(String nodeName) {
        if (!StringUtils.hasText(nodeName)) {
            return null;
        }
        return nodeMap.get(nodeName);
    }

    /**
     * 刷新不可用的节点列表
     */
    public void refreshDisabledNodes() {
        for (Map.Entry<String, BdrpNode> entry : getNodeMap().entrySet()) {
            if (entry.getValue().isHealth()) {
                disabledNodeSet.remove(entry.getKey());
            } else {
                disabledNodeSet.add(entry.getKey());
            }
        }
    }

    /**
     * 刷新节点信息
     *
     * @param nodes 形如“ip:port;ip:port”的节点信息
     */
    public void refreshNodes(String nodes) {
        refeshNodes(nodes, false);
    }

    public void refeshNodes(String nodes, boolean force) {
        if (nodes != null && !"".equals(nodes.trim())) {
            this.nodes = nodes;
        }
        String[] nodeArr = splitString(this.nodes, ";");
        addNewNodes(nodeArr, force);
        removeNodes(nodeArr);
    }

    /**
     * 获得可用节点
     *
     * @return 可用健康节点
     */
    public Map<String, BdrpNode> getAvailableNodes() {
        Map<String, BdrpNode> availableNodes = new HashMap<String, BdrpNode>();
        for (Map.Entry<String, BdrpNode> entry : getNodeMap().entrySet()) {
            if (!getDisabledNodeSet().contains(entry.getKey())) {
                availableNodes.put(entry.getKey(), entry.getValue());
            }
        }
        return availableNodes;
    }

    /**
     * 销毁所有实例
     */
    @Override
    public void destroy() throws Exception {
        if (healthChecker != null) {
            try {
                healthChecker.stopChecking();
            } catch (Exception e) {
                logger.warn("Exception occurred while destroying the health checker thread", e);
            }
        }
        for (BdrpNode node : nodeMap.values()) {
            try {
                node.destroy();
            } catch (Exception e) {
                logger.error("Exception occurred while destroying the BDRP Node " + node, e);
            }
        }
    }

    private void removeNodes(String[] nodeArr) {
        Set<String> keysToBeRemoved = new HashSet<String>();
        List<String> nodeList = Arrays.asList(nodeArr);
        for (String key : nodeMap.keySet()) {
            if (!nodeList.contains(key)) {
                keysToBeRemoved.add(key);
            }
        }
        for (String key : keysToBeRemoved) {
            BdrpNode node = nodeMap.remove(key);
            node.destroy();
        }
    }

    private void addNewNodes(String[] nodeArr, boolean force) {
        if (nodeArr.length <= 0) {
            return;
        }
        for (String node : nodeArr) {
            refreshNode(node, force);
        }
    }

    private void refreshNode(String node, boolean force) {
        if (nodeMap.containsKey(node) && !force) {
            return;
        }
        String[] nodeInfo = splitString(node, ":");
        if (nodeInfo == null) {
            return;
        }
        if (nodeInfo.length != 2) {
            logger.warn("Node format is NOT correct! Node:{}", node);
            return;
        }
        String host = nodeInfo[0];
        int port = -1;
        try {
            port = Integer.parseInt(nodeInfo[1]);
        } catch (NumberFormatException e) {
            logger.warn("Port of Node {} is NOT correct!", node);
            return;
        }
        nodeMap.put(node, new BdrpNode(node, healthCriteria, new BdrpPool(this, host, port, timeout)));
    }

    private String[] splitString(String str, String splitter) {
        if (null == str || str.length() == 0) {
            return new String[]{};
        }
        return str.split(splitter);
    }

    public Map<String, BdrpNode> getNodeMap() {
        return nodeMap;
    }

    public Set<String> getDisabledNodeSet() {
        return disabledNodeSet;
    }

    public void setHealthCriteria(HealthCriteria healthCriteria) {
        this.healthCriteria = healthCriteria;
    }

    public void setNodes(String nodes) {
        this.nodes = nodes;
    }

    public String getNodes() {
        return nodes;
    }

    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    public int getTimeout() {
        return timeout;
    }

    public void setRetryCountOnError(int retryCountOnError) {
        this.retryCountOnError = retryCountOnError;
    }

    public int getRetryCountOnError() {
        return retryCountOnError;
    }

    public NodeSelector getNodeSelector() {
        return nodeSelector;
    }
}
