/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node.health;

import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.driver4j.bdrp.node.BdrpNode;
import com.baidu.driver4j.bdrp.node.NodeManager;

import redis.clients.jedis.Jedis;

/**
 * 健康检查触发器
 *
 * @author dingxuefeng
 */
public class HealthChecker extends TimerTask {
    private static final Logger logger = LoggerFactory.getLogger(HealthChecker.class);

    /**
     * 检查间隔，单位秒
     */
    private long checkInterval = 60;

    private Lock lock = new ReentrantLock();
    private NodeManager nodeManager;
    private Timer timer = new Timer(true);

    /**
     * 定时触发的节点检查，如果当前有检查在执行，则忽略本次执行
     */
    @Override
    public void run() {
        try {
            if (!lock.tryLock()) {
                logger.warn("Last checking is not ended, ignore this checking.");
                return;
            }
            check();
        } catch (Exception e) {
            logger.error("Exception occurred while checking BDRP nodes", e);
        } finally {
            lock.unlock();
        }
    }

    /**
     * 进行节点检查，遍历节点管理器中的所有节点进行检查
     */
    public void check() {
        if (nodeManager == null) {
            logger.warn("NodeManager is Null!");
            return;
        }
        Map<String, BdrpNode> nodeMap = nodeManager.getNodeMap();
        for (Map.Entry<String, BdrpNode> entry : nodeMap.entrySet()) {
            BdrpNode node = entry.getValue();
            Jedis jedis = null;
            try {
                jedis = node.getJedis();
                jedis.get(HealthCriteria.TEST_KEY.getBytes());
                node.returnJedis(jedis);
                node.getHealthStatus().resetCheckingFailureCount();
            } catch (Exception e) {
                if (jedis != null) {
                    node.returnBrokenJedis(jedis);
                }
                logger.warn("Exception occurred while checking BDRP Node " + entry.getKey(), e);
                node.getHealthStatus().recordCheckingFailure();
            }
        }
        nodeManager.refreshDisabledNodes();
    }

    public void stopChecking(){
        timer.cancel();
    }

    /**
     * 启动定时检查
     */
    public void startChecking() {
        timer.scheduleAtFixedRate(this, checkInterval * 1000, checkInterval * 1000);
    }

    public void setNodeManager(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    public void setCheckInterval(long checkInterval) {
        this.checkInterval = checkInterval;
    }
}
