/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node.health;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * 节点健康状态维护类
 *
 * @author dingxuefeng
 */
public class HealthStatus {
    private HealthCriteria healthCriteria = new HealthCriteria();
    private AtomicInteger continuousCheckingFailureCount = new AtomicInteger(0);
    // 此处为了提高性能不使用AtomicInteger，可能带来误差，但问题不大
    private int continuousFailureCount = 0;
    private long lastRefreshTimeStamp = System.currentTimeMillis();

    /**
     * 空构造方法
     */
    public HealthStatus() {
    }

    /**
     * 创建对象时设置检查条件
     *
     * @param healthCriteria 健康检查条件
     */
    public HealthStatus(HealthCriteria healthCriteria) {
        this.healthCriteria = healthCriteria;
    }

    /**
     * 判断是否健康，主要是主动检查失败次数和使用时的失败计数
     *
     * @return 健康情况
     */
    public boolean isHealth() {
        return isHealthCheckingFailureCountUnderThreshold() && isFailureCountUnderThreshold();
    }

    /**
     * 主动健康检查失败次数是否低于阈值
     *
     * @return 判断结果
     */
    public boolean isHealthCheckingFailureCountUnderThreshold() {
        return continuousCheckingFailureCount.intValue() < healthCriteria.getCheckingFailThreshold();
    }

    /**
     * 使用时的失败次数是否低于阈值
     *
     * @return 判断结果
     */
    public boolean isFailureCountUnderThreshold() {
        return continuousFailureCount < healthCriteria.getDisableThreshold();
    }

    /**
     * 刷新失败计数值，如果已过计数周期则清零计数
     *
     * @return 是否已重置
     */
    public boolean refreshFailureCount() {
        long currentTimeStamp = System.currentTimeMillis();
        long duration = currentTimeStamp - lastRefreshTimeStamp;
        if (healthCriteria.getTimeUnit().toMillis(healthCriteria.getPeriod()) >= duration) {
            continuousFailureCount = 0;
            lastRefreshTimeStamp = currentTimeStamp;
            return true;
        }
        return false;
    }

    /**
     * 重置检查失败计数
     */
    public void resetCheckingFailureCount() {
        continuousCheckingFailureCount.set(0);
    }

    /**
     * 记录检查失败
     */
    public void recordCheckingFailure() {
        continuousCheckingFailureCount.incrementAndGet();
    }

    /**
     * 记录一次调用成功
     */
    public void recordSuccess() {
        continuousFailureCount = 0;
    }

    /**
     * 记录一次调用失败
     */
    public void recordFailure() {
        continuousFailureCount++;
    }

    public void setHealthCriteria(HealthCriteria healthCriteria) {
        this.healthCriteria = healthCriteria;
    }
}
